// Emacs: -*- C++ -*-

//
//	Copyright 1993, Center for Computer Vision and Visualization,
//	University of Florida.  All rights reserved.
//

//static char copyright[] = "Copyright 1993, Center for Computer Vision and Visualization,\nUniversity of Florida.  All rights reserved.\n";

//static char rcsid[] = "$Id: IntPS.c,v 1.38 1994/10/06 17:39:00 fjsoria Exp $";

//
// $Log: IntPS.c,v $
// Revision 1.38  1994/10/06  17:39:00  fjsoria
// commented rscid[] line
//
// Revision 1.37  1994/09/27  20:47:09  fjsoria
// g++ v2.6 does no like n0n-inlined destructors
// added __NO_INLINE_DESTRUCTORS
//
// Revision 1.36  1994/09/13  19:10:59  ljr
// Added workarounds for conditional expressions that create temporaries.
//
// Revision 1.35  1994/08/18  23:38:01  thoth
// DOS-inspired filename rework
//
// Revision 1.34  1994/07/25  16:58:52  thoth
// \Name sanitization
//
// Revision 1.33  1994/05/08  19:35:40  thoth
// New pointset concatenation capability.
// New intpointset decomposition capability.
//
// Revision 1.32  1994/04/26  18:13:57  thoth
// New array constructor that doesn't require dimension.
// White and Black Hole have been renamed to universal and empty _pset
//
// Revision 1.31  1994/03/14  15:38:32  thoth
// Check for errors in translation operation.
//
// Revision 1.30  1994/01/31  16:37:35  thoth
// We now document internal errors when they occur.
// We can now transpose boxy 2D pointsets.
//
// Revision 1.29  1993/11/17  18:03:47  thoth
// IPSIter is now PSIter<IntPoint>
// Pointcmp is now pointcmp
// extensivep() is now extensive()
//
// Revision 1.28  1993/10/20  15:50:54  thoth
// New naming scheme.  Closures are named Closure.
//
// Revision 1.27  1993/09/15  12:36:18  thoth
// Pointset behavior controls have moved to IA::
//
// Revision 1.26  93/08/08  13:17:33  thoth
// Parametrization of the pointsets on the point type using
// template specialization.
//
// Revision 1.25  93/07/15  11:45:17  thoth
// new lookup_and_set allows other classes (such as BasePSIter)
// to cooperate in refcounting the BasePSs.
//
// Revision 1.24  93/05/28  15:05:29  thoth
// removed dependency on with-ia-pointset*
//
// Revision 1.23  93/05/26  16:57:46  thoth
// Copyright Notices
//
// Revision 1.22  93/05/19  14:44:37  thoth
// LatticeRelation is now in IA::
//
// Revision 1.21  93/04/29  12:50:41  thoth
// Mondo PointSets are now templatized.
//
// Revision 1.20  93/04/19  23:27:40  thoth
// Convert BaseIPS to IA_BasePS<IA_IntPoint>
//
// Revision 1.19  93/04/17  15:19:51  thoth
// IA_IntPoint upgrades (dim, inf, sup)
//
// Revision 1.18  93/04/16  15:50:18  thoth
// use IA_IntPoint instead of IntPoint.
//
// Revision 1.17  93/02/12  16:23:41  thoth
// more IA_ prefixes.
// Removed #if 0 cruft.
// comp undefined bug is now gone from CFront
//
// Revision 1.16  93/01/20  11:36:18  thoth
// new PredicatePS class to allow for families of comprehensive pointsets.
//
// Revision 1.15  92/12/16  14:48:09  thoth
// conversion to IA_ mostly complete
//
// Revision 1.14  92/12/15  21:16:21  thoth
// added boxy predicate method.
//
// Revision 1.13  92/11/11  12:11:46  thoth
// added another restriction utility.
//
// Revision 1.12  92/11/04  11:58:51  thoth
// New union_with_structure function for image extension.
//
// Revision 1.11  92/10/25  20:53:24  thoth
// changed argument passing on set restriction routines.
//
// Revision 1.10  92/10/22  15:11:13  thoth
// lazy comparison handling (untested)
// pointset restriction utilities.
//
// Revision 1.9  92/10/07  23:15:31  thoth
// trim gratuitous includes from public headers.
// boxy boxy intersect should now be faster.
//
// Revision 1.8  92/09/30  10:30:52  thoth
// Lazies are now templatized.
// White and Black holes are now semi-public classes, unfortunately.
// reference counts are protected from IntPointSet, unfortunately.
// type() system now based on address of static data member.
//
// Revision 1.7  92/09/20  17:38:44  thoth
// Minkowski addition of point sets now supported.
//
// Revision 1.6  92/09/14  21:56:27  thoth
// the IntPointSet is now responsible for deleting hashed pointers.
// dimension is now a parameter to the IntPoint array constructor.
// added special code for ext/comp difference.
//
// Revision 1.5  92/09/07  13:21:31  thoth
// Black and White holes can be extensive
// |&^/ operators are now friends.
// intersection has untested code to handle comp&ext
//
// Revision 1.4  92/09/03  13:34:27  thoth
// hash_self now deals with unsigneds
// I unnamed unused formal parameters
// lazy point sets now exist
// (but operators & and / need adjustment)
//
// Revision 1.3  92/08/26  13:43:37  thoth
// meaningful exception types
// eliminated dummy parameter to constructor (was workaround for g++)
// aborts are clearly labeled and less common
//
// Revision 1.2  92/08/25  15:28:48  thoth
// new offsetted virtual functions
// new converter from IntPoint to IntPointSet.
// CFront supports agregates in ternary, wheee!
// set difference is operator/, not -
// operator+ is set translation, using the offsetted() function of BaseIPS
// lattice operations on sets defined in terms of compare() friend
//
// Revision 1.1  92/08/23  13:30:13  thoth
// Initial revision
//
//

#include <stdlib.h>
#include "IntPS.h"
#include "HashTable.h"
#include "PSIter.h"
#include "SetStructure.h"

#ifdef IA_PRII
#pragma implementation
#endif

#include "pointset_errors.h"
#include "BasePS.h"
#include "BoxyIPS.h"
#include "YoderIPS.h"
#include "LazyPS.h"
#include "MondoPS.h"
#include "ClosurePS.h"

IA_HashTable IA_Set<IA_Point<int> >::hash_table;

void IA_Set<IA_Point<int> >::lookup_and_set(IA_BasePS<IA_Point<int> > *ips)
{
    if (hash_table.reference(ips, &this->ps)) {
	ips->enref();
	if (ips->unref()<=0)
	    delete ips;
    } else {
	// ips->refcount=0;
    }

    ps->enref();
}

void IA_Set<IA_Point<int> >::disassociate_ps()
{
    if ( ps->unref() <= 0 ) {
	hash_table.remove(ps);
	delete ps;
    }
    ps=0;
}

IA_Set<IA_Point<int> >::IA_Set()
{
    this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(0));
}

IA_Set<IA_Point<int> >::IA_Set(IA::Type t, unsigned dim)
{
    if (t == IA_WhiteHole_<IA_Point<int> >::s_type())
	this->lookup_and_set(new IA_WhiteHole_<IA_Point<int> >(dim));
    else if (t == IA_BlackHole_<IA_Point<int> >::s_type())
	this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(dim));
    else {
	ia_throw(IA::TYPE_MISMATCH, __FILE__, __LINE__);
	this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(0));
    }
}

IA_Set<IA_Point<int> >::IA_Set(const IA_Point<int>& p)
{
    this->lookup_and_set(new IA_BoxyIPS(p, p));
}

IA_Set<IA_Point<int> >::IA_Set(const IA_Point<int>& p0, const IA_Point<int>& p1)
{
    this->lookup_and_set(new IA_BoxyIPS(p0, p1));
}

IA_Set<IA_Point<int> >::IA_Set(unsigned dimen, int(*f)(IA_Point<int>)) {
    this->lookup_and_set(new IA_PredicatePS_V<IA_Point<int> >(dimen, f));
}

IA_Set<IA_Point<int> >::IA_Set(unsigned dimen, int(*f)(const IA_Point<int>&)) {
    this->lookup_and_set(new IA_PredicatePS_CR<IA_Point<int> >(dimen, f));
}

IA_Set<IA_Point<int> >::IA_Set(const IA_ClosurePS<IA_Point<int> >&ps)
{
    this->lookup_and_set(ps.clone_self());
}

IA_Set<IA_Point<int> >::IA_Set(const IA_BoxyIPS& bps)
{
    this->lookup_and_set(new IA_BoxyIPS(bps));
}

IA_Set<IA_Point<int> >::IA_Set(const IA_YoderIPS& yps)
{
    if (yps.isboxy()){
	if (yps.nslices<1) {
	    // We've got a black hole which is, of course, boxy
	    this->lookup_and_set(new  IA_BlackHole_<IA_Point<int> >(yps.dimen));
	} else {
	    this->lookup_and_set(new IA_BoxyIPS(yps.min(), yps.max()));
	}
    } else this->lookup_and_set(new IA_YoderIPS(yps));
}


extern "C"
/*static*/ int ipcmp(const void *a, const void *b)
{
  return pointcmp(*(IA_Point<int>*)a,*(IA_Point<int>*)b);
}

IA_Set<IA_Point<int> >::IA_Set(unsigned dimen, const IA_Point<int> *aip, unsigned len)
{
    if (len<1) {
	this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(dimen));
	return;
    }

    for (unsigned i=0; i<len; i++) {
	if (aip[i].dim() != dimen) {
	    ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	    this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(0));
	    return;
	}
    }

    if (dimen<1) {
	if (len>0)
	    this->lookup_and_set(new IA_WhiteHole_<IA_Point<int> >(0));
	else
	    this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(0));
	return;
    }

    for (i=0; i+1<len; i++) {
	if (pointcmp(aip[i],aip[i+1]) ==1)
	    break;
    }

    IA_Point<int>	*newaip;
    if (i+1<len) {
	newaip = new IA_Point<int>[len];
	for (i=0; i<len; i++)
	    newaip[i] = aip[i];
	SortIntPointArray(newaip, len);
#if 0
	cout << "sorting array\n" << flush;
	for (i=0; i<len; i++)
	    cout << newaip[i] << ' ';
	cout << '\n';
#endif
    } else
	newaip = (IA_Point<int>*)aip;

    IA_YoderIPS *temp = new IA_YoderIPS(newaip, len);
    if (temp->isboxy()) {
	if (temp->nslices<1)
	    this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(temp->dimen));
	else
	    this->lookup_and_set(new IA_BoxyIPS(temp->min(), temp->max()));
	delete temp;
    } else
	this->lookup_and_set(temp);

    if (newaip!=aip)
	delete [] newaip;
}

IA_Set<IA_Point<int> >::IA_Set(const IA_Point<int> *aip, unsigned len)
{
    if (len<1) {
	ia_throw(Pointset_EmptyPointArray_Exception(__FILE__, __LINE__));
	this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(0));
	return;
    }

    const unsigned dimen = aip[0].dim();

    for (unsigned i=1; i<len; i++) {
	if (aip[i].dim() != dimen) {
	    ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	    this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(0));
	    return;
	}
    }

    if (dimen<1) {
	if (len>0)
	    this->lookup_and_set(new IA_WhiteHole_<IA_Point<int> >(0));
	else
	    this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(0));
	return;
    }

    for (i=0; i+1<len; i++) {
	if (pointcmp(aip[i],aip[i+1]) ==1)
	    break;
    }

    IA_Point<int
>	*newaip;
    if (i+1<len) {
	newaip = new IA_Point<int>[len];
	for (i=0; i<len; i++)
	    newaip[i] = aip[i];
	SortIntPointArray(newaip, len);
#if 0
	cout << "sorting array\n" << flush;
	for (i=0; i<len; i++)
	    cout << newaip[i] << ' ';
	cout << '\n';
#endif
    } else
	newaip = (IA_Point<int>*)aip;

    IA_YoderIPS *temp = new IA_YoderIPS(newaip, len);
    if (temp->isboxy()) {
	if (temp->nslices<1)
	    this->lookup_and_set(new IA_BlackHole_<IA_Point<int> >(temp->dimen));
	else
	    this->lookup_and_set(new IA_BoxyIPS(temp->min(), temp->max()));
	delete temp;
    } else
	this->lookup_and_set(temp);

    if (newaip!=aip)
	delete [] newaip;
}

#ifndef __NO_INLINE_DESTRUCTORS
IA_Set<IA_Point<int> >::~IA_Set()
{
    if (ps==0) {
	cerr << "Internal Error: pointset destructed twice!\n";
	return;
    }
    this->disassociate_ps();
    ps = 0;
}
#endif

IA_Set<IA_Point<int> >& IA_Set<IA_Point<int> >::operator=(const IA_Set<IA_Point<int> > &ips)
{ 
    IA_BasePS<IA_Point<int> >	*temp;
    temp = ips.ps;
    temp->enref();
    this->disassociate_ps();
    ps = temp;

    return *this;
}

int IA_Set<IA_Point<int> >::boxy() const
{
    return ps->type() == IA_BoxyIPS::s_type();
}

IA_Set<IA_Point<int> > operator|(const IA_Set<IA_Point<int> >& lhs,const IA_Set<IA_Point<int> > &rhs)
{
    if (lhs.dim() != rhs.dim()) {
	// need better message
	// cout << lhs.dim() << "!=" << rhs.dim() << '\n' << flush;
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }

    if (lhs.ps->type() == IA_BlackHole_<IA_Point<int> >::s_type())
	return rhs;
    if (lhs.ps->type() == IA_WhiteHole_<IA_Point<int> >::s_type())
	return lhs;

    if (rhs.ps->type() == IA_BlackHole_<IA_Point<int> >::s_type())
	return lhs;
    if (rhs.ps->type() == IA_WhiteHole_<IA_Point<int> >::s_type())
	return rhs;

    if (lhs.extensive() && rhs.extensive()) {

// the code enclosed by the following if block is obsolete
#if 0
	YoderIPS	*left, *right;
	switch (lhs.ps->type()) {
	case BOXY:
	    left = new YoderIPS(*(BoxyIPS*)lhs.ps);
	    break;
	case YODER:
	    left = (YoderIPS*)lhs.ps;
	    break;
	default:
	    // the other extensive pointsets were taken care of before.
	    IA::internal_error(__FILE__,__LINE__);
	    break;
	}

	switch (rhs.ps->type()) {
	case BOXY:
	    right = new YoderIPS(*(BoxyIPS*)rhs.ps);
	    break;
	case YODER:
	    right = (YoderIPS*)rhs.ps;
	    break;
	default:
	    // the other extensive pointsets were taken care of before.
	    IA::internal_error(__FILE__,__LINE__);
	    break;
	}

	YoderIPS result( *left | *right );

	if (left != lhs.ps)
	    delete left;
	if (right != rhs.ps)
	    delete right;
#endif

#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
    IA_YoderIPS *left, *right;

    if (lhs.ps->type()==IA_BoxyIPS::s_type())
        left = new IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps);
    else
        left = (IA_YoderIPS*)lhs.ps;

    if (rhs.ps->type()==IA_BoxyIPS::s_type())
        right = new IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps);
    else
        right = (IA_YoderIPS*)rhs.ps;

    IA_YoderIPS result( *left | *right );

    if (left != lhs.ps)
        delete left;
    if (right != rhs.ps)
        delete right;
#else
	IA_YoderIPS result
	  ( ( (lhs.ps->type()==IA_BoxyIPS::s_type()) ?
	     IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps) :
	     *(IA_YoderIPS*)lhs.ps )
	   |
	   ( (rhs.ps->type()==IA_BoxyIPS::s_type()) ?
	    IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps) :
	    *(IA_YoderIPS*)rhs.ps )
	   );
#endif
	// this constructor checks for emptiness and boxiness
	return IA_Set<IA_Point<int> >(result);
    } else {
	return new IA_LazyUnionPS<IA_Set<IA_Point<int> >,IA_Point<int> >(lhs, rhs);
    }
}

//

// intersect an extensive point set with a comprehensive one
// precondition:
// ext.extensive() ==1
// ext.dim() == comp.dim()
IA_Set<IA_Point<int> > ext_comp_intersect(const IA_Set<IA_Point<int> > &ext, const IA_Set<IA_Point<int> > &comp)
{
    IA_PSIter<IA_Point<int> >	iter(ext);
    IA_Point<int>	ip;

    IA_Point<int>	*arr=new IA_Point<int>[ext.card()];
    unsigned	count=0;

    while (iter(ip)) {
	if (comp.contains(ip)) {
	    arr[count++] = ip;
	}
    }

    return IA_Set<IA_Point<int> >(ext.dim(), arr,count);
}

IA_Set<IA_Point<int> > operator&(const IA_Set<IA_Point<int> >& lhs, const IA_Set<IA_Point<int> > &rhs)
{
    if (lhs.dim() != rhs.dim()) {
	// need better message
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }

    if (lhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type())
	return lhs;
    if (lhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	return rhs;

    if (rhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type())
	return rhs;
    if (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	return lhs;

    int lext, rext;
    lext = lhs.extensive();
    rext = rhs.extensive();
    if (lext && rext) {
	if (lhs.ps->type()==IA_BoxyIPS::s_type() &&
	    rhs.ps->type()==IA_BoxyIPS::s_type() ) {
	    IA_Point<int>	bottom(sup(lhs.ps->inf(), rhs.ps->inf()));
	    IA_Point<int>	top(inf (lhs.ps->sup(), rhs.ps->sup()));
	    if (bottom<=top)
		return IA_Set<IA_Point<int> >(bottom, top);
	    else
		return new IA_BlackHole_<IA_Point<int> >(lhs.dim());
	} else {
#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
       IA_YoderIPS *left, *right;

       if (lhs.ps->type()==IA_BoxyIPS::s_type())
           left = new IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps);
       else
           left = (IA_YoderIPS*)lhs.ps;

       if (rhs.ps->type()==IA_BoxyIPS::s_type())
           right = new IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps);
       else
           right = (IA_YoderIPS*)rhs.ps;

       IA_YoderIPS result(*left & *right);

       if (left != lhs.ps)
           delete left;
       if (right != rhs.ps)
           delete right;
#else
	    IA_YoderIPS result
		( ( (lhs.ps->type()==IA_BoxyIPS::s_type()) ?
		    IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps) :
		    *(IA_YoderIPS*)lhs.ps )
		  &
		  ( (rhs.ps->type()==IA_BoxyIPS::s_type()) ?
		    IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps) :
		    *(IA_YoderIPS*)rhs.ps )
		  );
#endif
	    return IA_Set<IA_Point<int> >(result);
	}
    } else if (lext) {
	return ext_comp_intersect(lhs, rhs);
    } else if (rext) {
	return ext_comp_intersect(rhs, lhs);
    } else {
	return new IA_LazyIntersectionPS<IA_Set<IA_Point<int> >,IA_Point<int> >(lhs,rhs);
    }
}


IA_Set<IA_Point<int> > intersect_with_structure
(const IA_Set<IA_Point<int> > &lhs, const IA_Set<IA_Point<int> > &rhs, IA_SetStructure *result)
{
    if (lhs.dim() != rhs.dim()) {
	// need better message
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }

    if (lhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type()) {
	return lhs;
    }
    if (lhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type()) {
	*result = (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	    ? IA_SetStructure::BOTH
	    : IA_SetStructure::FIRST_ONLY;
	return rhs;
    }

    if (rhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type()) {
	result->add_interval(lhs.card(), IA_SetStructure::FIRST_ONLY);
	return rhs;
    }
    if (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type()) {
	result->add_interval(lhs.card(), IA_SetStructure::BOTH);
	return lhs;
    }

    if (!lhs.extensive()) {
	// not rational
	return lhs & rhs;
    }

    if (!rhs.extensive()) {
	// deal with this later
	return lhs & rhs;
    }

#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
    IA_YoderIPS *left, *right;

    if (lhs.ps->type()==IA_BoxyIPS::s_type())
        left = new IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps);
    else
        left = (IA_YoderIPS*)lhs.ps;

    if (rhs.ps->type()==IA_BoxyIPS::s_type())
        right = new IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps);
    else
        right = (IA_YoderIPS*)rhs.ps;

    IA_Set<IA_Point<int> > retval((*left).intersect_with_structure(*right, result));

    if (left != lhs.ps)
        delete left;
    if (right != rhs.ps)
        delete right;

    return retval;
#else
    return ( (lhs.ps->type()==IA_BoxyIPS::s_type()) ?
	     IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps) :
	     *(IA_YoderIPS*)lhs.ps ). intersect_with_structure(
		 ( (rhs.ps->type()==IA_BoxyIPS::s_type()) ?
		   IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps) :
		   *(IA_YoderIPS*)rhs.ps ),
		 result);
#endif
}

IA_Set<IA_Point<int> > intersect_with_dualstruct
(const IA_Set<IA_Point<int> > &lhs, const IA_Set<IA_Point<int> > &rhs, IA_SetStructure *result)
{
    if (lhs.dim() != rhs.dim()) {
	// need better message
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }

    if (lhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type()) {
	result->add_interval(rhs.card(), IA_SetStructure::SECOND_ONLY);
	return lhs;
    }
    if (lhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type()) {
	*result = (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	    ? IA_SetStructure::BOTH
	    : IA_SetStructure::FIRST_ONLY;
	return rhs;
    }

    if (rhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type()) {
	result->add_interval(lhs.card(), IA_SetStructure::FIRST_ONLY);
	return rhs;
    }
    if (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type()) {
	result->add_interval(lhs.card(), IA_SetStructure::BOTH);
	return lhs;
    }

    if (!lhs.extensive()) {
	// not rational
	return lhs & rhs;
    }

    if (!rhs.extensive()) {
	// deal with this later
	return lhs & rhs;
    }

#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
    IA_YoderIPS *left, *right;

    if (lhs.ps->type()==IA_BoxyIPS::s_type())
        left = new IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps);
    else
        left = (IA_YoderIPS*)lhs.ps;

    if (rhs.ps->type()==IA_BoxyIPS::s_type())
        right = new IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps);
    else
        right = (IA_YoderIPS*)rhs.ps;

    IA_Set<IA_Point<int> > retval((*left).intersect_with_dualstruct(*right, result));

    if (left != lhs.ps)
        delete left;
    if (right != rhs.ps)
        delete right;

    return retval;
#else
    return ( (lhs.ps->type()==IA_BoxyIPS::s_type()) ?
	     IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps) :
	     *(IA_YoderIPS*)lhs.ps ). intersect_with_dualstruct(
		 ( (rhs.ps->type()==IA_BoxyIPS::s_type()) ?
		   IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps) :
		   *(IA_YoderIPS*)rhs.ps ),
		 result);
#endif
}

IA_Set<IA_Point<int> > union_with_structure
(const IA_Set<IA_Point<int> > &lhs, const IA_Set<IA_Point<int> > &rhs, IA_SetStructure *result)
{
    if (lhs.dim() != rhs.dim()) {
	// need better message
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }

    if (lhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type()) {
	result->add_interval(rhs.card(), IA_SetStructure::SECOND_ONLY);
	return rhs;
    }
    if (lhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type()) {
	*result = (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	    ? IA_SetStructure::BOTH
	    : IA_SetStructure::FIRST_ONLY;
	return lhs;
    }

    if (rhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type()) {
	result->add_interval(lhs.card(), IA_SetStructure::FIRST_ONLY);
	return lhs;
    }
    if (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type()) {
	*result = (lhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	    ? IA_SetStructure::BOTH
	    : IA_SetStructure::SECOND_ONLY;
	return rhs;
    }

    if (!lhs.extensive() || !rhs.extensive()) {
	return lhs | rhs;
    }

#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
    IA_YoderIPS *left, *right;

    if (lhs.ps->type()==IA_BoxyIPS::s_type())
        left = new IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps);
    else
        left = (IA_YoderIPS*)lhs.ps;

    if (rhs.ps->type()==IA_BoxyIPS::s_type())
        right = new IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps);
    else
        right = (IA_YoderIPS*)rhs.ps;

    IA_Set<IA_Point<int> > retval((*left).union_with_structure(*right, result));

    if (left != lhs.ps)
        delete left;
    if (right != rhs.ps)
        delete right;

    return retval;
#else
    return ( (lhs.ps->type()==IA_BoxyIPS::s_type()) ?
	     IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps) :
	     *(IA_YoderIPS*)lhs.ps ). union_with_structure(
		 ( (rhs.ps->type()==IA_BoxyIPS::s_type()) ?
		   IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps) :
		   *(IA_YoderIPS*)rhs.ps ),
		 result);
#endif
}


#if 1
IA_Set<IA_Point<int> > operator^(const IA_Set<IA_Point<int> >& lhs, const IA_Set<IA_Point<int> > &rhs)
{
    if (lhs.dim() != rhs.dim()) {
	// need better message
	// cout << lhs.dim() << "!=" << rhs.dim() << '\n' << flush;
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }

    if (lhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type())
	return rhs;
    if (lhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	return ~rhs;

    if (rhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type())
	return lhs;
    if (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	return ~lhs;

    if (lhs.extensive() && rhs.extensive()) {

#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
    IA_YoderIPS *left, *right;

    if (lhs.ps->type()==IA_BoxyIPS::s_type())
        left = new IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps);
    else
        left = (IA_YoderIPS*)lhs.ps;

    if (rhs.ps->type()==IA_BoxyIPS::s_type())
        right = new IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps);
    else
        right = (IA_YoderIPS*)rhs.ps;

    IA_YoderIPS result(*left ^ *right);

    if (left != lhs.ps)
        delete left;
    if (right != rhs.ps)
        delete right;
#else
	IA_YoderIPS result
	  ( ( (lhs.ps->type()==IA_BoxyIPS::s_type()) ?
	     IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps) :
	     *(IA_YoderIPS*)lhs.ps )
	   ^
	   ( (rhs.ps->type()==IA_BoxyIPS::s_type()) ?
	    IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps) :
	    *(IA_YoderIPS*)rhs.ps )
	   );
#endif
	return IA_Set<IA_Point<int> >(result);
    } else {
#if 1
	return new IA_LazyXORPS<IA_Set<IA_Point<int> >,IA_Point<int> >(lhs, rhs);
    }
#else
    } // this is where the curly brace should NOT be :(
    /* The following code belongs in the above else clause.
       However, if it is placed there, CFront generates erroneous C code.
       Since the code MAY reside outside the scope of the else and still
       be semantically equivalent, and doing so suppresses the error, 
       we will do this until CFront is fixed
       */
	int	kludge;	// if this declaration is removed, CFront flakes out
	IA_BasePS<IA_Point<int> >	*temp =  new IA_LazyXORPS<IntPointSet,IA_Point<int> >(lhs, rhs);
	return temp;
//    } // this is where the curly brace should be
#endif
}
#endif


/*
  asymmetric set difference
  */

// subtract a comprehensive point set from an extensive one
// precondition:
// ext.extensive() ==1
// ext.dim() == comp.dim()
IA_Set<IA_Point<int> > ext_comp_diff(const IA_Set<IA_Point<int> > &ext, const IA_Set<IA_Point<int> > &comp)
{
    IA_PSIter<IA_Point<int> >	iter(ext);
    IA_Point<int>	ip;

    IA_Point<int>	*arr=new IA_Point<int>[ext.card()];
    unsigned	count=0;

    while (iter(ip)) {
	if (!comp.contains(ip)) {
	    arr[count++] = ip;
	}
    }

    return IA_Set<IA_Point<int> >(ext.dim(), arr,count);
}

IA_Set<IA_Point<int> > operator/(const IA_Set<IA_Point<int> >& lhs, const IA_Set<IA_Point<int> > &rhs)
{
    if (lhs.dim() != rhs.dim()) {
	// need better message
	// cout << lhs.dim() << "!=" << rhs.dim() << '\n' << flush;
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }

    if (lhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type())
	return lhs;
    if (lhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	return ~rhs;

    if (rhs.ps->type()==IA_BlackHole_<IA_Point<int> >::s_type())
	return lhs;
    if (rhs.ps->type()==IA_WhiteHole_<IA_Point<int> >::s_type())
	return IA_BlackHole(lhs.dim());

    if (lhs.extensive() && rhs.extensive()) {

#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
    IA_YoderIPS *left, *right;

    if (lhs.ps->type()==IA_BoxyIPS::s_type())
        left = new IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps);
    else
        left = (IA_YoderIPS*)lhs.ps;

    if (rhs.ps->type()==IA_BoxyIPS::s_type())
        right = new IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps);
    else
        right = (IA_YoderIPS*)rhs.ps;

    IA_YoderIPS result(*left / *right);

    if (left != lhs.ps)
        delete left;
    if (right != rhs.ps)
        delete right;
#else
	IA_YoderIPS result
	  ( ( (lhs.ps->type()==IA_BoxyIPS::s_type()) ?
	     IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps) :
	     *(IA_YoderIPS*)lhs.ps )
	   /
	   ( (rhs.ps->type()==IA_BoxyIPS::s_type()) ?
	    IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps) :
	    *(IA_YoderIPS*)rhs.ps )
	   );
#endif
	return IA_Set<IA_Point<int> >(result);
    } else if (lhs.extensive()) {
	return ext_comp_diff(lhs, rhs);
    } else {
	return new IA_LazyMinusPS<IA_Set<IA_Point<int> >,IA_Point<int> >(lhs,rhs);
    }
}

IA_Set<IA_Point<int> > IA_Set<IA_Point<int> >::operator~() const
{
    if (ps->type() == IA_BlackHole_<IA_Point<int> >::s_type())
	return IA_WhiteHole(dim());
    else if (ps->type() == IA_WhiteHole_<IA_Point<int> >::s_type())
	return IA_BlackHole(dim());
    else 
	return new IA_LazyNotPS<IA_Set<IA_Point<int> >, IA_Point<int> >(*this);
}


IA_Set<IA_Point<int> > operator+(const IA_Set<IA_Point<int> >& ips, const IA_Point<int> &ip)
{
    if (ips.dim() != ip.dim()) {
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }
    if (ips.extensive())
	return ips.ps->offsetted(ip);
    else
	return new IA_LazyOffsettedPS<IA_Set<IA_Point<int> >, IA_Point<int> >(ips, ip);
}

static IA_Set<IA_Point<int> > Mink_aux(const IA_Set<IA_Point<int> >& small, const IA_Set<IA_Point<int> >& large)
{
    IA_PSIter<IA_Point<int> >	iter(small);
    IA_Point<int>	ip;
    IA_Set<IA_Point<int> >	rval(IA_BlackHole(small.dim()));

    while (iter(ip)) {
	rval |= large+ip;
    }

    return rval;
}

IA_Set<IA_Point<int> > operator+(const IA_Set<IA_Point<int> >& lhs, const IA_Set<IA_Point<int> >& rhs)
{
    if (lhs.dim() != rhs.dim()) {
	// cout << lhs.dim() << "!=" << rhs.dim() << '\n' << flush;
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA_Set<IA_Point<int> >();
    }

    if (lhs.extensive()) {
	if (rhs.extensive()) {

#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
      if (lhs.card() < rhs.card())
         return Mink_aux(lhs, rhs);
      else
         return Mink_aux(rhs, lhs);
#else
	    return (lhs.card() < rhs.card()) ?
		Mink_aux(lhs, rhs) : Mink_aux(rhs,lhs);
#endif
	} else {
	    return new IA_LazyMinkowskiPS<IA_PSIter<IA_Point<int> >,IA_Set<IA_Point<int> >, IA_Point<int> >(lhs,rhs);
	}
    } else {
	if (rhs.extensive()) {
	    return new IA_LazyMinkowskiPS<IA_PSIter<IA_Point<int> >,IA_Set<IA_Point<int> >, IA_Point<int> >(rhs,lhs);
	} else {
	    ia_throw(IA::PSET_REQUIRE_EXTENSIVE, __FILE__, __LINE__);
	}
    }
}


// this doesn't handle lazys
IA::LatticeRelation
compare(const IA_Set<IA_Point<int> > &lhs, const IA_Set<IA_Point<int> > &rhs)
{
    if (lhs.dim() != rhs.dim()) {
	// need better message
	// cout << lhs.dim() << "!=" << rhs.dim() << '\n' << flush;
	ia_throw(IA::PSET_DIMENSION_MISMATCH, __FILE__, __LINE__);
	return IA::UNKNOWN;
    }

    if (lhs.ps==rhs.ps)
	return IA::EQUAL;

    if (!lhs.extensive()) {
	if (!rhs.extensive())
	    return IA::UNKNOWN;
	// else, we can check if lhs>=rhs
	IA_PSIter<IA_Point<int> >	iter(rhs);
	IA_Point<int>	ip;
	while (iter(ip)) {
	    if (!lhs.contains(ip))
		return IA::UNKNOWN;
	}
	return IA::MP_SUPERSET;
    } else if (!rhs.extensive()) {
	// else, we can check if lhs<=rhs
	IA_PSIter<IA_Point<int> >	iter(lhs);
	IA_Point<int>	ip;
	while (iter(ip)) {
	    if (!rhs.contains(ip))
		return IA::UNKNOWN;
	}
	return IA::MP_SUBSET;
    }

    // invariant: lhs.extensive() && rhs.extensive()

    IA::Type ltype=lhs.ps->type();
    IA::Type rtype=rhs.ps->type();
    if (ltype==IA_WhiteHole_<IA_Point<int> >::s_type()) {
	if (rtype==IA_WhiteHole_<IA_Point<int> >::s_type())
	    return IA::EQUAL;
	else if (rtype==IA_BlackHole_<IA_Point<int> >::s_type() ||
		 rtype==IA_YoderIPS::s_type() ||
		 rtype==IA_BoxyIPS::s_type())
	    return IA::P_SUPERSET;
	else {
	    IA::internal_error(__FILE__,__LINE__);
	}
    } else if (ltype==IA_BlackHole_<IA_Point<int> >::s_type()) {
	if (rtype==IA_BlackHole_<IA_Point<int> >::s_type())
	    return IA::EQUAL;
	else if (rtype==IA_WhiteHole_<IA_Point<int> >::s_type() ||
		 rtype==IA_YoderIPS::s_type() ||
		 rtype==IA_BoxyIPS::s_type())
	    return IA::P_SUBSET;
	else {
	    IA::internal_error(__FILE__,__LINE__);
	}
    } else if (ltype==IA_BoxyIPS::s_type() ||
	       ltype==IA_YoderIPS::s_type()) {
	if (rtype==IA_WhiteHole_<IA_Point<int> >::s_type())
	    return IA::P_SUBSET;
	else if (rtype==IA_BlackHole_<IA_Point<int> >::s_type())
	    return IA::P_SUPERSET;
	else if (rtype==IA_BoxyIPS::s_type() ||
		 rtype==IA_YoderIPS::s_type())

#ifdef IA_NO_TEMPS_IN_CONDITIONAL_EXPRS
   {
       IA_YoderIPS *left, *right;

       if (lhs.ps->type()==IA_BoxyIPS::s_type())
           left = new IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps);
       else
           left = (IA_YoderIPS*)lhs.ps;

       if (rhs.ps->type()==IA_BoxyIPS::s_type())
           right = new IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps);
       else
           right = (IA_YoderIPS*)rhs.ps;

       IA::LatticeRelation retval(compare(*left, *right));

       if (left != lhs.ps)
           delete left;
       if (right != rhs.ps)
           delete right;

       return retval;
   }
#else
	    return compare( (lhs.ps->type()==IA_BoxyIPS::s_type())?
			    IA_YoderIPS(*(IA_BoxyIPS*)lhs.ps) : *(IA_YoderIPS*)lhs.ps ,
			    (rhs.ps->type()==IA_BoxyIPS::s_type())?
			    IA_YoderIPS(*(IA_BoxyIPS*)rhs.ps) : *(IA_YoderIPS*)rhs.ps);
#endif
	else {
	    IA::internal_error(__FILE__,__LINE__);
	}
    }
    IA::internal_error(__FILE__,__LINE__);
}


int operator<(const IA_Set<IA_Point<int> > &lhs, const IA_Set<IA_Point<int> > &rhs)
{
    IA::LatticeRelation	result;
    result = compare(lhs,rhs);
    return result==IA::P_SUBSET;
}

int operator<=(const IA_Set<IA_Point<int> > &lhs, const IA_Set<IA_Point<int> > &rhs)
{
    IA::LatticeRelation	result;
    result = compare(lhs,rhs);
    return result==IA::P_SUBSET
	|| result==IA::MP_SUBSET
	    || result==IA::EQUAL;
}

int operator>=(const IA_Set<IA_Point<int> > &lhs, const IA_Set<IA_Point<int> > &rhs)
{
    IA::LatticeRelation	result;
    result = compare(lhs,rhs);
    return result==IA::P_SUPERSET
	|| result==IA::MP_SUPERSET
	    || result==IA::EQUAL;
}

int operator>(const IA_Set<IA_Point<int> > &lhs, const IA_Set<IA_Point<int> > &rhs)
{
    IA::LatticeRelation	result;
    result = compare(lhs,rhs);
    return result==IA::P_SUPERSET;
}

//
//
//


IA_Set<IA_Point<int> > IA_Set<IA_Point<int> >::empty_pset(unsigned dim)
{
    return IA_Set<IA_Point<int> >(IA_BlackHole_<IA_Point<int> >::s_type(), dim);
}

IA_Set<IA_Point<int> > IA_Set<IA_Point<int> >::universal_pset(unsigned dim)
{
    return IA_Set<IA_Point<int> >(IA_WhiteHole_<IA_Point<int> >::s_type(), dim);
}

IA_Set<IA_Point<int> > IA_boxy_pset(IA_Point<int> p0, IA_Point<int> p1)
{
    return IA_Set<IA_Point<int> >(p0,p1);
}

//
//

IA_Set<IA_Point<int> > concat(int a, const IA_Set<IA_Point<int> >&ps)
{
    return IA_Set<IA_Point<int> >( IA_YoderIPS(a,a,ps) );
}

//
//

void SortIntPointArray(IA_Point<int> *aip, unsigned len)
{
  // We can use qsort because a swap of the bytes in
  // two IntPoints does not do any damage.
  qsort((char*)aip, len, sizeof(*aip), ipcmp);
}

//
//


IA_Set<IA_Point<int> > transpose(const IA_Set<IA_Point<int> > &ps)
{
    if (!ps.boxy() || ps.dim() != 2) {
	ia_throw(Pointset_BadTranspose_Exception(__FILE__, __LINE__));
	return IA_Set<IA_Point<int> >();
    }

    return IA_Set<IA_Point<int> >(ps.inf().derange(1,0), ps.sup().derange(1,0));
}


#ifdef IA_NO_AUTOINST_OF_STATIC_DATA
char IA_WhiteHole_<IA_Point<int> >::type_;
char IA_BlackHole_<IA_Point<int> >::type_;
#endif
		   
