
 

// Emacs: -*- C++ -*-

static char rcsid[] = "$Id: Point.c.m4,v 1.11 93/05/28 15:43:31 thoth Exp $";

/*
        Copyright 1993, Center for Computer Vision and Visualization,
        University of Florida.  All rights reserved.

// $Log:	Point.c.m4,v $
// Revision 1.11  93/05/28  15:43:31  thoth
// fix comment errors.
// 
// Revision 1.10  1993/05/26  16:59:57  rjj
// Added copyright notice
//
// Revision 1.9  93/04/06  22:28:17  rjj
// Changed name of file from "Point.C.m4" to "Point.c.m4"
// Fixed lattice bugs with != by defining functions to invoke ==
// Changed "const {int|double}" to "{int|double}"
// Fixed a leftover FloatPoint name ref in trunc()
// Changed bitwise unsigned parameters to int
// 
//Revision 1.8  93/04/05  23:18:00  rjj
//Many changes to support changes necessary to mutate to the
//  interface defined by newpoint.  Please see the list for
//  Point.h.m4 revison 2.10
//  Error in operator ~ () corrected
//  Lattice operator != is probably still wrong, need to check !
//  m4 defines *_*_RELATION changed to LATTICE_*_*
//  Postfix ++ and -- code added
//  file rearanged to suit my tastes, your milage my vary ;-)
//
//Revision 1.7  92/12/11  01:10:35  rjj
//Modified to support the IA_ prefix naming convention.
//Also put #ifdef IA_PRII around the #pragma stmt
//
//Revision 1.6  92/09/02  14:51:27  rjj
//Corrected code in FloatPoint( IntPoint ) to allocated array of type int
//Changed enorm() cast from (float) to (double)
//
//Revision 1.5  92/09/01  21:08:17  rjj
//Changed max(IA_IntPoint,IA_IntPoint) to be supremum(IA_IntPoint,IA_IntPoint)
//Changed min(IA_IntPoint,IA_IntPoint) to be infimum(IA_IntPoint,IA_IntPoint)
//
//Revision 1.4  92/09/01  20:46:24  rjj
//Added support to ensure float op IntPoint and IntPoint op float yield FloatPoint.
//
//Revision 1.3  92/08/27  12:11:55  jnw
//Added C style comments around Log Messages to fixx gcc-cpp bug
//
//Revision 1.2  92/08/23  13:33:19  thoth
//Cfrontified
//didn't hint to inline for loop stuff any more (braindead compiler).
//hack-around for switch statements that CFront's flow analysis
//can't fully understand.
//throw changed to ia_throw.
//removed duplicate default values for formal parameters (ARM spec).
//
// Revision 1.1  92/08/20  22:24:54  rjj
// Initial revision
// 
// 
// 

*/

#include <math.h>
#include "IntPoint.h"

#ifdef IA_PRII
#pragma implementation
#endif

////////////////////////////////////////////////////////////
// C O N S T R U C T O R S
////////////////////////////////////////////////////////////

IA_IntPoint extend_to_point (int initial_value, unsigned length) {
    IA_IntPoint pt;

    pt.coordinates = new int[pt.dimen = length];

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < length; i++)
	pt.coordinates[i] = initial_value;
#else
    switch (pt.dim()) {
    case 5: pt.coordinates[4] = initial_value;
    case 4: pt.coordinates[3] = initial_value;
    case 3: pt.coordinates[2] = initial_value;
    case 2: pt.coordinates[1] = initial_value;
    case 1: pt.coordinates[0] = initial_value;
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < length; i++)
		    pt.coordinates[i] = initial_value;
	    }
    }
#endif    
    return pt;
}

IA_IntPoint::IA_IntPoint (const IA_IntPoint &ip)
{

// new int[0] creates a pointer to one int; we never use it.
// This is to keep from having to do "if (dim() == 0) ..." everywhere.
// The destructor will delete it, though.
    
    this->coordinates = new int[this->dimen = ip.dim()];

#ifdef NO_LOOP_UNROLLING    
    for (unsigned i=0; i<this->dim(); i++)
	this->coordinates[i] = ip.coordinates[i];
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] = ip.coordinates[4];
    case 4: this->coordinates[3] = ip.coordinates[3];
    case 3: this->coordinates[2] = ip.coordinates[2];
    case 2: this->coordinates[1] = ip.coordinates[1];
    case 1: this->coordinates[0] = ip.coordinates[0];
    case 0: break;
    default:
	    {
		for (unsigned i=0; i<this->dim(); i++)
		    this->coordinates[i] = ip.coordinates[i];
	    }
    }
#endif
}

IA_IntPoint::IA_IntPoint (int *vec, unsigned length, int giveaway)
{
    this->dimen = length;
    if (giveaway) {
	this->coordinates = vec;
	return;
    }
    
    this->coordinates = new int[this->dim()];
    
#ifdef NO_LOOP_UNROLLING    
    for (unsigned length=0; length<this->dim(); length++)
	this->coordinates[length] = vec[length];
#else	
    switch (this->dim()) {
    case 5: this->coordinates[4] = vec[4];
    case 4: this->coordinates[3] = vec[3];
    case 3: this->coordinates[2] = vec[2];
    case 2: this->coordinates[1] = vec[1];
    case 1: this->coordinates[0] = vec[0];
    case 0: break;
    default:
	    {
		for (unsigned length=0; length<this->dim(); length++)
		    this->coordinates[length] = vec[length];
	    }
    }
#endif
}

// IA_IntPoint = concat (IA_IntPoint, IA_IntPoint)
//
// Given a combination of Points and numbers, create a new Point that is
// the concatenation of the arguments.  Lets the user construct larger
// points than the usual 5 arg constructor.

// TO DO -- These could be modified to not check for zero-length
// points (since we now know that "new foo[0]" returns a deletable
// pointer to one foo that we will never access), and also we could
// do some loop unrolling here as well.

IA_IntPoint concat(const IA_IntPoint& lhs, const IA_IntPoint& rhs)
{
    IA_IntPoint result_point;
    
    if (lhs.dim() == 0) {
	if (rhs.dim() == 0)
	    return result_point;
	else 
	    return rhs;
    } else if (rhs.dim() == 0)
	return lhs;

    unsigned int total_dimen = lhs.dim() + rhs.dim();

    result_point.coordinates = new int[result_point.dimen = total_dimen];
    
    for (unsigned int i = 0; i < lhs.dim(); i++) 
	result_point.coordinates[i] = lhs.coordinates[i];

    for (unsigned int j = 0; j < rhs.dim(); j++) 
	result_point.coordinates[i+j] = rhs.coordinates[j];

    return result_point;
}

IA_IntPoint concat(const IA_IntPoint& lhs, int rhs)
{
    IA_IntPoint result_point;

    if (lhs.dim() == 0) 
	return IA_IntPoint (rhs);
    
    unsigned int total_dimen = lhs.dim() + 1;

    result_point.coordinates = new int[total_dimen];
    result_point.dimen = total_dimen;
    
    for (unsigned int i = 0; i < lhs.dim(); i++) 
	result_point.coordinates[i] = lhs.coordinates[i];
    
    result_point.coordinates[i] = rhs;
    
    return result_point;
}

IA_IntPoint concat(int lhs, const IA_IntPoint& rhs)
{
    IA_IntPoint result_point;
    
    if (rhs.dim() == 0)
	return IA_IntPoint (lhs);

    unsigned int total_dimen = rhs.dim() + 1;

    result_point.coordinates = new int[total_dimen];
    result_point.dimen = total_dimen;

    result_point.coordinates[0] = lhs;

    for (unsigned int i = 0; i < rhs.dim(); i++) 
	result_point.coordinates[i+1] = rhs.coordinates[i];

    return result_point;
}

//  cdr(IA_IntPoint, skip=1):
//
//  Given a point of dimension n, return a point made of the n-skip last
//  coordinates (or if skip is negative, the n+skip first coordinates),
//  where skip defaults to 1.

IA_IntPoint cdr(const IA_IntPoint& pt, int skip) {

    unsigned  abs_skip;

    abs_skip = (skip < 0) ? -skip : skip;
    
    if (pt.dim() < abs_skip) {
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);
	return IA_IntPoint ();
    } else if (pt.dim() == skip)
	return IA_IntPoint ();

    if (skip < 0)
	return IA_IntPoint (pt.coordinates, pt.dim()+skip);
    else
	return IA_IntPoint (pt.coordinates+skip, pt.dim()-skip);
}

////////////////////////////////////////////////////////////
// M I S C    O P E R A T I O N S
////////////////////////////////////////////////////////////

IA_IntPoint &IA_IntPoint::operator = (const IA_IntPoint &point) {
    // return directly when self-assignment
    //  (not sure the if is worth it in general)

    if (this == &point)
	return *this;

    // We need to deallocate/reallocate if differing dimens.
    
    if (point.dim() != this->dim()) {
        delete[] this->coordinates;
        this->coordinates = new int[this->dimen = point.dim()];
    }

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)
	this->coordinates[i] = point.coordinates[i];
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] = point.coordinates[4];
    case 4: this->coordinates[3] = point.coordinates[3];
    case 3: this->coordinates[2] = point.coordinates[2];
    case 2: this->coordinates[1] = point.coordinates[1];
    case 1: this->coordinates[0] = point.coordinates[0];
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)
		    this->coordinates[i] = point.coordinates[i];
	    }
    }
#endif
    return *this;
}

// inf returns an empty IA_IntPoint if passed two empty points.

IA_IntPoint inf (const IA_IntPoint& lhs, const IA_IntPoint& rhs) 
{ 
    if (lhs.dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return IA_IntPoint ();  
    }  

    IA_IntPoint p = rhs;

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < p.dim(); i++) 
	p.coordinates[i]
	    = (lhs.coordinates[i] < rhs.coordinates[i])
		?  lhs.coordinates[i] : rhs.coordinates[i];
#else
    switch (p.dim()) {
    case 5: p.coordinates[4] =	
	(lhs.coordinates[4] < rhs.coordinates[4])
	    ? lhs.coordinates[4] : rhs.coordinates[4];
    case 4: p.coordinates[3] = 
	(lhs.coordinates[3] < rhs.coordinates[3])
	    ? lhs.coordinates[3] : rhs.coordinates[3];
    case 3: p.coordinates[2] = 
	(lhs.coordinates[2] < rhs.coordinates[2])
	    ? lhs.coordinates[2] : rhs.coordinates[2];
    case 2: p.coordinates[1] = 
	(lhs.coordinates[1] < rhs.coordinates[1])
	    ? lhs.coordinates[1] : rhs.coordinates[1];
    case 1: p.coordinates[0] = 
	(lhs.coordinates[0] < rhs.coordinates[0])
	    ? lhs.coordinates[0] : rhs.coordinates[0];
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < p.dim(); i++) 
		    p.coordinates[i]
			= (lhs.coordinates[i] < rhs.coordinates[i])
			    ?  lhs.coordinates[i] : rhs.coordinates[i];
	    }
    }
#endif    
    return p; 
}

// sup returns an empty IA_IntPoint if passed two empty points.

IA_IntPoint sup (const IA_IntPoint& lhs, const IA_IntPoint& rhs) 
{ 
    if (lhs.dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return IA_IntPoint ();  
    }  

    IA_IntPoint p = rhs;

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < p.dim(); i++) 
	p.coordinates[i]
	    = (lhs.coordinates[i] > rhs.coordinates[i])
		?  lhs.coordinates[i] : rhs.coordinates[i];
#else
    switch (p.dim()) {
    case 5: p.coordinates[4] =	
	(lhs.coordinates[4] > rhs.coordinates[4])
	    ? lhs.coordinates[4] : rhs.coordinates[4];
    case 4: p.coordinates[3] = 
	(lhs.coordinates[3] > rhs.coordinates[3])
	    ? lhs.coordinates[3] : rhs.coordinates[3];
    case 3: p.coordinates[2] = 
	(lhs.coordinates[2] > rhs.coordinates[2])
	    ? lhs.coordinates[2] : rhs.coordinates[2];
    case 2: p.coordinates[1] = 
	(lhs.coordinates[1] > rhs.coordinates[1])
	    ? lhs.coordinates[1] : rhs.coordinates[1];
    case 1: p.coordinates[0] = 
	(lhs.coordinates[0] > rhs.coordinates[0])
	    ? lhs.coordinates[0] : rhs.coordinates[0];
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < p.dim(); i++) 
		    p.coordinates[i]
			= (lhs.coordinates[i] > rhs.coordinates[i])
			    ?  lhs.coordinates[i] : rhs.coordinates[i];
	    }
    }
#endif    
    return p; 
}

int min (const IA_IntPoint& point)
{

    if (point.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return 0;
    }

    int result = point.coordinates[0];

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 1; i < point.dim(); i++) 
	result =  (point.coordinates[i] < result)
			? point.coordinates[i] : result;
#else
    switch (point.dim()) {
    case 5: result = (point.coordinates[4] < result)
			? point.coordinates[4] : result;
    case 4: result = (point.coordinates[3] < result)
			? point.coordinates[3] : result;
    case 3: result = (point.coordinates[2] < result)
			? point.coordinates[2] : result;
    case 2: result = (point.coordinates[1] < result)
			? point.coordinates[1] : result;
    case 1: break;
    default:
	    {
		for (unsigned i = 1; i < point.dim(); i++) 
		    result =  (point.coordinates[i] < result)
				? point.coordinates[i] : result;
	    }
    }
#endif
    return result; 
}

int max (const IA_IntPoint& point)
{

    if (point.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return 0;
    }

    int result = point.coordinates[0];

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 1; i < point.dim(); i++) 
	result =  (point.coordinates[i] > result)
			? point.coordinates[i] : result;
#else
    switch (point.dim()) {
    case 5: result = (point.coordinates[4] > result)
			? point.coordinates[4] : result;
    case 4: result = (point.coordinates[3] > result)
			? point.coordinates[3] : result;
    case 3: result = (point.coordinates[2] > result)
			? point.coordinates[2] : result;
    case 2: result = (point.coordinates[1] > result)
			? point.coordinates[1] : result;
    case 1: break;
    default:
	    {
		for (unsigned i = 1; i < point.dim(); i++) 
		    result =  (point.coordinates[i] > result)
				? point.coordinates[i] : result;
	    }
    }
#endif
    return result; 
}

ostream &operator << (ostream &stream, const IA_IntPoint& point)
{
    int last_index = point.dim() - 1;
    stream << "<";
    for  ( int i = 0; i < last_index; ++i ) {
	stream << point.coordinates[i] << " ";
    }
    if ( point.dim() > 0 ) {
	stream << point.coordinates[last_index];
    };
    stream << ">";
    return stream;
}

////////////////////////////////////////////////////////////
// I N D E X I N G    O P E R A T I O N S
////////////////////////////////////////////////////////////
// Index derangement: "IA_IntPoint (point_index, point_index..)"
//
// Construct new points from old points, possibly with different
// dimensions and indices rearanged.
// For instance, let  a=Point(10, 20, 30).
// Then b=a.derange(2,0) is equivalent to b=Point(30,10).

IA_IntPoint IA_IntPoint::derange (unsigned i0) const
{
    IA_IntPoint p(0);
    
    if (i0 >= this->dim()) {
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);
	return IA_IntPoint ();
    }
    p.coordinates[0] = this->coordinates[i0];
    return p;
}

IA_IntPoint IA_IntPoint::derange (unsigned i0, unsigned i1) const
{
    IA_IntPoint p((int)0, (int)0);

    if (i0 >= this->dim() || i1 >= this->dim()) {
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);
	return IA_IntPoint ();
    }
    p.coordinates[0] = this->coordinates[i0];
    p.coordinates[1] = this->coordinates[i1];
    return p;
}

IA_IntPoint IA_IntPoint::derange (unsigned i0, unsigned i1, unsigned i2) const
{
    IA_IntPoint p((int)0, (int)0, (int)0);

    if (i0 >= this->dim() || i1 >= this->dim() || i2 >= this->dim()) {
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);
	return IA_IntPoint ();
    }
    
    p.coordinates[0] = this->coordinates[i0];
    p.coordinates[1] = this->coordinates[i1];
    p.coordinates[2] = this->coordinates[i2];
    return p;
}

IA_IntPoint IA_IntPoint::derange (unsigned i0, unsigned i1, unsigned i2, unsigned i3) const
{
    IA_IntPoint p(0,0,0,0);

    if (i0 >= this->dim() || i1 >= this->dim() || i2 >= this->dim() || i3 >= this->dim()) {
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);
	return IA_IntPoint ();
    }

    p.coordinates[0] = this->coordinates[i0];
    p.coordinates[1] = this->coordinates[i1];
    p.coordinates[2] = this->coordinates[i2];
    p.coordinates[3] = this->coordinates[i3];
    return p;
}

IA_IntPoint IA_IntPoint::derange (unsigned i0, unsigned i1, unsigned i2, unsigned i3, unsigned i4) const
{
    IA_IntPoint p(0,0,0,0,0);

    if (i0 >= this->dim() || i1 >= this->dim() || i2 >= this->dim() ||
	i3 >= this->dim() || i4 >= this->dim()) {
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);
	return IA_IntPoint ();
    }

    p.coordinates[0] = this->coordinates[i0];
    p.coordinates[1] = this->coordinates[i1];
    p.coordinates[2] = this->coordinates[i2];
    p.coordinates[3] = this->coordinates[i3];
    p.coordinates[4] = this->coordinates[i4];
    return p;
}
    
IA_IntPoint IA_IntPoint::derange (unsigned *vec, const unsigned vecLen) const
{
    // We need to deallocate/reallocate if differing dimens.
    
    for (unsigned i=0; i<vecLen; i++) {
	if (vec[i] >= this->dim()) {
	    ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__ );
	    return IA_IntPoint ();  // An empty point
	}
    }

    IA_IntPoint  point;  // Our new point
    point.coordinates = new int[point.dimen = vecLen];

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)
	this->coordinates[i] = point.coordinates[vec[i]];
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] = point.coordinates[vec[4]];
    case 4: this->coordinates[3] = point.coordinates[vec[3]];
    case 3: this->coordinates[2] = point.coordinates[vec[2]];
    case 2: this->coordinates[1] = point.coordinates[vec[1]];
    case 1: this->coordinates[0] = point.coordinates[vec[0]];
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)
		    this->coordinates[i] = point.coordinates[vec[i]];
	    }
    }
#endif
    return point;
}

////////////////////////////////////////////////////////////
// A R I T H M E T I C   O P E R A T I O N S
////////////////////////////////////////////////////////////

  
IA_IntPoint& IA_IntPoint::operator += (const IA_IntPoint& rhs) {
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] += rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] += rhs.coordinates[4];  
    case 4: this->coordinates[3] += rhs.coordinates[3];  
    case 3: this->coordinates[2] += rhs.coordinates[2];  
    case 2: this->coordinates[1] += rhs.coordinates[1];  
    case 1: this->coordinates[0] += rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    coordinates[i] += rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator -= (const IA_IntPoint& rhs) {
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] -= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] -= rhs.coordinates[4];  
    case 4: this->coordinates[3] -= rhs.coordinates[3];  
    case 3: this->coordinates[2] -= rhs.coordinates[2];  
    case 2: this->coordinates[1] -= rhs.coordinates[1];  
    case 1: this->coordinates[0] -= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    coordinates[i] -= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator *= (const IA_IntPoint& rhs) {
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] *= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] *= rhs.coordinates[4];  
    case 4: this->coordinates[3] *= rhs.coordinates[3];  
    case 3: this->coordinates[2] *= rhs.coordinates[2];  
    case 2: this->coordinates[1] *= rhs.coordinates[1];  
    case 1: this->coordinates[0] *= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    coordinates[i] *= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator /= (const IA_IntPoint& rhs) {
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] /= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] /= rhs.coordinates[4];  
    case 4: this->coordinates[3] /= rhs.coordinates[3];  
    case 3: this->coordinates[2] /= rhs.coordinates[2];  
    case 2: this->coordinates[1] /= rhs.coordinates[1];  
    case 1: this->coordinates[0] /= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    coordinates[i] /= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}

  
IA_IntPoint& IA_IntPoint::operator += (int rhs) { 
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] += rhs;  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] += rhs;  
    case 4: this->coordinates[3] += rhs;  
    case 3: this->coordinates[2] += rhs;  
    case 2: this->coordinates[1] += rhs;  
    case 1: this->coordinates[0] += rhs;  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] += rhs;  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator -= (int rhs) { 
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] -= rhs;  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] -= rhs;  
    case 4: this->coordinates[3] -= rhs;  
    case 3: this->coordinates[2] -= rhs;  
    case 2: this->coordinates[1] -= rhs;  
    case 1: this->coordinates[0] -= rhs;  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] -= rhs;  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator *= (int rhs) { 
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] *= rhs;  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] *= rhs;  
    case 4: this->coordinates[3] *= rhs;  
    case 3: this->coordinates[2] *= rhs;  
    case 2: this->coordinates[1] *= rhs;  
    case 1: this->coordinates[0] *= rhs;  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] *= rhs;  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator /= (int rhs) { 
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] /= rhs;  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] /= rhs;  
    case 4: this->coordinates[3] /= rhs;  
    case 3: this->coordinates[2] /= rhs;  
    case 2: this->coordinates[1] /= rhs;  
    case 1: this->coordinates[0] /= rhs;  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] /= rhs;  
	    }
    }
#endif
    return *this;  
}
    

IA_IntPoint IA_IntPoint::operator + (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point += rhs;  
}

IA_IntPoint IA_IntPoint::operator - (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point -= rhs;  
}

IA_IntPoint IA_IntPoint::operator * (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point *= rhs;  
}

IA_IntPoint IA_IntPoint::operator / (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point /= rhs;  
}
    

IA_IntPoint operator + (int lhs, const IA_IntPoint& rhs)  
{ 
  IA_IntPoint result_point = extend_to_point (lhs, rhs.dim()); 
  return result_point += rhs;  
}

IA_IntPoint operator - (int lhs, const IA_IntPoint& rhs)  
{ 
  IA_IntPoint result_point = extend_to_point (lhs, rhs.dim()); 
  return result_point -= rhs;  
}

IA_IntPoint operator * (int lhs, const IA_IntPoint& rhs)  
{ 
  IA_IntPoint result_point = extend_to_point (lhs, rhs.dim()); 
  return result_point *= rhs;  
}

IA_IntPoint operator / (int lhs, const IA_IntPoint& rhs)  
{ 
  IA_IntPoint result_point = extend_to_point (lhs, rhs.dim()); 
  return result_point /= rhs;  
}
    

IA_IntPoint IA_IntPoint::operator + (int rhs) const
{ 
    IA_IntPoint result_point = *this;
    return result_point += rhs; 
}

IA_IntPoint IA_IntPoint::operator - (int rhs) const
{ 
    IA_IntPoint result_point = *this;
    return result_point -= rhs; 
}

IA_IntPoint IA_IntPoint::operator * (int rhs) const
{ 
    IA_IntPoint result_point = *this;
    return result_point *= rhs; 
}

IA_IntPoint IA_IntPoint::operator / (int rhs) const
{ 
    IA_IntPoint result_point = *this;
    return result_point /= rhs; 
}
    

IA_IntPoint IA_IntPoint::operator - () const
{
    IA_IntPoint result_point;

    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);
	return result_point;
    }

    result_point.coordinates = new int[result_point.dimen = this->dim()];

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < result_point.dim(); i++)
	result_point.coordinates[i] =  -this->coordinates[i];
#else
    switch (result_point.dim()) {
    case 5: result_point.coordinates[4] = -this->coordinates[4];
    case 4: result_point.coordinates[3] = -this->coordinates[3];
    case 3: result_point.coordinates[2] = -this->coordinates[2];
    case 2: result_point.coordinates[1] = -this->coordinates[1];
    case 1: result_point.coordinates[0] = -this->coordinates[0];
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < result_point.dim(); i++)
		    result_point.coordinates[i] =  - this->coordinates[i];
	    }
    }
#endif
    return result_point;
}

// Prefix ++
IA_IntPoint& IA_IntPoint::operator ++ () {
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);
	return *this;
    }
    *this += 1;
    return  *this;
}

// Prefix --
IA_IntPoint& IA_IntPoint::operator -- () {
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);
	return *this;
    }
    *this -= 1;
    return  *this;
}

// Postfix ++
IA_IntPoint  IA_IntPoint::operator ++ (int) {
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);
	return *this;
    }
    IA_IntPoint  result = *this;
    *this += 1;
    return  result;
}

// Postfix --
IA_IntPoint  IA_IntPoint::operator -- (int) {
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);
	return *this;
    }
    IA_IntPoint  result = *this;
    *this -= 1;
    return  result;
}

// enorm: Euclidean distance from origin.

double IA_IntPoint::enorm () const
{
    double result_number = 0;
    
    for (unsigned i = 0; i < this->dim(); i++)
	result_number = result_number +
	    (double) (this->coordinates[i] * this->coordinates[i]);

    return sqrt(result_number);
}

////////////////////////////////////////////////////////////
// L A T T I C E   O P E R A T I O N S
////////////////////////////////////////////////////////////

// Relational operators

// IA_IntPoint < IA_IntPoint    

 
int IA_IntPoint::operator < (const IA_IntPoint& rhs) const  
{  
    if (this->dim() == 0 || rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  
    if (this->dim() != rhs.dim())  
	return  0;  

#ifdef NO_LOOP_UNROLLING    
    for (unsigned i = 0; i < this->dim() ; i++)
	if (! (this->coordinates[i] < rhs.coordinates[i])) return 0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] < rhs.coordinates[4])) return 0;  
    case 4: if (! (this->coordinates[3] < rhs.coordinates[3])) return 0;  
    case 3: if (! (this->coordinates[2] < rhs.coordinates[2])) return 0;  
    case 2: if (! (this->coordinates[1] < rhs.coordinates[1])) return 0;  
    case 1: if (! (this->coordinates[0] < rhs.coordinates[0])) return 0;  
    case 0: break;
    default: 
	    {
		for (unsigned i = 0; i < this->dim() ; i++)   
		    if (! (this->coordinates[i] < rhs.coordinates[i]))
			return 0;  
	    }
    }
#endif
    return 1;  
}
 
int IA_IntPoint::operator <= (const IA_IntPoint& rhs) const  
{  
    if (this->dim() == 0 || rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  
    if (this->dim() != rhs.dim())  
	return  0;  

#ifdef NO_LOOP_UNROLLING    
    for (unsigned i = 0; i < this->dim() ; i++)
	if (! (this->coordinates[i] <= rhs.coordinates[i])) return 0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] <= rhs.coordinates[4])) return 0;  
    case 4: if (! (this->coordinates[3] <= rhs.coordinates[3])) return 0;  
    case 3: if (! (this->coordinates[2] <= rhs.coordinates[2])) return 0;  
    case 2: if (! (this->coordinates[1] <= rhs.coordinates[1])) return 0;  
    case 1: if (! (this->coordinates[0] <= rhs.coordinates[0])) return 0;  
    case 0: break;
    default: 
	    {
		for (unsigned i = 0; i < this->dim() ; i++)   
		    if (! (this->coordinates[i] <= rhs.coordinates[i]))
			return 0;  
	    }
    }
#endif
    return 1;  
}
 
int IA_IntPoint::operator > (const IA_IntPoint& rhs) const  
{  
    if (this->dim() == 0 || rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  
    if (this->dim() != rhs.dim())  
	return  0;  

#ifdef NO_LOOP_UNROLLING    
    for (unsigned i = 0; i < this->dim() ; i++)
	if (! (this->coordinates[i] > rhs.coordinates[i])) return 0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] > rhs.coordinates[4])) return 0;  
    case 4: if (! (this->coordinates[3] > rhs.coordinates[3])) return 0;  
    case 3: if (! (this->coordinates[2] > rhs.coordinates[2])) return 0;  
    case 2: if (! (this->coordinates[1] > rhs.coordinates[1])) return 0;  
    case 1: if (! (this->coordinates[0] > rhs.coordinates[0])) return 0;  
    case 0: break;
    default: 
	    {
		for (unsigned i = 0; i < this->dim() ; i++)   
		    if (! (this->coordinates[i] > rhs.coordinates[i]))
			return 0;  
	    }
    }
#endif
    return 1;  
}
 
int IA_IntPoint::operator >= (const IA_IntPoint& rhs) const  
{  
    if (this->dim() == 0 || rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  
    if (this->dim() != rhs.dim())  
	return  0;  

#ifdef NO_LOOP_UNROLLING    
    for (unsigned i = 0; i < this->dim() ; i++)
	if (! (this->coordinates[i] >= rhs.coordinates[i])) return 0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] >= rhs.coordinates[4])) return 0;  
    case 4: if (! (this->coordinates[3] >= rhs.coordinates[3])) return 0;  
    case 3: if (! (this->coordinates[2] >= rhs.coordinates[2])) return 0;  
    case 2: if (! (this->coordinates[1] >= rhs.coordinates[1])) return 0;  
    case 1: if (! (this->coordinates[0] >= rhs.coordinates[0])) return 0;  
    case 0: break;
    default: 
	    {
		for (unsigned i = 0; i < this->dim() ; i++)   
		    if (! (this->coordinates[i] >= rhs.coordinates[i]))
			return 0;  
	    }
    }
#endif
    return 1;  
}
 
int IA_IntPoint::operator == (const IA_IntPoint& rhs) const  
{  
    if (this->dim() == 0 || rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  
    if (this->dim() != rhs.dim())  
	return  0;  

#ifdef NO_LOOP_UNROLLING    
    for (unsigned i = 0; i < this->dim() ; i++)
	if (! (this->coordinates[i] == rhs.coordinates[i])) return 0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] == rhs.coordinates[4])) return 0;  
    case 4: if (! (this->coordinates[3] == rhs.coordinates[3])) return 0;  
    case 3: if (! (this->coordinates[2] == rhs.coordinates[2])) return 0;  
    case 2: if (! (this->coordinates[1] == rhs.coordinates[1])) return 0;  
    case 1: if (! (this->coordinates[0] == rhs.coordinates[0])) return 0;  
    case 0: break;
    default: 
	    {
		for (unsigned i = 0; i < this->dim() ; i++)   
		    if (! (this->coordinates[i] == rhs.coordinates[i]))
			return 0;  
	    }
    }
#endif
    return 1;  
}

int IA_IntPoint::operator != (const IA_IntPoint& rhs) const
{
    return ! (*this == rhs);
}

// IA_IntPoint < int

int  IA_IntPoint::operator < (int rhs) const
{  
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] < rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] < rhs)) return  0;  
    case 4: if (! (this->coordinates[3] < rhs)) return  0;  
    case 3: if (! (this->coordinates[2] < rhs)) return  0;  
    case 2: if (! (this->coordinates[1] < rhs)) return  0;  
    case 1: if (! (this->coordinates[0] < rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] < rhs)) return  0;  
	    }
    }
#endif  
    return  1;  
}

int  IA_IntPoint::operator <= (int rhs) const
{  
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] <= rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] <= rhs)) return  0;  
    case 4: if (! (this->coordinates[3] <= rhs)) return  0;  
    case 3: if (! (this->coordinates[2] <= rhs)) return  0;  
    case 2: if (! (this->coordinates[1] <= rhs)) return  0;  
    case 1: if (! (this->coordinates[0] <= rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] <= rhs)) return  0;  
	    }
    }
#endif  
    return  1;  
}

int  IA_IntPoint::operator > (int rhs) const
{  
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] > rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] > rhs)) return  0;  
    case 4: if (! (this->coordinates[3] > rhs)) return  0;  
    case 3: if (! (this->coordinates[2] > rhs)) return  0;  
    case 2: if (! (this->coordinates[1] > rhs)) return  0;  
    case 1: if (! (this->coordinates[0] > rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] > rhs)) return  0;  
	    }
    }
#endif  
    return  1;  
}

int  IA_IntPoint::operator >= (int rhs) const
{  
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] >= rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] >= rhs)) return  0;  
    case 4: if (! (this->coordinates[3] >= rhs)) return  0;  
    case 3: if (! (this->coordinates[2] >= rhs)) return  0;  
    case 2: if (! (this->coordinates[1] >= rhs)) return  0;  
    case 1: if (! (this->coordinates[0] >= rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] >= rhs)) return  0;  
	    }
    }
#endif  
    return  1;  
}

int  IA_IntPoint::operator == (int rhs) const
{  
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] == rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] == rhs)) return  0;  
    case 4: if (! (this->coordinates[3] == rhs)) return  0;  
    case 3: if (! (this->coordinates[2] == rhs)) return  0;  
    case 2: if (! (this->coordinates[1] == rhs)) return  0;  
    case 1: if (! (this->coordinates[0] == rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] == rhs)) return  0;  
	    }
    }
#endif  
    return  1;  
}

int  IA_IntPoint::operator != (int rhs) const
{
    return ! (*this == rhs);
}

// int < IA_IntPoint 

 
int operator < (int lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs < rhs.coordinates[i])) return  0;  
#else
    switch (rhs.dim()) {
    case 5: if (! (lhs < rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs < rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs < rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs < rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs < rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs < rhs.coordinates[i])) return  0;  
	    }
    }
#endif  

    return  1;  
}
 
int operator <= (int lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs <= rhs.coordinates[i])) return  0;  
#else
    switch (rhs.dim()) {
    case 5: if (! (lhs <= rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs <= rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs <= rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs <= rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs <= rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs <= rhs.coordinates[i])) return  0;  
	    }
    }
#endif  

    return  1;  
}
 
int operator > (int lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs > rhs.coordinates[i])) return  0;  
#else
    switch (rhs.dim()) {
    case 5: if (! (lhs > rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs > rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs > rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs > rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs > rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs > rhs.coordinates[i])) return  0;  
	    }
    }
#endif  

    return  1;  
}
 
int operator >= (int lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs >= rhs.coordinates[i])) return  0;  
#else
    switch (rhs.dim()) {
    case 5: if (! (lhs >= rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs >= rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs >= rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs >= rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs >= rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs >= rhs.coordinates[i])) return  0;  
	    }
    }
#endif  

    return  1;  
}
 
int operator == (int lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs == rhs.coordinates[i])) return  0;  
#else
    switch (rhs.dim()) {
    case 5: if (! (lhs == rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs == rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs == rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs == rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs == rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs == rhs.coordinates[i])) return  0;  
	    }
    }
#endif  

    return  1;  
}

int operator != (int lhs, const IA_IntPoint& rhs)  
{
    return !(rhs == lhs);
}

// Intpoint op float
 
int IA_IntPoint::operator <  (double rhs) const
{  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] < rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] < rhs)) return  0;  
    case 4: if (! (this->coordinates[3] < rhs)) return  0;  
    case 3: if (! (this->coordinates[2] < rhs)) return  0;  
    case 2: if (! (this->coordinates[1] < rhs)) return  0;  
    case 1: if (! (this->coordinates[0] < rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] < rhs)) return  0;  
	    }
    }
#endif
    return  1;  
}
 
int IA_IntPoint::operator <=  (double rhs) const
{  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] <= rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] <= rhs)) return  0;  
    case 4: if (! (this->coordinates[3] <= rhs)) return  0;  
    case 3: if (! (this->coordinates[2] <= rhs)) return  0;  
    case 2: if (! (this->coordinates[1] <= rhs)) return  0;  
    case 1: if (! (this->coordinates[0] <= rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] <= rhs)) return  0;  
	    }
    }
#endif
    return  1;  
}
 
int IA_IntPoint::operator >  (double rhs) const
{  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] > rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] > rhs)) return  0;  
    case 4: if (! (this->coordinates[3] > rhs)) return  0;  
    case 3: if (! (this->coordinates[2] > rhs)) return  0;  
    case 2: if (! (this->coordinates[1] > rhs)) return  0;  
    case 1: if (! (this->coordinates[0] > rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] > rhs)) return  0;  
	    }
    }
#endif
    return  1;  
}
 
int IA_IntPoint::operator >=  (double rhs) const
{  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] >= rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] >= rhs)) return  0;  
    case 4: if (! (this->coordinates[3] >= rhs)) return  0;  
    case 3: if (! (this->coordinates[2] >= rhs)) return  0;  
    case 2: if (! (this->coordinates[1] >= rhs)) return  0;  
    case 1: if (! (this->coordinates[0] >= rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] >= rhs)) return  0;  
	    }
    }
#endif
    return  1;  
}
 
int IA_IntPoint::operator ==  (double rhs) const
{  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < this->dim(); i++)  
	if (! (this->coordinates[i] == rhs)) return  0;  
#else
    switch (this->dim()) {
    case 5: if (! (this->coordinates[4] == rhs)) return  0;  
    case 4: if (! (this->coordinates[3] == rhs)) return  0;  
    case 3: if (! (this->coordinates[2] == rhs)) return  0;  
    case 2: if (! (this->coordinates[1] == rhs)) return  0;  
    case 1: if (! (this->coordinates[0] == rhs)) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < this->dim(); i++)  
		    if (! (this->coordinates[i] == rhs)) return  0;  
	    }
    }
#endif
    return  1;  
}

int IA_IntPoint::operator !=  (double rhs) const
{
    return ! (*this == rhs);
}

// float < 

int operator < (double lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs < rhs.coordinates[i]))  
	    return  0;  

#else
    switch (rhs.dim()) {
    case 5: if (! (lhs < rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs < rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs < rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs < rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs < rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs < rhs.coordinates[i])) return  0;  
	    }
    }
#endif    
    return  1;  
}

int operator <= (double lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs <= rhs.coordinates[i]))  
	    return  0;  

#else
    switch (rhs.dim()) {
    case 5: if (! (lhs <= rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs <= rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs <= rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs <= rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs <= rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs <= rhs.coordinates[i])) return  0;  
	    }
    }
#endif    
    return  1;  
}

int operator > (double lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs > rhs.coordinates[i]))  
	    return  0;  

#else
    switch (rhs.dim()) {
    case 5: if (! (lhs > rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs > rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs > rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs > rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs > rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs > rhs.coordinates[i])) return  0;  
	    }
    }
#endif    
    return  1;  
}

int operator >= (double lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs >= rhs.coordinates[i]))  
	    return  0;  

#else
    switch (rhs.dim()) {
    case 5: if (! (lhs >= rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs >= rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs >= rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs >= rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs >= rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs >= rhs.coordinates[i])) return  0;  
	    }
    }
#endif    
    return  1;  
}

int operator == (double lhs, const IA_IntPoint& rhs)  
{  
    if (rhs.dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return  0;  
    }	  

#ifdef NO_LOOP_UNROLLING
    for (unsigned int i = 0; i < rhs.dim(); i++)  
	if (! (lhs == rhs.coordinates[i]))  
	    return  0;  

#else
    switch (rhs.dim()) {
    case 5: if (! (lhs == rhs.coordinates[4])) return  0;  
    case 4: if (! (lhs == rhs.coordinates[3])) return  0;  
    case 3: if (! (lhs == rhs.coordinates[2])) return  0;  
    case 2: if (! (lhs == rhs.coordinates[1])) return  0;  
    case 1: if (! (lhs == rhs.coordinates[0])) return  0;
    case 0: break;
    default:
	    {
		for (unsigned int i = 0; i < rhs.dim(); i++)  
		    if (! (lhs == rhs.coordinates[i])) return  0;  
	    }
    }
#endif    
    return  1;  
}

int operator != (double lhs, const IA_IntPoint& rhs)  
{
    return ! (rhs == lhs);
}

//  Point comparison: Pointcmp(IA_IntPoint, IA_IntPoint)
//
// A sort of lexicographic comparison of Points, except that
// points are zero extended for comparison, e.g., <1 0> is equal
// to <1>;  <> is equal <0 0 0>, etc. Returns 0 for equality, -1
// if the first is less than the second, 1 if the first is greater
// than the second,

int Pointcmp(const IA_IntPoint &lhs, const IA_IntPoint &rhs)
{
    if (lhs.dim() > rhs.dim()) {
	for (unsigned i=0; i < rhs.dim(); i++) {
	    if (lhs[i] < rhs[i])
		return -1;
	    else if (lhs[i] > rhs[i])
		return 1;
	}
	for ( ; i<lhs.dim(); i++) {
	    if (lhs[i] < 0)
		return -1;
	    else if (lhs[i] > 0)
		return 1;
	}
    } else {
	for (unsigned i=0; i < lhs.dim(); i++) {
	    if (lhs[i] < rhs[i])
		return -1;
	    else if (lhs[i] > rhs[i])
		return 1;
	}
	for ( ; i < rhs.dim(); i++) {
	    if (0 < rhs[i])
		return  -1;
	    else if (0 > rhs[i])
		return  1;
	}
    }
    return 0;

    // If this were to be used for a lexicographic sorter, then we would
    // need the following additional code:
    //
    // At this point the zero-extended Coordinates do not differ in value,
    // so check for number of dimensions; shorter Points are considered
    // less than longer Points.
    // 
    //     if (lhs.dim() == rhs.dim())
    // 		return 0;
    //     else if (lhs.dim() > rhs.dim())
    // 		return 1;
    //     else
    // 		return -1;

}

////////////////////////////////////////////////////////////
// B I T W I S E   O P E R A T I O N S
////////////////////////////////////////////////////////////

IA_IntPoint IA_IntPoint::operator ~ () const
{
    IA_IntPoint result_point;
    
    if (this->dim() == 0) {
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);
	return *this;
    }

    result_point.coordinates = new int[result_point.dimen = this->dim()];

#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < result_point.dim(); i++)
	result_point.coordinates[i] =  ~(this->coordinates[i]);
#else
    switch (result_point.dim()) {
    case 5: result_point.coordinates[4] = ~(this->coordinates[4]);
    case 4: result_point.coordinates[3] = ~(this->coordinates[3]);
    case 3: result_point.coordinates[2] = ~(this->coordinates[2]);
    case 2: result_point.coordinates[1] = ~(this->coordinates[1]);
    case 1: result_point.coordinates[0] = ~(this->coordinates[0]);
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < result_point.dim(); i++)
		    result_point.coordinates[i] = ~(this->coordinates[i]);
	    }
    }
#endif    
    return result_point;
}

  
IA_IntPoint& IA_IntPoint::operator %= (const IA_IntPoint& rhs) {  
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] %= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] %= rhs.coordinates[4];  
    case 4: this->coordinates[3] %= rhs.coordinates[3];  
    case 3: this->coordinates[2] %= rhs.coordinates[2];  
    case 2: this->coordinates[1] %= rhs.coordinates[1];  
    case 1: this->coordinates[0] %= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] %= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator <<= (const IA_IntPoint& rhs) {  
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] <<= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] <<= rhs.coordinates[4];  
    case 4: this->coordinates[3] <<= rhs.coordinates[3];  
    case 3: this->coordinates[2] <<= rhs.coordinates[2];  
    case 2: this->coordinates[1] <<= rhs.coordinates[1];  
    case 1: this->coordinates[0] <<= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] <<= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator >>= (const IA_IntPoint& rhs) {  
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] >>= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] >>= rhs.coordinates[4];  
    case 4: this->coordinates[3] >>= rhs.coordinates[3];  
    case 3: this->coordinates[2] >>= rhs.coordinates[2];  
    case 2: this->coordinates[1] >>= rhs.coordinates[1];  
    case 1: this->coordinates[0] >>= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] >>= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator &= (const IA_IntPoint& rhs) {  
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] &= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] &= rhs.coordinates[4];  
    case 4: this->coordinates[3] &= rhs.coordinates[3];  
    case 3: this->coordinates[2] &= rhs.coordinates[2];  
    case 2: this->coordinates[1] &= rhs.coordinates[1];  
    case 1: this->coordinates[0] &= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] &= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator ^= (const IA_IntPoint& rhs) {  
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] ^= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] ^= rhs.coordinates[4];  
    case 4: this->coordinates[3] ^= rhs.coordinates[3];  
    case 3: this->coordinates[2] ^= rhs.coordinates[2];  
    case 2: this->coordinates[1] ^= rhs.coordinates[1];  
    case 1: this->coordinates[0] ^= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] ^= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator |= (const IA_IntPoint& rhs) {  
    if (this->dim() != rhs.dim()) {  
	ia_throw(IA::POINT_INDEX_MISMATCH, __FILE__, __LINE__);  
	return *this;  
    }  
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] |= rhs.coordinates[i];  
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] |= rhs.coordinates[4];  
    case 4: this->coordinates[3] |= rhs.coordinates[3];  
    case 3: this->coordinates[2] |= rhs.coordinates[2];  
    case 2: this->coordinates[1] |= rhs.coordinates[1];  
    case 1: this->coordinates[0] |= rhs.coordinates[0];  
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] |= rhs.coordinates[i];  
	    }
    }
#endif
    return *this;  
}

  
IA_IntPoint& IA_IntPoint::operator %= (int rhs) {
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] %= rhs;
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] %= rhs;
    case 4: this->coordinates[3] %= rhs;
    case 3: this->coordinates[2] %= rhs;
    case 2: this->coordinates[1] %= rhs;
    case 1: this->coordinates[0] %= rhs;
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] %= rhs;
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator <<= (int rhs) {
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] <<= rhs;
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] <<= rhs;
    case 4: this->coordinates[3] <<= rhs;
    case 3: this->coordinates[2] <<= rhs;
    case 2: this->coordinates[1] <<= rhs;
    case 1: this->coordinates[0] <<= rhs;
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] <<= rhs;
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator >>= (int rhs) {
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] >>= rhs;
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] >>= rhs;
    case 4: this->coordinates[3] >>= rhs;
    case 3: this->coordinates[2] >>= rhs;
    case 2: this->coordinates[1] >>= rhs;
    case 1: this->coordinates[0] >>= rhs;
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] >>= rhs;
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator &= (int rhs) {
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] &= rhs;
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] &= rhs;
    case 4: this->coordinates[3] &= rhs;
    case 3: this->coordinates[2] &= rhs;
    case 2: this->coordinates[1] &= rhs;
    case 1: this->coordinates[0] &= rhs;
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] &= rhs;
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator ^= (int rhs) {
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] ^= rhs;
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] ^= rhs;
    case 4: this->coordinates[3] ^= rhs;
    case 3: this->coordinates[2] ^= rhs;
    case 2: this->coordinates[1] ^= rhs;
    case 1: this->coordinates[0] ^= rhs;
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] ^= rhs;
	    }
    }
#endif
    return *this;  
}
  
IA_IntPoint& IA_IntPoint::operator |= (int rhs) {
    if (this->dim() == 0) {  
	ia_throw(IA::POINT_UNINITIALIZED, __FILE__, __LINE__);  
	return *this;  
    }  
#ifdef NO_LOOP_UNROLLING
    for (unsigned i = 0; i < this->dim(); i++)  
	this->coordinates[i] |= rhs;
#else
    switch (this->dim()) {
    case 5: this->coordinates[4] |= rhs;
    case 4: this->coordinates[3] |= rhs;
    case 3: this->coordinates[2] |= rhs;
    case 2: this->coordinates[1] |= rhs;
    case 1: this->coordinates[0] |= rhs;
    case 0: break;
    default:
	    {
		for (unsigned i = 0; i < this->dim(); i++)  
		    this->coordinates[i] |= rhs;
	    }
    }
#endif
    return *this;  
}

IA_IntPoint IA_IntPoint::operator % (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point %= rhs;  
}

IA_IntPoint IA_IntPoint::operator << (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point <<= rhs;  
}

IA_IntPoint IA_IntPoint::operator >> (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point >>= rhs;  
}

IA_IntPoint IA_IntPoint::operator & (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point &= rhs;  
}

IA_IntPoint IA_IntPoint::operator ^ (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point ^= rhs;  
}

IA_IntPoint IA_IntPoint::operator | (const IA_IntPoint& rhs) const  
{ 
   IA_IntPoint result_point = *this;    
   return result_point |= rhs;  
}

IA_IntPoint operator % (int lhs, const IA_IntPoint& rhs) {
    IA_IntPoint result_point = extend_to_point (lhs, rhs.dim());  
    return result_point %= rhs;  
}

IA_IntPoint operator << (int lhs, const IA_IntPoint& rhs) {
    IA_IntPoint result_point = extend_to_point (lhs, rhs.dim());  
    return result_point <<= rhs;  
}

IA_IntPoint operator >> (int lhs, const IA_IntPoint& rhs) {
    IA_IntPoint result_point = extend_to_point (lhs, rhs.dim());  
    return result_point >>= rhs;  
}

IA_IntPoint operator & (int lhs, const IA_IntPoint& rhs) {
    IA_IntPoint result_point = extend_to_point (lhs, rhs.dim());  
    return result_point &= rhs;  
}

IA_IntPoint operator ^ (int lhs, const IA_IntPoint& rhs) {
    IA_IntPoint result_point = extend_to_point (lhs, rhs.dim());  
    return result_point ^= rhs;  
}

IA_IntPoint operator | (int lhs, const IA_IntPoint& rhs) {
    IA_IntPoint result_point = extend_to_point (lhs, rhs.dim());  
    return result_point |= rhs;  
}

IA_IntPoint IA_IntPoint::operator % (int rhs) const
{ 
   IA_IntPoint result_point = *this;    
   return result_point %= rhs;  
}

IA_IntPoint IA_IntPoint::operator << (int rhs) const
{ 
   IA_IntPoint result_point = *this;    
   return result_point <<= rhs;  
}

IA_IntPoint IA_IntPoint::operator >> (int rhs) const
{ 
   IA_IntPoint result_point = *this;    
   return result_point >>= rhs;  
}

IA_IntPoint IA_IntPoint::operator & (int rhs) const
{ 
   IA_IntPoint result_point = *this;    
   return result_point &= rhs;  
}

IA_IntPoint IA_IntPoint::operator ^ (int rhs) const
{ 
   IA_IntPoint result_point = *this;    
   return result_point ^= rhs;  
}

IA_IntPoint IA_IntPoint::operator | (int rhs) const
{ 
   IA_IntPoint result_point = *this;    
   return result_point |= rhs;  
}

