// Emacs -*- C++ -*-

// 	Copyright 1994, Center for Computer Vision and Visualization,
// 	University of Florida.  All rights reserved.

//
// RGB.h
//
// Defines the class IA_RGB, which can be employed to represent
// colors in Red-Green-Blue space with one byte precision for each
// color component

//
//  $Log: RGB.h,v $
// Revision 1.7  1994/09/06  18:58:44  fjsoria
// corrected "warning: return to `int' from `double'" by casting to int
//
// Revision 1.6  1994/08/17  02:15:09  gmt
// Now includes compat.h instead of ia.h or sys/types.h .
//
// Revision 1.5  1994/08/16  03:56:07  gmt
// Made to include <ia.h> instead of <sys/types.h> since types are now
// dealt with in one place.  Has a Mac specific ifdef wrapper for now,
// but should change to all soon.
// Also changed a case of "int friend" to "friend int" since SC++ v7 did
// not like it.
//
// Revision 1.4  1994/04/15  17:44:57  jnw
// Added decl for pseudocolor
//
// Revision 1.3  1994/03/03  22:57:20  jnw
// added include of sys/types.h for u_char
//
// Revision 1.2  1994/02/14  18:30:26  jnw
// Added initial comment
//
// Revision 1.1  1994/02/14  18:23:49  jnw
// Initial revision
//

#ifndef IA_RGB_h_
#define IA_RGB_h_

#include <iostream.h>

#include "compat.h"

class IA_RGB {
private:
    u_char rgb_array[3];
public:
    IA_RGB() {}

    IA_RGB(int r, int g, int b) {
	rgb_array[0] = r;
	rgb_array[1] = g;
	rgb_array[2] = b;
    }

    IA_RGB(u_char* ar) {
	rgb_array[0] = ar[0];
	rgb_array[1] = ar[1];
	rgb_array[2] = ar[2];
    }

    IA_RGB(const IA_RGB &v){
	rgb_array[0] = v.rgb_array[0];
	rgb_array[1] = v.rgb_array[1];
	rgb_array[2] = v.rgb_array[2];
    }
    
    IA_RGB(int gray) {
	rgb_array[0] = gray;
	rgb_array[1] = gray;
	rgb_array[2] = gray;
    }

    int to_int() const {
	// constants cribbed from ppmtopgm man page
	return (int) ((  .299*rgb_array[0]
		+ .587*rgb_array[1]
		+ .114*rgb_array[2])/3);
    }

    float to_float() const {
	// constants cribbed from ppmtopgm man page
	return (  .299*rgb_array[0]
		+ .587*rgb_array[1]
		+ .114*rgb_array[2])/3.0;
    }

    IA_RGB& operator=(const IA_RGB &v) { 
	rgb_array[0] = v.rgb_array[0];
	rgb_array[1] = v.rgb_array[1];
	rgb_array[2] = v.rgb_array[2];
	return *this;
    }

    IA_RGB& operator=(int v) { 
	rgb_array[0] = v;
	rgb_array[1] = v;
	rgb_array[2] = v;
	return *this;
    }

    u_char& red() {
	return rgb_array[0];
    }
    
    u_char& green() {
	return rgb_array[1];
    }
    
    u_char& blue() {
	return rgb_array[2];
    }

    friend int operator==(IA_RGB lhs, IA_RGB rhs) {
	return lhs.rgb_array[0] == rhs.rgb_array[0]
	    && lhs.rgb_array[1] == rhs.rgb_array[1]
	    && lhs.rgb_array[2] == rhs.rgb_array[2];
    }

    friend int operator!=(IA_RGB lhs, IA_RGB rhs) {
	return !(lhs == rhs);
    }

    friend ostream& operator<<(ostream &o, IA_RGB rgb) {
	return o << "#RGB<" << rgb.rgb_array[0] << ","
		 << rgb.rgb_array[1] << ","
		 << rgb.rgb_array[2] << ">";
    }
};

IA_RGB pseudocolor(u_char);
#endif // IA_RGB_h_
