/* Licensed under a BSD license. See license.html for license */
"use strict";

var vertexShaderSource = `#version 300 es
uniform mat4 u_worldViewProjection;
uniform vec3 u_lightWorldPos;
uniform mat4 u_world;
uniform mat4 u_viewInverse;
uniform mat4 u_worldInverseTranspose;

in vec4 a_position;
in vec3 a_normal;

out vec4 v_position;
out vec3 v_normal;
out vec3 v_surfaceToLight;
out vec3 v_surfaceToView;

void main() {
  v_position = (u_worldViewProjection * a_position);
  v_normal = (u_worldInverseTranspose * vec4(a_normal, 0)).xyz;
  v_surfaceToLight = u_lightWorldPos - (u_world * a_position).xyz;
  v_surfaceToView = (u_viewInverse[3] - (u_world * a_position)).xyz;
  gl_Position = v_position;
}
`;

var fragmentShaderSource = `#version 300 es
#ifdef GL_FRAGMENT_PRECISION_HIGH
  precision highp float;
#else
  precision mediump float;
#endif

in vec4 v_position;
in vec3 v_normal;
in vec3 v_surfaceToLight;
in vec3 v_surfaceToView;

out vec4 outColor;

uniform vec4 u_lightColor;
uniform vec4 u_ambient;
uniform vec4 u_diffuse;
uniform vec4 u_specular;
uniform float u_shininess;
uniform float u_specularFactor;

vec4 lit(float l ,float h, float m) {
  return vec4(1.0,
              pow(abs(l), 2.5),
              (l > 0.0) ? pow(max(0.0, h), m) : 0.0,
              1.0);
}

void main() {
  float depth = gl_FragCoord.z / gl_FragCoord.w;
  vec4 diffuseColor = u_diffuse;
  vec3 a_normal = normalize(v_normal);
  vec3 surfaceToLight = normalize(v_surfaceToLight);
  vec3 surfaceToView = normalize(v_surfaceToView);
  vec3 halfVector = normalize(surfaceToLight + surfaceToView);
  vec4 litR = lit(dot(a_normal, surfaceToLight),
                    dot(a_normal, halfVector), u_shininess);
  outColor = vec4((
  u_lightColor * (diffuseColor * litR.y + diffuseColor * u_ambient +
                u_specular * litR.z * u_specularFactor)).rgb,
      diffuseColor.a);
}
`;

var model = {"position":{"numCompnents":3,"data":[0.195329,0.1,-0.690561,0.238931,0.1,-0.689215,0.209,0.1,-0.691,0.181796,0.1,-0.689241,0.238931,0.1,-0.689215,0.195329,0.1,-0.690561,0.168375,0.1,-0.687031,0.238931,0.1,-0.689215,0.181796,0.1,-0.689241,0.168375,0.1,-0.687031,0.267116,0.1,-0.683972,0.238931,0.1,-0.689215,0.155037,0.1,-0.683926,0.267116,0.1,-0.683972,0.168375,0.1,-0.687031,0.155037,0.1,-0.683926,0.293391,0.1,-0.675438,0.267116,0.1,-0.683972,0.141755,0.1,-0.679918,0.293391,0.1,-0.675438,0.155037,0.1,-0.683926,0.1285,0.1,-0.675,0.293391,0.1,-0.675438,0.141755,0.1,-0.679918,0.1285,0.1,-0.675,0.317593,0.1,-0.663778,0.293391,0.1,-0.675438,0.115245,0.1,-0.669166,0.317593,0.1,-0.663778,0.1285,0.1,-0.675,0.101963,0.1,-0.662407,0.317593,0.1,-0.663778,0.115245,0.1,-0.669166,0.101963,0.1,-0.662407,0.339558,0.1,-0.64916,0.317593,0.1,-0.663778,0.088625,0.1,-0.654719,0.339558,0.1,-0.64916,0.101963,0.1,-0.662407,0.075204,0.1,-0.646093,0.339558,0.1,-0.64916,0.088625,0.1,-0.654719,0.075204,0.1,-0.646093,0.359125,0.1,-0.63175,0.339558,0.1,-0.64916,0.061671,0.1,-0.636522,0.359125,0.1,-0.63175,0.075204,0.1,-0.646093,0.048,0.1,-0.626,0.359125,0.1,-0.63175,0.061671,0.1,-0.636522,0.048,0.1,-0.626,0.376129,0.1,-0.611715,0.359125,0.1,-0.63175,0.048,0.1,-0.503,0.061648,0.1,-0.519515,0.048,0.1,-0.626,0.061648,0.1,-0.519515,0.075102,0.1,-0.53412,0.048,0.1,-0.626,0.075102,0.1,-0.53412,0.088375,0.1,-0.546906,0.048,0.1,-0.626,0.088375,0.1,-0.546906,0.101481,0.1,-0.557963,0.048,0.1,-0.626,0.101481,0.1,-0.557963,0.114435,0.1,-0.567381,0.048,0.1,-0.626,0.114435,0.1,-0.567381,0.12725,0.1,-0.57525,0.048,0.1,-0.626,0.12725,0.1,-0.57525,0.13994,0.1,-0.581661,0.048,0.1,-0.626,0.13994,0.1,-0.581661,0.152519,0.1,-0.586704,0.048,0.1,-0.626,0.152519,0.1,-0.586704,0.165,0.1,-0.590469,0.048,0.1,-0.626,0.165,0.1,-0.590469,0.177398,0.1,-0.593046,0.048,0.1,-0.626,0.177398,0.1,-0.593046,0.189727,0.1,-0.594527,0.048,0.1,-0.626,0.189727,0.1,-0.594527,0.376129,0.1,-0.611715,0.048,0.1,-0.626,0.189727,0.1,-0.594527,0.202,0.1,-0.595,0.376129,0.1,-0.611715,0.202,0.1,-0.595,0.390407,0.1,-0.589222,0.376129,0.1,-0.611715,0.218377,0.1,-0.593914,0.390407,0.1,-0.589222,0.202,0.1,-0.595,0.233931,0.1,-0.590727,0.390407,0.1,-0.589222,0.218377,0.1,-0.593914,0.248547,0.1,-0.585547,0.390407,0.1,-0.589222,0.233931,0.1,-0.590727,0.248547,0.1,-0.585547,0.401797,0.1,-0.564438,0.390407,0.1,-0.589222,0.262111,0.1,-0.578481,0.401797,0.1,-0.564438,0.248547,0.1,-0.585547,0.274509,0.1,-0.569638,0.401797,0.1,-0.564438,0.262111,0.1,-0.578481,0.285625,0.1,-0.559125,0.401797,0.1,-0.564438,0.274509,0.1,-0.569638,0.285625,0.1,-0.559125,0.410134,0.1,-0.537528,0.401797,0.1,-0.564438,0.295345,0.1,-0.547049,0.410134,0.1,-0.537528,0.285625,0.1,-0.559125,0.303556,0.1,-0.533518,0.410134,0.1,-0.537528,0.295345,0.1,-0.547049,0.303556,0.1,-0.533518,0.415256,0.1,-0.50866,0.410134,0.1,-0.537528,0.310141,0.1,-0.518641,0.415256,0.1,-0.50866,0.303556,0.1,-0.533518,0.314986,0.1,-0.502523,0.415256,0.1,-0.50866,0.310141,0.1,-0.518641,0.314986,0.1,-0.502523,0.417,0.1,-0.478,0.415256,0.1,-0.50866,0.317977,0.1,-0.485274,0.417,0.1,-0.478,0.314986,0.1,-0.502523,0.319,0.1,-0.467,0.417,0.1,-0.478,0.317977,0.1,-0.485274,0.319,0.1,-0.467,0.415731,0.1,-0.450564,0.417,0.1,-0.478,0.318667,0.1,-0.453204,0.415731,0.1,-0.450564,0.319,0.1,-0.467,0.317667,0.1,-0.440213,0.415731,0.1,-0.450564,0.318667,0.1,-0.453204,0.317667,0.1,-0.440213,0.411852,0.1,-0.423509,0.415731,0.1,-0.450564,0.316,0.1,-0.427875,0.411852,0.1,-0.423509,0.317667,0.1,-0.440213,0.313667,0.1,-0.416037,0.411852,0.1,-0.423509,0.316,0.1,-0.427875,0.313667,0.1,-0.416037,0.40525,0.1,-0.396469,0.411852,0.1,-0.423509,0.310667,0.1,-0.404546,0.40525,0.1,-0.396469,0.313667,0.1,-0.416037,0.307,0.1,-0.39325,0.40525,0.1,-0.396469,0.310667,0.1,-0.404546,0.307,0.1,-0.39325,0.395815,0.1,-0.369074,0.40525,0.1,-0.396469,0.302667,0.1,-0.381995,0.395815,0.1,-0.369074,0.307,0.1,-0.39325,0.297667,0.1,-0.37063,0.395815,0.1,-0.369074,0.302667,0.1,-0.381995,0.292,0.1,-0.359,0.395815,0.1,-0.369074,0.297667,0.1,-0.37063,0.292,0.1,-0.359,0.383435,0.1,-0.340957,0.395815,0.1,-0.369074,0.285667,0.1,-0.346954,0.383435,0.1,-0.340957,0.292,0.1,-0.359,0.278667,0.1,-0.334338,0.383435,0.1,-0.340957,0.285667,0.1,-0.346954,0.278667,0.1,-0.334338,0.368,0.1,-0.31175,0.383435,0.1,-0.340957,0.271,0.1,-0.321,0.368,0.1,-0.31175,0.278667,0.1,-0.334338,0.255585,0.1,-0.295763,0.368,0.1,-0.31175,0.271,0.1,-0.321,0.255585,0.1,-0.295763,0.349398,0.1,-0.281085,0.368,0.1,-0.31175,0.238514,0.1,-0.269685,0.349398,0.1,-0.281085,0.255585,0.1,-0.295763,0.238514,0.1,-0.269685,0.327519,0.1,-0.248593,0.349398,0.1,-0.281085,0.220047,0.1,-0.242969,0.327519,0.1,-0.248593,0.238514,0.1,-0.269685,0.220047,0.1,-0.242969,0.30225,0.1,-0.213906,0.327519,0.1,-0.248593,0.200444,0.1,-0.215815,0.30225,0.1,-0.213906,0.220047,0.1,-0.242969,0.179967,0.1,-0.188425,0.30225,0.1,-0.213906,0.200444,0.1,-0.215815,0.179967,0.1,-0.188425,0.273481,0.1,-0.176657,0.30225,0.1,-0.213906,0.158875,0.1,-0.161,0.273481,0.1,-0.176657,0.179967,0.1,-0.188425,0.158875,0.1,-0.161,0.241102,0.1,-0.136478,0.273481,0.1,-0.176657,0.137429,0.1,-0.133742,0.241102,0.1,-0.136478,0.158875,0.1,-0.161,0.137429,0.1,-0.133742,0.205,0.1,-0.093,0.241102,0.1,-0.136478,0.115889,0.1,-0.106852,0.205,0.1,-0.093,0.137429,0.1,-0.133742,0.094516,0.1,-0.080531,0.205,0.1,-0.093,0.115889,0.1,-0.106852,0.094516,0.1,-0.080531,0.466,0.1,-0.093,0.205,0.1,-0.093,0.094516,0.1,-0.080531,0.466,0.1,0,0.466,0.1,-0.093,0.073569,0.1,-0.054981,0.466,0.1,0,0.094516,0.1,-0.080531,0.053311,0.1,-0.030404,0.466,0.1,0,0.073569,0.1,-0.054981,0.034,0.1,-0.007,0.466,0.1,0,0.053311,0.1,-0.030404,0.034,0.1,0,0.466,0.1,0,0.034,0.1,-0.007,0.238931,-0.1,-0.689215,0.195329,-0.1,-0.690561,0.209,-0.1,-0.691,0.238931,-0.1,-0.689215,0.181796,-0.1,-0.689241,0.195329,-0.1,-0.690561,0.238931,-0.1,-0.689215,0.168375,-0.1,-0.687031,0.181796,-0.1,-0.689241,0.267116,-0.1,-0.683972,0.168375,-0.1,-0.687031,0.238931,-0.1,-0.689215,0.267116,-0.1,-0.683972,0.155037,-0.1,-0.683926,0.168375,-0.1,-0.687031,0.293391,-0.1,-0.675438,0.155037,-0.1,-0.683926,0.267116,-0.1,-0.683972,0.293391,-0.1,-0.675438,0.141755,-0.1,-0.679918,0.155037,-0.1,-0.683926,0.293391,-0.1,-0.675438,0.1285,-0.1,-0.675,0.141755,-0.1,-0.679918,0.317593,-0.1,-0.663778,0.1285,-0.1,-0.675,0.293391,-0.1,-0.675438,0.317593,-0.1,-0.663778,0.115245,-0.1,-0.669166,0.1285,-0.1,-0.675,0.317593,-0.1,-0.663778,0.101963,-0.1,-0.662407,0.115245,-0.1,-0.669166,0.339558,-0.1,-0.64916,0.101963,-0.1,-0.662407,0.317593,-0.1,-0.663778,0.339558,-0.1,-0.64916,0.088625,-0.1,-0.654719,0.101963,-0.1,-0.662407,0.339558,-0.1,-0.64916,0.075204,-0.1,-0.646093,0.088625,-0.1,-0.654719,0.359125,-0.1,-0.63175,0.075204,-0.1,-0.646093,0.339558,-0.1,-0.64916,0.359125,-0.1,-0.63175,0.061671,-0.1,-0.636522,0.075204,-0.1,-0.646093,0.359125,-0.1,-0.63175,0.048,-0.1,-0.626,0.061671,-0.1,-0.636522,0.376129,-0.1,-0.611715,0.048,-0.1,-0.626,0.359125,-0.1,-0.63175,0.061648,-0.1,-0.519515,0.048,-0.1,-0.503,0.048,-0.1,-0.626,0.075102,-0.1,-0.53412,0.061648,-0.1,-0.519515,0.048,-0.1,-0.626,0.088375,-0.1,-0.546906,0.075102,-0.1,-0.53412,0.048,-0.1,-0.626,0.101481,-0.1,-0.557963,0.088375,-0.1,-0.546906,0.048,-0.1,-0.626,0.114435,-0.1,-0.567381,0.101481,-0.1,-0.557963,0.048,-0.1,-0.626,0.12725,-0.1,-0.57525,0.114435,-0.1,-0.567381,0.048,-0.1,-0.626,0.13994,-0.1,-0.581661,0.12725,-0.1,-0.57525,0.048,-0.1,-0.626,0.152519,-0.1,-0.586704,0.13994,-0.1,-0.581661,0.048,-0.1,-0.626,0.165,-0.1,-0.590469,0.152519,-0.1,-0.586704,0.048,-0.1,-0.626,0.177398,-0.1,-0.593046,0.165,-0.1,-0.590469,0.048,-0.1,-0.626,0.189727,-0.1,-0.594527,0.177398,-0.1,-0.593046,0.048,-0.1,-0.626,0.376129,-0.1,-0.611715,0.189727,-0.1,-0.594527,0.048,-0.1,-0.626,0.202,-0.1,-0.595,0.189727,-0.1,-0.594527,0.376129,-0.1,-0.611715,0.390407,-0.1,-0.589222,0.202,-0.1,-0.595,0.376129,-0.1,-0.611715,0.390407,-0.1,-0.589222,0.218377,-0.1,-0.593914,0.202,-0.1,-0.595,0.390407,-0.1,-0.589222,0.233931,-0.1,-0.590727,0.218377,-0.1,-0.593914,0.390407,-0.1,-0.589222,0.248547,-0.1,-0.585547,0.233931,-0.1,-0.590727,0.401797,-0.1,-0.564438,0.248547,-0.1,-0.585547,0.390407,-0.1,-0.589222,0.401797,-0.1,-0.564438,0.262111,-0.1,-0.578481,0.248547,-0.1,-0.585547,0.401797,-0.1,-0.564438,0.274509,-0.1,-0.569638,0.262111,-0.1,-0.578481,0.401797,-0.1,-0.564438,0.285625,-0.1,-0.559125,0.274509,-0.1,-0.569638,0.410134,-0.1,-0.537528,0.285625,-0.1,-0.559125,0.401797,-0.1,-0.564438,0.410134,-0.1,-0.537528,0.295345,-0.1,-0.547049,0.285625,-0.1,-0.559125,0.410134,-0.1,-0.537528,0.303556,-0.1,-0.533518,0.295345,-0.1,-0.547049,0.415256,-0.1,-0.50866,0.303556,-0.1,-0.533518,0.410134,-0.1,-0.537528,0.415256,-0.1,-0.50866,0.310141,-0.1,-0.518641,0.303556,-0.1,-0.533518,0.415256,-0.1,-0.50866,0.314986,-0.1,-0.502523,0.310141,-0.1,-0.518641,0.417,-0.1,-0.478,0.314986,-0.1,-0.502523,0.415256,-0.1,-0.50866,0.417,-0.1,-0.478,0.317977,-0.1,-0.485274,0.314986,-0.1,-0.502523,0.417,-0.1,-0.478,0.319,-0.1,-0.467,0.317977,-0.1,-0.485274,0.415731,-0.1,-0.450564,0.319,-0.1,-0.467,0.417,-0.1,-0.478,0.415731,-0.1,-0.450564,0.318667,-0.1,-0.453204,0.319,-0.1,-0.467,0.415731,-0.1,-0.450564,0.317667,-0.1,-0.440213,0.318667,-0.1,-0.453204,0.411852,-0.1,-0.423509,0.317667,-0.1,-0.440213,0.415731,-0.1,-0.450564,0.411852,-0.1,-0.423509,0.316,-0.1,-0.427875,0.317667,-0.1,-0.440213,0.411852,-0.1,-0.423509,0.313667,-0.1,-0.416037,0.316,-0.1,-0.427875,0.40525,-0.1,-0.396469,0.313667,-0.1,-0.416037,0.411852,-0.1,-0.423509,0.40525,-0.1,-0.396469,0.310667,-0.1,-0.404546,0.313667,-0.1,-0.416037,0.40525,-0.1,-0.396469,0.307,-0.1,-0.39325,0.310667,-0.1,-0.404546,0.395815,-0.1,-0.369074,0.307,-0.1,-0.39325,0.40525,-0.1,-0.396469,0.395815,-0.1,-0.369074,0.302667,-0.1,-0.381995,0.307,-0.1,-0.39325,0.395815,-0.1,-0.369074,0.297667,-0.1,-0.37063,0.302667,-0.1,-0.381995,0.395815,-0.1,-0.369074,0.292,-0.1,-0.359,0.297667,-0.1,-0.37063,0.383435,-0.1,-0.340957,0.292,-0.1,-0.359,0.395815,-0.1,-0.369074,0.383435,-0.1,-0.340957,0.285667,-0.1,-0.346954,0.292,-0.1,-0.359,0.383435,-0.1,-0.340957,0.278667,-0.1,-0.334338,0.285667,-0.1,-0.346954,0.368,-0.1,-0.31175,0.278667,-0.1,-0.334338,0.383435,-0.1,-0.340957,0.368,-0.1,-0.31175,0.271,-0.1,-0.321,0.278667,-0.1,-0.334338,0.368,-0.1,-0.31175,0.255585,-0.1,-0.295763,0.271,-0.1,-0.321,0.349398,-0.1,-0.281085,0.255585,-0.1,-0.295763,0.368,-0.1,-0.31175,0.349398,-0.1,-0.281085,0.238514,-0.1,-0.269685,0.255585,-0.1,-0.295763,0.327519,-0.1,-0.248593,0.238514,-0.1,-0.269685,0.349398,-0.1,-0.281085,0.327519,-0.1,-0.248593,0.220047,-0.1,-0.242969,0.238514,-0.1,-0.269685,0.30225,-0.1,-0.213906,0.220047,-0.1,-0.242969,0.327519,-0.1,-0.248593,0.30225,-0.1,-0.213906,0.200444,-0.1,-0.215815,0.220047,-0.1,-0.242969,0.30225,-0.1,-0.213906,0.179967,-0.1,-0.188425,0.200444,-0.1,-0.215815,0.273481,-0.1,-0.176657,0.179967,-0.1,-0.188425,0.30225,-0.1,-0.213906,0.273481,-0.1,-0.176657,0.158875,-0.1,-0.161,0.179967,-0.1,-0.188425,0.241102,-0.1,-0.136478,0.158875,-0.1,-0.161,0.273481,-0.1,-0.176657,0.241102,-0.1,-0.136478,0.137429,-0.1,-0.133742,0.158875,-0.1,-0.161,0.205,-0.1,-0.093,0.137429,-0.1,-0.133742,0.241102,-0.1,-0.136478,0.205,-0.1,-0.093,0.115889,-0.1,-0.106852,0.137429,-0.1,-0.133742,0.205,-0.1,-0.093,0.094516,-0.1,-0.080531,0.115889,-0.1,-0.106852,0.466,-0.1,-0.093,0.094516,-0.1,-0.080531,0.205,-0.1,-0.093,0.466,-0.1,0,0.094516,-0.1,-0.080531,0.466,-0.1,-0.093,0.466,-0.1,0,0.073569,-0.1,-0.054981,0.094516,-0.1,-0.080531,0.466,-0.1,0,0.053311,-0.1,-0.030404,0.073569,-0.1,-0.054981,0.466,-0.1,0,0.034,-0.1,-0.007,0.053311,-0.1,-0.030404,0.466,-0.1,0,0.034,-0.1,0,0.034,-0.1,-0.007,0.034,0.1,0,0.034,0.1,-0.007,0.034,-0.1,-0.007,0.034,0.1,0,0.034,-0.1,-0.007,0.034,-0.1,0,0.034,0.1,-0.007,0.053311,0.1,-0.030404,0.053311,-0.1,-0.030404,0.034,0.1,-0.007,0.053311,-0.1,-0.030404,0.034,-0.1,-0.007,0.053311,0.1,-0.030404,0.073569,0.1,-0.054981,0.073569,-0.1,-0.054981,0.053311,0.1,-0.030404,0.073569,-0.1,-0.054981,0.053311,-0.1,-0.030404,0.073569,0.1,-0.054981,0.094516,0.1,-0.080531,0.094516,-0.1,-0.080531,0.073569,0.1,-0.054981,0.094516,-0.1,-0.080531,0.073569,-0.1,-0.054981,0.094516,0.1,-0.080531,0.115889,0.1,-0.106852,0.115889,-0.1,-0.106852,0.094516,0.1,-0.080531,0.115889,-0.1,-0.106852,0.094516,-0.1,-0.080531,0.115889,0.1,-0.106852,0.137429,0.1,-0.133742,0.137429,-0.1,-0.133742,0.115889,0.1,-0.106852,0.137429,-0.1,-0.133742,0.115889,-0.1,-0.106852,0.137429,0.1,-0.133742,0.158875,0.1,-0.161,0.158875,-0.1,-0.161,0.137429,0.1,-0.133742,0.158875,-0.1,-0.161,0.137429,-0.1,-0.133742,0.158875,0.1,-0.161,0.179967,0.1,-0.188425,0.179967,-0.1,-0.188425,0.158875,0.1,-0.161,0.179967,-0.1,-0.188425,0.158875,-0.1,-0.161,0.179967,0.1,-0.188425,0.200444,0.1,-0.215815,0.200444,-0.1,-0.215815,0.179967,0.1,-0.188425,0.200444,-0.1,-0.215815,0.179967,-0.1,-0.188425,0.200444,0.1,-0.215815,0.220047,0.1,-0.242969,0.220047,-0.1,-0.242969,0.200444,0.1,-0.215815,0.220047,-0.1,-0.242969,0.200444,-0.1,-0.215815,0.220047,0.1,-0.242969,0.238514,0.1,-0.269685,0.238514,-0.1,-0.269685,0.220047,0.1,-0.242969,0.238514,-0.1,-0.269685,0.220047,-0.1,-0.242969,0.238514,0.1,-0.269685,0.255585,0.1,-0.295763,0.255585,-0.1,-0.295763,0.238514,0.1,-0.269685,0.255585,-0.1,-0.295763,0.238514,-0.1,-0.269685,0.255585,0.1,-0.295763,0.271,0.1,-0.321,0.271,-0.1,-0.321,0.255585,0.1,-0.295763,0.271,-0.1,-0.321,0.255585,-0.1,-0.295763,0.271,0.1,-0.321,0.278667,0.1,-0.334338,0.278667,-0.1,-0.334338,0.271,0.1,-0.321,0.278667,-0.1,-0.334338,0.271,-0.1,-0.321,0.278667,0.1,-0.334338,0.285667,0.1,-0.346954,0.285667,-0.1,-0.346954,0.278667,0.1,-0.334338,0.285667,-0.1,-0.346954,0.278667,-0.1,-0.334338,0.285667,0.1,-0.346954,0.292,0.1,-0.359,0.292,-0.1,-0.359,0.285667,0.1,-0.346954,0.292,-0.1,-0.359,0.285667,-0.1,-0.346954,0.292,0.1,-0.359,0.297667,0.1,-0.37063,0.297667,-0.1,-0.37063,0.292,0.1,-0.359,0.297667,-0.1,-0.37063,0.292,-0.1,-0.359,0.297667,0.1,-0.37063,0.302667,0.1,-0.381995,0.302667,-0.1,-0.381995,0.297667,0.1,-0.37063,0.302667,-0.1,-0.381995,0.297667,-0.1,-0.37063,0.302667,0.1,-0.381995,0.307,0.1,-0.39325,0.307,-0.1,-0.39325,0.302667,0.1,-0.381995,0.307,-0.1,-0.39325,0.302667,-0.1,-0.381995,0.307,0.1,-0.39325,0.310667,0.1,-0.404546,0.310667,-0.1,-0.404546,0.307,0.1,-0.39325,0.310667,-0.1,-0.404546,0.307,-0.1,-0.39325,0.310667,0.1,-0.404546,0.313667,0.1,-0.416037,0.313667,-0.1,-0.416037,0.310667,0.1,-0.404546,0.313667,-0.1,-0.416037,0.310667,-0.1,-0.404546,0.313667,0.1,-0.416037,0.316,0.1,-0.427875,0.316,-0.1,-0.427875,0.313667,0.1,-0.416037,0.316,-0.1,-0.427875,0.313667,-0.1,-0.416037,0.316,0.1,-0.427875,0.317667,0.1,-0.440213,0.317667,-0.1,-0.440213,0.316,0.1,-0.427875,0.317667,-0.1,-0.440213,0.316,-0.1,-0.427875,0.317667,0.1,-0.440213,0.318667,0.1,-0.453204,0.318667,-0.1,-0.453204,0.317667,0.1,-0.440213,0.318667,-0.1,-0.453204,0.317667,-0.1,-0.440213,0.318667,0.1,-0.453204,0.319,0.1,-0.467,0.319,-0.1,-0.467,0.318667,0.1,-0.453204,0.319,-0.1,-0.467,0.318667,-0.1,-0.453204,0.319,0.1,-0.467,0.317977,0.1,-0.485274,0.317977,-0.1,-0.485274,0.319,0.1,-0.467,0.317977,-0.1,-0.485274,0.319,-0.1,-0.467,0.317977,0.1,-0.485274,0.314986,0.1,-0.502523,0.314986,-0.1,-0.502523,0.317977,0.1,-0.485274,0.314986,-0.1,-0.502523,0.317977,-0.1,-0.485274,0.314986,0.1,-0.502523,0.310141,0.1,-0.518641,0.310141,-0.1,-0.518641,0.314986,0.1,-0.502523,0.310141,-0.1,-0.518641,0.314986,-0.1,-0.502523,0.310141,0.1,-0.518641,0.303556,0.1,-0.533518,0.303556,-0.1,-0.533518,0.310141,0.1,-0.518641,0.303556,-0.1,-0.533518,0.310141,-0.1,-0.518641,0.303556,0.1,-0.533518,0.295345,0.1,-0.547049,0.295345,-0.1,-0.547049,0.303556,0.1,-0.533518,0.295345,-0.1,-0.547049,0.303556,-0.1,-0.533518,0.295345,0.1,-0.547049,0.285625,0.1,-0.559125,0.285625,-0.1,-0.559125,0.295345,0.1,-0.547049,0.285625,-0.1,-0.559125,0.295345,-0.1,-0.547049,0.285625,0.1,-0.559125,0.274509,0.1,-0.569638,0.274509,-0.1,-0.569638,0.285625,0.1,-0.559125,0.274509,-0.1,-0.569638,0.285625,-0.1,-0.559125,0.274509,0.1,-0.569638,0.262111,0.1,-0.578481,0.262111,-0.1,-0.578481,0.274509,0.1,-0.569638,0.262111,-0.1,-0.578481,0.274509,-0.1,-0.569638,0.262111,0.1,-0.578481,0.248547,0.1,-0.585547,0.248547,-0.1,-0.585547,0.262111,0.1,-0.578481,0.248547,-0.1,-0.585547,0.262111,-0.1,-0.578481,0.248547,0.1,-0.585547,0.233931,0.1,-0.590727,0.233931,-0.1,-0.590727,0.248547,0.1,-0.585547,0.233931,-0.1,-0.590727,0.248547,-0.1,-0.585547,0.233931,0.1,-0.590727,0.218377,0.1,-0.593914,0.218377,-0.1,-0.593914,0.233931,0.1,-0.590727,0.218377,-0.1,-0.593914,0.233931,-0.1,-0.590727,0.218377,0.1,-0.593914,0.202,0.1,-0.595,0.202,-0.1,-0.595,0.218377,0.1,-0.593914,0.202,-0.1,-0.595,0.218377,-0.1,-0.593914,0.202,0.1,-0.595,0.189727,0.1,-0.594527,0.189727,-0.1,-0.594527,0.202,0.1,-0.595,0.189727,-0.1,-0.594527,0.202,-0.1,-0.595,0.189727,0.1,-0.594527,0.177398,0.1,-0.593046,0.177398,-0.1,-0.593046,0.189727,0.1,-0.594527,0.177398,-0.1,-0.593046,0.189727,-0.1,-0.594527,0.177398,0.1,-0.593046,0.165,0.1,-0.590469,0.165,-0.1,-0.590469,0.177398,0.1,-0.593046,0.165,-0.1,-0.590469,0.177398,-0.1,-0.593046,0.165,0.1,-0.590469,0.152519,0.1,-0.586704,0.152519,-0.1,-0.586704,0.165,0.1,-0.590469,0.152519,-0.1,-0.586704,0.165,-0.1,-0.590469,0.152519,0.1,-0.586704,0.13994,0.1,-0.581661,0.13994,-0.1,-0.581661,0.152519,0.1,-0.586704,0.13994,-0.1,-0.581661,0.152519,-0.1,-0.586704,0.13994,0.1,-0.581661,0.12725,0.1,-0.57525,0.12725,-0.1,-0.57525,0.13994,0.1,-0.581661,0.12725,-0.1,-0.57525,0.13994,-0.1,-0.581661,0.12725,0.1,-0.57525,0.114435,0.1,-0.567381,0.114435,-0.1,-0.567381,0.12725,0.1,-0.57525,0.114435,-0.1,-0.567381,0.12725,-0.1,-0.57525,0.114435,0.1,-0.567381,0.101481,0.1,-0.557963,0.101481,-0.1,-0.557963,0.114435,0.1,-0.567381,0.101481,-0.1,-0.557963,0.114435,-0.1,-0.567381,0.101481,0.1,-0.557963,0.088375,0.1,-0.546906,0.088375,-0.1,-0.546906,0.101481,0.1,-0.557963,0.088375,-0.1,-0.546906,0.101481,-0.1,-0.557963,0.088375,0.1,-0.546906,0.075102,0.1,-0.53412,0.075102,-0.1,-0.53412,0.088375,0.1,-0.546906,0.075102,-0.1,-0.53412,0.088375,-0.1,-0.546906,0.075102,0.1,-0.53412,0.061648,0.1,-0.519515,0.061648,-0.1,-0.519515,0.075102,0.1,-0.53412,0.061648,-0.1,-0.519515,0.075102,-0.1,-0.53412,0.061648,0.1,-0.519515,0.048,0.1,-0.503,0.048,-0.1,-0.503,0.061648,0.1,-0.519515,0.048,-0.1,-0.503,0.061648,-0.1,-0.519515,0.048,0.1,-0.503,0.048,0.1,-0.626,0.048,-0.1,-0.626,0.048,0.1,-0.503,0.048,-0.1,-0.626,0.048,-0.1,-0.503,0.048,0.1,-0.626,0.061671,0.1,-0.636522,0.061671,-0.1,-0.636522,0.048,0.1,-0.626,0.061671,-0.1,-0.636522,0.048,-0.1,-0.626,0.061671,0.1,-0.636522,0.075204,0.1,-0.646093,0.075204,-0.1,-0.646093,0.061671,0.1,-0.636522,0.075204,-0.1,-0.646093,0.061671,-0.1,-0.636522,0.075204,0.1,-0.646093,0.088625,0.1,-0.654719,0.088625,-0.1,-0.654719,0.075204,0.1,-0.646093,0.088625,-0.1,-0.654719,0.075204,-0.1,-0.646093,0.088625,0.1,-0.654719,0.101963,0.1,-0.662407,0.101963,-0.1,-0.662407,0.088625,0.1,-0.654719,0.101963,-0.1,-0.662407,0.088625,-0.1,-0.654719,0.101963,0.1,-0.662407,0.115245,0.1,-0.669166,0.115245,-0.1,-0.669166,0.101963,0.1,-0.662407,0.115245,-0.1,-0.669166,0.101963,-0.1,-0.662407,0.115245,0.1,-0.669166,0.1285,0.1,-0.675,0.1285,-0.1,-0.675,0.115245,0.1,-0.669166,0.1285,-0.1,-0.675,0.115245,-0.1,-0.669166,0.1285,0.1,-0.675,0.141755,0.1,-0.679918,0.141755,-0.1,-0.679918,0.1285,0.1,-0.675,0.141755,-0.1,-0.679918,0.1285,-0.1,-0.675,0.141755,0.1,-0.679918,0.155037,0.1,-0.683926,0.155037,-0.1,-0.683926,0.141755,0.1,-0.679918,0.155037,-0.1,-0.683926,0.141755,-0.1,-0.679918,0.155037,0.1,-0.683926,0.168375,0.1,-0.687031,0.168375,-0.1,-0.687031,0.155037,0.1,-0.683926,0.168375,-0.1,-0.687031,0.155037,-0.1,-0.683926,0.168375,0.1,-0.687031,0.181796,0.1,-0.689241,0.181796,-0.1,-0.689241,0.168375,0.1,-0.687031,0.181796,-0.1,-0.689241,0.168375,-0.1,-0.687031,0.181796,0.1,-0.689241,0.195329,0.1,-0.690561,0.195329,-0.1,-0.690561,0.181796,0.1,-0.689241,0.195329,-0.1,-0.690561,0.181796,-0.1,-0.689241,0.195329,0.1,-0.690561,0.209,0.1,-0.691,0.209,-0.1,-0.691,0.195329,0.1,-0.690561,0.209,-0.1,-0.691,0.195329,-0.1,-0.690561,0.209,0.1,-0.691,0.238931,0.1,-0.689215,0.238931,-0.1,-0.689215,0.209,0.1,-0.691,0.238931,-0.1,-0.689215,0.209,-0.1,-0.691,0.238931,0.1,-0.689215,0.267116,0.1,-0.683972,0.267116,-0.1,-0.683972,0.238931,0.1,-0.689215,0.267116,-0.1,-0.683972,0.238931,-0.1,-0.689215,0.267116,0.1,-0.683972,0.293391,0.1,-0.675438,0.293391,-0.1,-0.675438,0.267116,0.1,-0.683972,0.293391,-0.1,-0.675438,0.267116,-0.1,-0.683972,0.293391,0.1,-0.675438,0.317593,0.1,-0.663778,0.317593,-0.1,-0.663778,0.293391,0.1,-0.675438,0.317593,-0.1,-0.663778,0.293391,-0.1,-0.675438,0.317593,0.1,-0.663778,0.339558,0.1,-0.64916,0.339558,-0.1,-0.64916,0.317593,0.1,-0.663778,0.339558,-0.1,-0.64916,0.317593,-0.1,-0.663778,0.339558,0.1,-0.64916,0.359125,0.1,-0.63175,0.359125,-0.1,-0.63175,0.339558,0.1,-0.64916,0.359125,-0.1,-0.63175,0.339558,-0.1,-0.64916,0.359125,0.1,-0.63175,0.376129,0.1,-0.611715,0.376129,-0.1,-0.611715,0.359125,0.1,-0.63175,0.376129,-0.1,-0.611715,0.359125,-0.1,-0.63175,0.376129,0.1,-0.611715,0.390407,0.1,-0.589222,0.390407,-0.1,-0.589222,0.376129,0.1,-0.611715,0.390407,-0.1,-0.589222,0.376129,-0.1,-0.611715,0.390407,0.1,-0.589222,0.401797,0.1,-0.564438,0.401797,-0.1,-0.564438,0.390407,0.1,-0.589222,0.401797,-0.1,-0.564438,0.390407,-0.1,-0.589222,0.401797,0.1,-0.564438,0.410134,0.1,-0.537528,0.410134,-0.1,-0.537528,0.401797,0.1,-0.564438,0.410134,-0.1,-0.537528,0.401797,-0.1,-0.564438,0.410134,0.1,-0.537528,0.415256,0.1,-0.50866,0.415256,-0.1,-0.50866,0.410134,0.1,-0.537528,0.415256,-0.1,-0.50866,0.410134,-0.1,-0.537528,0.415256,0.1,-0.50866,0.417,0.1,-0.478,0.417,-0.1,-0.478,0.415256,0.1,-0.50866,0.417,-0.1,-0.478,0.415256,-0.1,-0.50866,0.417,0.1,-0.478,0.415731,0.1,-0.450564,0.415731,-0.1,-0.450564,0.417,0.1,-0.478,0.415731,-0.1,-0.450564,0.417,-0.1,-0.478,0.415731,0.1,-0.450564,0.411852,0.1,-0.423509,0.411852,-0.1,-0.423509,0.415731,0.1,-0.450564,0.411852,-0.1,-0.423509,0.415731,-0.1,-0.450564,0.411852,0.1,-0.423509,0.40525,0.1,-0.396469,0.40525,-0.1,-0.396469,0.411852,0.1,-0.423509,0.40525,-0.1,-0.396469,0.411852,-0.1,-0.423509,0.40525,0.1,-0.396469,0.395815,0.1,-0.369074,0.395815,-0.1,-0.369074,0.40525,0.1,-0.396469,0.395815,-0.1,-0.369074,0.40525,-0.1,-0.396469,0.395815,0.1,-0.369074,0.383435,0.1,-0.340957,0.383435,-0.1,-0.340957,0.395815,0.1,-0.369074,0.383435,-0.1,-0.340957,0.395815,-0.1,-0.369074,0.383435,0.1,-0.340957,0.368,0.1,-0.31175,0.368,-0.1,-0.31175,0.383435,0.1,-0.340957,0.368,-0.1,-0.31175,0.383435,-0.1,-0.340957,0.368,0.1,-0.31175,0.349398,0.1,-0.281085,0.349398,-0.1,-0.281085,0.368,0.1,-0.31175,0.349398,-0.1,-0.281085,0.368,-0.1,-0.31175,0.349398,0.1,-0.281085,0.327519,0.1,-0.248593,0.327519,-0.1,-0.248593,0.349398,0.1,-0.281085,0.327519,-0.1,-0.248593,0.349398,-0.1,-0.281085,0.327519,0.1,-0.248593,0.30225,0.1,-0.213906,0.30225,-0.1,-0.213906,0.327519,0.1,-0.248593,0.30225,-0.1,-0.213906,0.327519,-0.1,-0.248593,0.30225,0.1,-0.213906,0.273481,0.1,-0.176657,0.273481,-0.1,-0.176657,0.30225,0.1,-0.213906,0.273481,-0.1,-0.176657,0.30225,-0.1,-0.213906,0.273481,0.1,-0.176657,0.241102,0.1,-0.136478,0.241102,-0.1,-0.136478,0.273481,0.1,-0.176657,0.241102,-0.1,-0.136478,0.273481,-0.1,-0.176657,0.241102,0.1,-0.136478,0.205,0.1,-0.093,0.205,-0.1,-0.093,0.241102,0.1,-0.136478,0.205,-0.1,-0.093,0.241102,-0.1,-0.136478,0.205,0.1,-0.093,0.466,0.1,-0.093,0.466,-0.1,-0.093,0.205,0.1,-0.093,0.466,-0.1,-0.093,0.205,-0.1,-0.093,0.466,0.1,-0.093,0.466,0.1,0,0.466,-0.1,0,0.466,0.1,-0.093,0.466,-0.1,0,0.466,-0.1,-0.093,0.466,0.1,0,0.034,0.1,0,0.034,-0.1,0,0.466,0.1,0,0.034,-0.1,0,0.466,-0.1,0]},"normal":{"numComponents":3,"data":[0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,0,-1,0,-0.7071,0,0.7071,-0.9411,0,-0.3381,-0.9411,0,-0.3381,-0.7071,0,0.7071,-0.9411,0,-0.3381,-0.7071,0,0.7071,-0.9411,0,-0.3381,-0.7715,0,-0.6362,-0.7715,0,-0.6362,-0.9411,0,-0.3381,-0.7715,0,-0.6362,-0.9411,0,-0.3381,-0.7715,0,-0.6362,-0.7725,0,-0.635,-0.7725,0,-0.635,-0.7715,0,-0.6362,-0.7725,0,-0.635,-0.7715,0,-0.6362,-0.7725,0,-0.635,-0.7748,0,-0.6322,-0.7748,0,-0.6322,-0.7725,0,-0.635,-0.7748,0,-0.6322,-0.7725,0,-0.635,-0.7748,0,-0.6322,-0.7784,0,-0.6278,-0.7784,0,-0.6278,-0.7748,0,-0.6322,-0.7784,0,-0.6278,-0.7748,0,-0.6322,-0.7784,0,-0.6278,-0.7832,0,-0.6218,-0.7832,0,-0.6218,-0.7784,0,-0.6278,-0.7832,0,-0.6218,-0.7784,0,-0.6278,-0.7832,0,-0.6218,-0.7893,0,-0.614,-0.7893,0,-0.614,-0.7832,0,-0.6218,-0.7893,0,-0.614,-0.7832,0,-0.6218,-0.7893,0,-0.614,-0.7968,0,-0.6042,-0.7968,0,-0.6042,-0.7893,0,-0.614,-0.7968,0,-0.6042,-0.7893,0,-0.614,-0.7968,0,-0.6042,-0.8059,0,-0.5921,-0.8059,0,-0.5921,-0.7968,0,-0.6042,-0.8059,0,-0.5921,-0.7968,0,-0.6042,-0.8059,0,-0.5921,-0.8167,0,-0.577,-0.8167,0,-0.577,-0.8059,0,-0.5921,-0.8167,0,-0.577,-0.8059,0,-0.5921,-0.8167,0,-0.577,-0.8297,0,-0.5582,-0.8297,0,-0.5582,-0.8167,0,-0.577,-0.8297,0,-0.5582,-0.8167,0,-0.577,-0.8297,0,-0.5582,-0.8451,0,-0.5345,-0.8451,0,-0.5345,-0.8297,0,-0.5582,-0.8451,0,-0.5345,-0.8297,0,-0.5582,-0.8451,0,-0.5345,-0.8603,0,-0.5098,-0.8603,0,-0.5098,-0.8451,0,-0.5345,-0.8603,0,-0.5098,-0.8451,0,-0.5345,-0.8603,0,-0.5098,-0.8707,0,-0.4918,-0.8707,0,-0.4918,-0.8603,0,-0.5098,-0.8707,0,-0.4918,-0.8603,0,-0.5098,-0.8707,0,-0.4918,-0.8798,0,-0.4753,-0.8798,0,-0.4753,-0.8707,0,-0.4918,-0.8798,0,-0.4753,-0.8707,0,-0.4918,-0.8798,0,-0.4753,-0.8921,0,-0.4517,-0.8921,0,-0.4517,-0.8798,0,-0.4753,-0.8921,0,-0.4517,-0.8798,0,-0.4753,-0.8921,0,-0.4517,-0.9073,0,-0.4204,-0.9073,0,-0.4204,-0.8921,0,-0.4517,-0.9073,0,-0.4204,-0.8921,0,-0.4517,-0.9073,0,-0.4204,-0.9245,0,-0.3811,-0.9245,0,-0.3811,-0.9073,0,-0.4204,-0.9245,0,-0.3811,-0.9073,0,-0.4204,-0.9245,0,-0.3811,-0.9425,0,-0.3341,-0.9425,0,-0.3341,-0.9245,0,-0.3811,-0.9425,0,-0.3341,-0.9245,0,-0.3811,-0.9425,0,-0.3341,-0.9597,0,-0.2808,-0.9597,0,-0.2808,-0.9425,0,-0.3341,-0.9597,0,-0.2808,-0.9425,0,-0.3341,-0.9597,0,-0.2808,-0.9748,0,-0.2231,-0.9748,0,-0.2231,-0.9597,0,-0.2808,-0.9748,0,-0.2231,-0.9597,0,-0.2808,-0.9748,0,-0.2231,-0.9865,0,-0.1637,-0.9865,0,-0.1637,-0.9748,0,-0.2231,-0.9865,0,-0.1637,-0.9748,0,-0.2231,-0.9865,0,-0.1637,-0.9944,0,-0.1053,-0.9944,0,-0.1053,-0.9865,0,-0.1637,-0.9944,0,-0.1053,-0.9865,0,-0.1637,-0.9944,0,-0.1053,-0.9987,0,-0.0504,-0.9987,0,-0.0504,-0.9944,0,-0.1053,-0.9987,0,-0.0504,-0.9944,0,-0.1053,-0.9987,0,-0.0504,-0.9998,0,0.0159,-0.9998,0,0.0159,-0.9987,0,-0.0504,-0.9998,0,0.0159,-0.9987,0,-0.0504,-0.9998,0,0.0159,-0.9935,0,0.1135,-0.9935,0,0.1135,-0.9998,0,0.0159,-0.9935,0,0.1135,-0.9998,0,0.0159,-0.9935,0,0.1135,-0.9732,0,0.2298,-0.9732,0,0.2298,-0.9935,0,0.1135,-0.9732,0,0.2298,-0.9935,0,0.1135,-0.9732,0,0.2298,-0.9379,0,0.347,-0.9379,0,0.347,-0.9732,0,0.2298,-0.9379,0,0.347,-0.9732,0,0.2298,-0.9379,0,0.347,-0.8865,0,0.4627,-0.8865,0,0.4627,-0.9379,0,0.347,-0.8865,0,0.4627,-0.9379,0,0.347,-0.8865,0,0.4627,-0.8187,0,0.5741,-0.8187,0,0.5741,-0.8865,0,0.4627,-0.8187,0,0.5741,-0.8865,0,0.4627,-0.8187,0,0.5741,-0.7347,0,0.6783,-0.7347,0,0.6783,-0.8187,0,0.5741,-0.7347,0,0.6783,-0.8187,0,0.5741,-0.7347,0,0.6783,-0.6354,0,0.7721,-0.6354,0,0.7721,-0.7347,0,0.6783,-0.6354,0,0.7721,-0.7347,0,0.6783,-0.6354,0,0.7721,-0.5226,0,0.8526,-0.5226,0,0.8526,-0.6354,0,0.7721,-0.5226,0,0.8526,-0.6354,0,0.7721,-0.5226,0,0.8526,-0.399,0,0.917,-0.399,0,0.917,-0.5226,0,0.8526,-0.399,0,0.917,-0.5226,0,0.8526,-0.399,0,0.917,-0.268,0,0.9634,-0.268,0,0.9634,-0.399,0,0.917,-0.268,0,0.9634,-0.399,0,0.917,-0.268,0,0.9634,-0.1338,0,0.991,-0.1338,0,0.991,-0.268,0,0.9634,-0.1338,0,0.991,-0.268,0,0.9634,-0.1338,0,0.991,-0.0138,0,0.9999,-0.0138,0,0.9999,-0.1338,0,0.991,-0.0138,0,0.9999,-0.1338,0,0.991,-0.0138,0,0.9999,0.0789,0,0.9969,0.0789,0,0.9969,-0.0138,0,0.9999,0.0789,0,0.9969,-0.0138,0,0.9999,0.0789,0,0.9969,0.1615,0,0.9868,0.1615,0,0.9868,0.0789,0,0.9969,0.1615,0,0.9868,0.0789,0,0.9969,0.1615,0,0.9868,0.2464,0,0.9691,0.2464,0,0.9691,0.1615,0,0.9868,0.2464,0,0.9691,0.1615,0,0.9868,0.2464,0,0.9691,0.3308,0,0.9437,0.3308,0,0.9437,0.2464,0,0.9691,0.3308,0,0.9437,0.2464,0,0.9691,0.3308,0,0.9437,0.4119,0,0.9112,0.4119,0,0.9112,0.3308,0,0.9437,0.4119,0,0.9112,0.3308,0,0.9437,0.4119,0,0.9112,0.4875,0,0.8731,0.4875,0,0.8731,0.4119,0,0.9112,0.4875,0,0.8731,0.4119,0,0.9112,0.4875,0,0.8731,0.5561,0,0.8311,0.5561,0,0.8311,0.4875,0,0.8731,0.5561,0,0.8311,0.4875,0,0.8731,0.5561,0,0.8311,0.6168,0,0.7871,0.6168,0,0.7871,0.5561,0,0.8311,0.6168,0,0.7871,0.5561,0,0.8311,0.6168,0,0.7871,0.6696,0,0.7427,0.6696,0,0.7427,0.6168,0,0.7871,0.6696,0,0.7427,0.6168,0,0.7871,0.6696,0,0.7427,0.7149,0,0.6991,0.7149,0,0.6991,0.6696,0,0.7427,0.7149,0,0.6991,0.6696,0,0.7427,0.7149,0,0.6991,0.7534,0,0.6575,0.7534,0,0.6575,0.7149,0,0.6991,0.7534,0,0.6575,0.7149,0,0.6991,0.7534,0,0.6575,-0.3385,0,0.9409,-0.3385,0,0.9409,0.7534,0,0.6575,-0.3385,0,0.9409,0.7534,0,0.6575,-0.3385,0,0.9409,-0.8972,0,-0.4416,-0.8972,0,-0.4416,-0.3385,0,0.9409,-0.8972,0,-0.4416,-0.3385,0,0.9409,-0.8972,0,-0.4416,-0.5938,0,-0.8046,-0.5938,0,-0.8046,-0.8972,0,-0.4416,-0.5938,0,-0.8046,-0.8972,0,-0.4416,-0.5938,0,-0.8046,-0.5592,0,-0.829,-0.5592,0,-0.829,-0.5938,0,-0.8046,-0.5592,0,-0.829,-0.5938,0,-0.8046,-0.5592,0,-0.829,-0.5202,0,-0.854,-0.5202,0,-0.854,-0.5592,0,-0.829,-0.5202,0,-0.854,-0.5592,0,-0.829,-0.5202,0,-0.854,-0.4766,0,-0.8791,-0.4766,0,-0.8791,-0.5202,0,-0.854,-0.4766,0,-0.8791,-0.5202,0,-0.854,-0.4766,0,-0.8791,-0.4283,0,-0.9036,-0.4283,0,-0.9036,-0.4766,0,-0.8791,-0.4283,0,-0.9036,-0.4766,0,-0.8791,-0.4283,0,-0.9036,-0.3755,0,-0.9268,-0.3755,0,-0.9268,-0.4283,0,-0.9036,-0.3755,0,-0.9268,-0.4283,0,-0.9036,-0.3755,0,-0.9268,-0.3185,0,-0.9479,-0.3185,0,-0.9479,-0.3755,0,-0.9268,-0.3185,0,-0.9479,-0.3755,0,-0.9268,-0.3185,0,-0.9479,-0.2579,0,-0.9662,-0.2579,0,-0.9662,-0.3185,0,-0.9479,-0.2579,0,-0.9662,-0.3185,0,-0.9479,-0.2579,0,-0.9662,-0.1947,0,-0.9808,-0.1947,0,-0.9808,-0.2579,0,-0.9662,-0.1947,0,-0.9808,-0.2579,0,-0.9662,-0.1947,0,-0.9808,-0.1298,0,-0.9915,-0.1298,0,-0.9915,-0.1947,0,-0.9808,-0.1298,0,-0.9915,-0.1947,0,-0.9808,-0.1298,0,-0.9915,-0.0646,0,-0.9979,-0.0646,0,-0.9979,-0.1298,0,-0.9915,-0.0646,0,-0.9979,-0.1298,0,-0.9915,-0.0646,0,-0.9979,0.0137,0,-0.9999,0.0137,0,-0.9999,-0.0646,0,-0.9979,0.0137,0,-0.9999,-0.0646,0,-0.9979,0.0137,0,-0.9999,0.1214,0,-0.9926,0.1214,0,-0.9926,0.0137,0,-0.9999,0.1214,0,-0.9926,0.0137,0,-0.9999,0.1214,0,-0.9926,0.2464,0,-0.9691,0.2464,0,-0.9691,0.1214,0,-0.9926,0.2464,0,-0.9691,0.1214,0,-0.9926,0.2464,0,-0.9691,0.3723,0,-0.9281,0.3723,0,-0.9281,0.2464,0,-0.9691,0.3723,0,-0.9281,0.2464,0,-0.9691,0.3723,0,-0.9281,0.4952,0,-0.8688,0.4952,0,-0.8688,0.3723,0,-0.9281,0.4952,0,-0.8688,0.3723,0,-0.9281,0.4952,0,-0.8688,0.6109,0,-0.7917,0.6109,0,-0.7917,0.4952,0,-0.8688,0.6109,0,-0.7917,0.4952,0,-0.8688,0.6109,0,-0.7917,0.7153,0,-0.6988,0.7153,0,-0.6988,0.6109,0,-0.7917,0.7153,0,-0.6988,0.6109,0,-0.7917,0.7153,0,-0.6988,0.8052,0,-0.5929,0.8052,0,-0.5929,0.7153,0,-0.6988,0.8052,0,-0.5929,0.7153,0,-0.6988,0.8052,0,-0.5929,0.8784,0,-0.4778,0.8784,0,-0.4778,0.8052,0,-0.5929,0.8784,0,-0.4778,0.8052,0,-0.5929,0.8784,0,-0.4778,0.9339,0,-0.3575,0.9339,0,-0.3575,0.8784,0,-0.4778,0.9339,0,-0.3575,0.8784,0,-0.4778,0.9339,0,-0.3575,0.9718,0,-0.2358,0.9718,0,-0.2358,0.9339,0,-0.3575,0.9718,0,-0.2358,0.9339,0,-0.3575,0.9718,0,-0.2358,0.9933,0,-0.1159,0.9933,0,-0.1159,0.9718,0,-0.2358,0.9933,0,-0.1159,0.9718,0,-0.2358,0.9933,0,-0.1159,1,0,-0.0053,1,0,-0.0053,0.9933,0,-0.1159,1,0,-0.0053,0.9933,0,-0.1159,1,0,-0.0053,0.9955,0,0.0941,0.9955,0,0.0941,1,0,-0.0053,0.9955,0,0.0941,1,0,-0.0053,0.9955,0,0.0941,0.9818,0,0.1898,0.9818,0,0.1898,0.9955,0,0.0941,0.9818,0,0.1898,0.9955,0,0.0941,0.9818,0,0.1898,0.9595,0,0.2817,0.9595,0,0.2817,0.9818,0,0.1898,0.9595,0,0.2817,0.9818,0,0.1898,0.9595,0,0.2817,0.9312,0,0.3646,0.9312,0,0.3646,0.9595,0,0.2817,0.9312,0,0.3646,0.9595,0,0.2817,0.9312,0,0.3646,0.9002,0,0.4354,0.9002,0,0.4354,0.9312,0,0.3646,0.9002,0,0.4354,0.9312,0,0.3646,0.9002,0,0.4354,0.8699,0,0.4931,0.8699,0,0.4931,0.9002,0,0.4354,0.8699,0,0.4931,0.9002,0,0.4354,0.8699,0,0.4931,0.8425,0,0.5387,0.8425,0,0.5387,0.8699,0,0.4931,0.8425,0,0.5387,0.8699,0,0.4931,0.8425,0,0.5387,0.819,0,0.5738,0.819,0,0.5738,0.8425,0,0.5387,0.819,0,0.5738,0.8425,0,0.5387,0.819,0,0.5738,0.7999,0,0.6001,0.7999,0,0.6001,0.819,0,0.5738,0.7999,0,0.6001,0.819,0,0.5738,0.7999,0,0.6001,0.7851,0,0.6194,0.7851,0,0.6194,0.7999,0,0.6001,0.7851,0,0.6194,0.7999,0,0.6001,0.7851,0,0.6194,0.774,0,0.6332,0.774,0,0.6332,0.7851,0,0.6194,0.774,0,0.6332,0.7851,0,0.6194,0.774,0,0.6332,0.9052,0,-0.4249,0.9052,0,-0.4249,0.774,0,0.6332,0.9052,0,-0.4249,0.774,0,0.6332,0.9052,0,-0.4249,0.7071,0,-0.7071,0.7071,0,-0.7071,0.9052,0,-0.4249,0.7071,0,-0.7071,0.9052,0,-0.4249,0.7071,0,-0.7071,0.7071,0,0.7071,0.7071,0,0.7071,0.7071,0,-0.7071,0.7071,0,0.7071,0.7071,0,-0.7071,0.7071,0,0.7071,-0.7071,0,0.7071,-0.7071,0,0.7071,0.7071,0,0.7071,-0.7071,0,0.7071,0.7071,0,0.7071]}};

function main() {
  // Get A WebGL context

  // Here we do this one 1 of 2 ways like many WebGL libraries. Either
  // we have a canvas on the page. Or else we have container and we
  // insert a canvas inside that container.
  // If we don't find a container we use the body of the document.
  var container = document.querySelector("#canvas") || document.body;
  var isCanvas = (container instanceof HTMLCanvasElement);
  var canvas = isCanvas ? container : document.createElement("canvas");
  if (!isCanvas) {
    container.appendChild(canvas);
  }

  var gl = canvas.getContext("webgl2", { alpha: false });
  if (!gl) {
    return;
  }


  // setup GLSL program
  var programInfo = twgl.createProgramInfo(gl, [vertexShaderSource, fragmentShaderSource]);

  twgl.setAttributePrefix("a_");  // Tell the webglUtils to match position with a_position etc..
  var bufferInfo = twgl.createBufferInfoFromArrays(gl, model);
  var vao = gl.createVertexArray();
  gl.bindVertexArray(vao);
  twgl.setBuffersAndAttributes(gl, programInfo, bufferInfo);


  function degToRad(d) {
    return d * Math.PI / 180;
  }

  var cameraAngleRadians = degToRad(0);
  var fieldOfViewRadians = degToRad(40);
  var cameraHeight = 40;
  var zNear = 1;
  var zFar  = 500;

  var uniformsThatAreTheSameForAllObjects = {
    u_lightWorldPos:         [5000, 3000, 10000],
    u_viewInverse:           m4.identity(),
    u_lightColor:            [1, 1, 1, 1],
  };

  var uniformsThatAreComputedForEachObject = {
    u_worldViewProjection:   m4.identity(),
    u_world:                 m4.identity(),
    u_worldInverseTranspose: m4.identity(),
  };

  var materialUniforms = {
    u_ambient:               [.75, 0, 0, 0],
    u_diffuse:               [1, 1, 1, 1],
    u_specular:              [1, 1, 1, 1],
    u_shininess:             100,
    u_specularFactor:        1,
  };

  var randomSeed = 0;
  var RANDOM_RANGE = Math.pow(2, 32);

  function pseudoRandom() {
    return (randomSeed =
            (134775813 * randomSeed + 1) %
            RANDOM_RANGE) / RANDOM_RANGE;
  }

  function r(min, max) {
    if (max === undefined) {
      max = min;
      min = 0;
    }
    return pseudoRandom() * (max - min) + min;
  }

  function resetPseudoRandom() {
    randomSeed = 0;
  }

  requestAnimationFrame(drawScene);

  // Draw the scene.
  function drawScene(time) {
    twgl.resizeCanvasToDisplaySize(canvas);

    time *= 0.001;  // convert to seconds

    // Set the viewport to match the canvas
    gl.viewport(0, 0, canvas.width, canvas.height);

    // Clear the canvas AND the depth buffer.
    gl.clearColor(71/255 * .8, 255/255 * .8, 176/255 * .8, 1);
    gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

    gl.enable(gl.CULL_FACE);
    gl.enable(gl.DEPTH_TEST);

    // Compute the projection matrix
    var aspect = canvas.clientWidth / canvas.clientHeight;
    var projectionMatrix = m4.perspective(fieldOfViewRadians, aspect, zNear, zFar);

    // Compute the camera's matrix using look at.
    var orbitRadius = 100;
    var orbitTime = time * -0.05;
    var cameraPosition = [Math.cos(orbitTime) * orbitRadius, Math.sin(orbitTime * 1.123) * orbitRadius, Math.sin(orbitTime) * orbitRadius];
    var target = [0, 0, 0];
    var up = [0, 1, 0];
    var cameraMatrix = m4.lookAt(cameraPosition, target, up, uniformsThatAreTheSameForAllObjects.u_viewInverse);

    // Make a view matrix from the camera matrix.
    var viewMatrix = m4.inverse(cameraMatrix);

    gl.useProgram(programInfo.program);

    resetPseudoRandom();

    // Setup all the needed attributes.
    gl.bindVertexArray(vao);

    // Set the uniforms that are the same for all objects.
    twgl.setUniforms(programInfo, uniformsThatAreTheSameForAllObjects);

    // Draw objects
    var num = 4;
    var spread = 20;
    for (var zz = -num; zz <= num; ++zz) {
      for (var yy = -num; yy <= num; ++yy) {
        for (var xx = -num; xx <= num; ++xx) {
          var matrix = m4.identity(uniformsThatAreComputedForEachObject.u_world);
          matrix = m4.translate(matrix, xx * spread, yy * spread, zz * spread);
          matrix = m4.axisRotate(matrix, m4.normalize([r(-1, 1), r(-1, 1), r(-1, 1)]), time * .2 + r(Math.PI * 2));
          matrix = m4.scale(matrix, 20, 20, 20, uniformsThatAreComputedForEachObject.u_world);
          var worldMatrix = matrix;

          // Multiply the matrices.
          var matrix = m4.multiply(viewMatrix, worldMatrix);
          m4.multiply(projectionMatrix, matrix, uniformsThatAreComputedForEachObject.u_worldViewProjection);
          m4.transpose(m4.inverse(worldMatrix), uniformsThatAreComputedForEachObject.u_worldInverseTranspose);

          // Set the uniforms we just computed
          twgl.setUniforms(programInfo, uniformsThatAreComputedForEachObject);

          // Set a color for this object.
//          materialUniforms.u_diffuse[0] = 1;
//          materialUniforms.u_diffuse[1] = 1;
//          materialUniforms.u_diffuse[2] = 1;

          // Set the uniforms that are specific to the this object.
          twgl.setUniforms(programInfo, materialUniforms);

          // Draw the geometry.
          gl.drawArrays(gl.TRIANGLES, 0, bufferInfo.numElements);
        }
      }
    }

    requestAnimationFrame(drawScene);
  }
}

// Check if we're running in jQuery
if (window.$) {
  window.$(function(){
    main();
  });
} else {
  window.addEventListener('load', main);
}



