% ----------------------------------------------------------------------
% csMIX.m
% ---------------------------------------------------------------------- 
% Gaussian Mixture Robust Point Matching.
% -- symmetric modifications.
% Only for TPS.
%
% Usage: 
%
% [c,d] = cMIX (x, y, cn, frac, Tinit, Tfinalfac);
%
% extra: [disp_flag], [m_method], [lam1, lam2], [perT_maxit],
%        [anneal_rate].
% 
% 1. m_method: 'mixture', 'mix-rpm', 'rpm', ...
%        'icp' -- 'icp0', 'icp3', 'icp5'. --> to set k_sigma.
% 2. lam1: positive -- anneal it; negative -- take abs value, no anealing.
%
% 07/21/00

function [o1,o2,o3,o4,o5,o6,o7,o8] =  cMIX (in1,in2,in3,in4,in5,in6,in7,...
    in8,in9,in10,in11,in12);

%  o1 = cx_tps;
%  o2 = dx_tps;
%  o3 = cy_tps;
%  o4 = dy_tps;
%  o5 = mx;
%  o6 = my;
%  o7 = v;
%  o8 = u;


% Default Init control parameters:
% --------------------------------
perT_maxit  = 2;
relax_maxit = 1;
anneal_rate = 0.93;

lamda1_init = 1;
lamda2_init = 0.01;

disp_flag   = 1;
m_method    = 'mixture';

debug_flag  = 0;
trans_type  = 'tps';  
sigma       = 1;

% Check input:
% ------------

% Default Input:  
% [c,d] = cMIX (x, y, frac, Tinit, Tfinalfac).
x          = in1;
y          = in2;
cn         = in3; 
frac       = in4;
T_init     = in5;
T_finalfac = in6;

z            = x;

na = 6;
if nargin >= na+1; disp_flag   = in7;  end;
if nargin >= na+2; m_method    = in8;  end;
if nargin >= na+3; lamda1_init = in9;  end;
if nargin >= na+4; lamda2_init = in10;  end;
if nargin >= na+5; perT_maxit  = in11; end;
if nargin >= na+6; anneal_rate = in12; end;


% take care of 'icp' k_sigma stuff.
if (strcmp(m_method(1:3), 'icp'))
  if length(m_method) == 3
    k_sigma  = 0;
    m_method = 'icp';
  else
    k_sigma  = str2num(m_method(4));
    m_method = 'icp';
  end;
else
  k_sigma = 0;
end;


% keyboard

% Init:
% -----

% init x,y,z:
[xmax, dim] = size(x); x = x (1:frac:xmax, :); [xmax, dim] = size(x); 
[ymax, tmp] = size(y); y = y (1:frac:ymax, :); [ymax, tmp] = size(y);  
[zmax, tmp] = size(z); 
if strcmp(trans_type, 'tps')
  z = x;
end;

% init m:
mx = ones (cn, xmax) ./ (cn*xmax) + randn(cn,xmax) * (1/xmax) * 0.001;
my = ones (cn, ymax) ./ (cn*ymax) + randn(cn,ymax) * (1/ymax) * 0.001;  
T0              = max(x(:,1))^2;
moutlier        = 1/sqrt(T0)*exp(-1)*0;
mx_outliers_row = ones (1,xmax) * moutlier; 
my_outliers_row = ones (1,ymax) * moutlier; 
mx_outliers_col = ones (cn,1)   * 0;
my_outliers_col = ones (cn,1)   * 0;

% init transformation parameters:
theta  = 0; 
t      = zeros (2,1); 
s      = 1;
cx_tps = zeros (xmax,dim+1); 
dx_tps = eye   (dim+1, dim+1); 
cy_tps = zeros (xmax,dim+1); 
dy_tps = eye   (dim+1, dim+1); 
w      = zeros (xmax+dim+1, dim);


% -------------------------------------------------------------------
% Annealing procedure:
% -------------------------------------------------------------------

T       = T_init;
T_final = T_init / T_finalfac;

v  = ones(cn,1)*mean(x);
u  = ones(cn,1)*mean(y);
vf = v;
uf = u;

it_total = 1;
flag_stop = 0;
while (flag_stop ~= 1)
  
  for i=1:perT_maxit     % repeat at each termperature.

    lamda1 = lamda1_init*length(x)*T; 
    lamda2 = lamda2_init*length(x)*T;

    % SIDE X UPDATE:
    % --------------
    % Given vf, x, update mx:
    if debug_flag; disp ('calc m ...'); end;
    mx = cMIX_calc_m (v, x, T, m_method, mx_outliers_row, mx_outliers_col,it_total,k_sigma);

    % Given mx, uf, update v:
    v = (mx * x ./((sum(mx'))' * ones(1,dim)) + uf)/2;
    
    % Given v, update transformation cx,dx:    
    if debug_flag; disp ('calc c,d ...'); end;
    [cx_tps, dx_tps, w] = cMIX_calc_transformation (trans_type,lamda1, lamda2, sigma, v, u,z);

    % w(1:length(z),:) = w(1:length(z),:)*0
    % d_tps = d_tps * 0 + eye(dim+1,dim+1); 
    % c_tps = c_tps *0;

    % Transform v --> vf: 
    if debug_flag; disp ('calc new x ...'); end;
    [vf] = cMIX_warp_pts (trans_type, v, v, cx_tps, dx_tps, w, sigma);

    % SIDE Y UPDATE:
    % --------------
    % Given uf, y, update my:
    if debug_flag; disp ('calc m ...'); end;
    my = cMIX_calc_m (u, y, T, m_method, my_outliers_row, my_outliers_col,it_total,k_sigma);

    % Given my, vf, update u:
    u = (my * y ./((sum(my'))' * ones(1,dim)) + vf)/2;
    
    % Given v, update transformation cx,dx:    
    if debug_flag; disp ('calc c,d ...'); end;
    [cy_tps, dy_tps, w] = cMIX_calc_transformation (trans_type,lamda1, lamda2, sigma, u, v,z);

    % w(1:length(z),:) = w(1:length(z),:)*0
    % d_tps = d_tps * 0 + eye(dim+1,dim+1); 
    % c_tps = c_tps *0;

    % Transform v --> vf: 
    if debug_flag; disp ('calc new x ...'); end;
    [uf] = cMIX_warp_pts (trans_type, u, u, cy_tps, dy_tps, w, sigma);
    
  end  % end of iteration/perT


  
  T = T * anneal_rate;
  
  % Determine if it's time to stop:
  % -------------------------------
  %  when T <= cluster variance, stop.

  if T < T_final; flag_stop = 1; end;
  
  str = sprintf ('%s, T = %.4f:\t lamda1: %.4f lamda2: %.4f', ...
      m_method,T, lamda1, lamda2);
  disp (str);

  % Display:
  % -------- 
  it_total = it_total + 1;
  if (disp_flag == 1) | (flag_stop == 1) | ...
	(disp_flag ==2 & mod(it_total,10)==0) | ...
	(it_total == 1)
    m_threshold = 1.5 * (1/min(length(x),length(y)));
    figure(1); clf;
    csMIX_plot_simple (2, x, y, z, v, vf, u, uf, mx, my, m_threshold, T, ...
	trans_type, cx_tps, dx_tps, cy_tps, dy_tps, ...
	w, sigma, m_method);  
  end;

  % cpause;
  
end % end of annealing.



% return outputs:
% ---------------
if strcmp (trans_type, 'tps')
  o1 = cx_tps;
  o2 = dx_tps;
  o3 = cy_tps;
  o4 = dy_tps;
  o5 = mx;
  o6 = my;
  o7 = v;
  o8 = u;
  
elseif strcmp (trans_type, 'rbf')
  o1 = w;
  o2 = [];
  o3 = [];
elseif strcmp (trans_type, 'r+t')
  o1 = theta;
  o2 = tx;
  o3 = ty;
end;





%%%%%
% 1 % %%% cMIX_calc_m %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%
%
% Update m (correspondence).
%
% Usage:
% [m] = cMIX_calc_m (vx, y, T, 'icp');
% [m] = cMIX_calc_m (vx, y, T, 'mixture');
% [m] = cMIX_calc_m (vx, y, T, 'rpm');
%
% Notes: for "icp", set k_sigma = 0 -- no outlier.
%
% 01/31/00

function [m, m_outliers_row, m_outliers_col] = cMIX_calc_m ...
    (vx, y, T, m_method, m_outliers_row, m_outliers_col, it_total, icp_sigma);

[xmax,dim] = size(vx);
[ymax,dim] = size(y);


% ------------------------------------------------------------------ ICP ---
if strcmp (m_method, 'icp')

  k_sigma = icp_sigma;
  [m, dist_threshold] = cMIX_calc_m_ICP (vx, y, k_sigma);
  m = m + randn(xmax, ymax) * (1/xmax) * 0.001; 
  

% ------------------------------------------------------ one way mixture ---  
elseif strcmp (m_method, 'mixture')
	
  % Given v=tranformed(x), update m:
  y_tmp = zeros (xmax, ymax);
  for it_dim=1:dim
    y_tmp = y_tmp + (vx(:,it_dim) * ones(1,ymax) - ones(xmax,1) * y(:,it_dim)').^2;
  end;  
  
  m_tmp = 1/sqrt(T) .* exp (-y_tmp/T); 
  
  if min(max(m_tmp)) < 1e-10; disp ('### Alert, m close to zero!'); end;
  m = m_tmp;
  
  % normalize accross the outliers as well:
  sy = sum (m) + m_outliers_row;
  m  = m ./ (ones(xmax,1) * sy); 
  m  = m + randn(xmax, ymax) * (1/xmax) * 0.001; 

  % sx = sum(m')' + m_outliers_col;
  % m2 = m ./ (sx * ones(1,ymax));
  % m = (m+m2)/2;

  
% -------------------------------------------------------- mixture - RPM ---
elseif strcmp (m_method, 'mix-rpm')
	
  % Given v=tranformed(x), update m:
  y_tmp = zeros (xmax, ymax);
  for it_dim=1:dim
    y_tmp = y_tmp + (vx(:,it_dim) * ones(1,ymax) - ones(xmax,1) * y(:,it_dim)').^2;
  end;  
  
  m_tmp = 1/sqrt(T) .* exp (-y_tmp/T); 
  %if min(max(m_tmp)) < 1e-10; disp ('### Alert, m close to zero!'); end;
    
  m = m_tmp;

  [m, junk1, junk2] = cMIX_normalize_m (m_tmp, m_outliers_col, m_outliers_row);
  m = m + randn(xmax, ymax) * (1/xmax) * 0.01; 

  % normalize accross the outliers as well:
  %sy         = sum (m) + m_outliers_row;
  %m          = m ./ (ones(xmax,1) * sy); 

  %sx = sum(m')' + m_outliers_col;
  %m2 = m ./ (sx * ones(1,ymax));
  %m = (m+m2)/2;

  
% --------------------------------------------- RPM, double normalization ---
elseif strcmp (m_method, 'rpm')
  % Given v=tranformed(x), update m:
  y_tmp = zeros (xmax, ymax);
  for it_dim=1:dim
    y_tmp = y_tmp + (vx(:,it_dim) * ones(1,ymax) - ones(xmax,1) * y(:,it_dim)').^2;
  end;  
  
  m_tmp = exp (-y_tmp/T); 
  m_tmp = m_tmp + randn(xmax, ymax) * (1/xmax) * 0.001; 
  
  % double normalization, but keep outlier entries constant.
  moutlier       = 1/xmax * 0.1;
  m_outliers_row = ones (1,ymax) * moutlier; 
  m_outliers_col = ones (xmax,1) * moutlier; 
  
  [m, junk1, junk2] = cMIX_normalize_m (m_tmp, m_outliers_col, m_outliers_row);
  
  
% --------------------------------------------- RPM, double normalization ---
elseif strcmp (m_method, 'rpm-old')
  % Given v=tranformed(x), update m:
  y_tmp = zeros (xmax, ymax);
  for it_dim=1:dim
    y_tmp = y_tmp + (vx(:,it_dim) * ones(1,ymax) - ones(xmax,1) * y(:,it_dim)').^2;
  end;  
  
  m_tmp = exp (-y_tmp/T); 
  m_tmp = m_tmp + randn(xmax, ymax) * (1/xmax) * 0.001; 
  
  % double normalization, also update outlier entries.
  if (it_total == 1)
    moutlier       = 1/xmax * 0.1;
    m_outliers_row = ones (1,ymax) * moutlier; 
    m_outliers_col = ones (xmax,1) * moutlier; 
  end;
  [m, m_outliers_row, m_outliers_col] = cMIX_normalize_m (m_tmp, m_outliers_col, m_outliers_row);
  
else	
  disp ('# ERROR #: cMIX_calc_m -- wrong input!');
end
    





