/* Pcp2Nurb:
 * convert submesh of a planar cut polyhedron (pcp) into a nurbs patch 
 *
 * A pcp is a polyhedron with every interior vertex surrounded
 * by four facets. The first and third facet are
 * four-sided, the other two must be planar affine $n$-gons if they 
 * have more than four edges.
 *
 * generated by jorg@cs.purdue.edu 
 * all commercial rights reserved!
 *
 * Input :  Ci -- 9-point subnet of a pcp
 *	    valence -- number of patches joining at the vertex
 * Output:  ctl_pt -- 36 B-spline control points
 *	    knots -- knot sequence 
 *
 *	cf the sample driver routines in nurb_iv{1,2}.c
 *
 * Example pcp obtained by chopping a cube corner;
 *         the subnet is C0,C1,...,C8 
 *
 *
 *                     +---------------------------C7
 *                    /                           /  \
 *                  /            C8             /      \
 *                /                           /        C6
 *      	C1--------------------------C0       /  |
 *              /                           /\     /    |
 *             /                           /C4 \ /      |
 *            C2--------------------------C3----C5      |
 *            |                           |     |       |
 *            |                           |     |       |
 *            |                           |     |      /
 *            |                           |     |    /
 *            |                           |     |  /
 *            |                           |     |/
 *            +---------------------------+ - -       
 *
 *      HINT: more information can be found in the ACM TOMS paper
 *
 *	     Pcp2Nurb: smooth free-form surfacing
 *	     with linearly-trimmed bicubic B-splines
 *				by Jorg Peters
 *           and a preprint on the authors homepage
 *           (the example appears as Figure 5)
 *      HINT: there are >= 2 driver routines this subroutine
 *      HINT: while the orientation of the input subnet does
 *		not matter to the routine inside/outside coloring
 *		of your display may be affected
 */

#include <math.h>
#define DIM 3

void Pcp2Nurb(float Ci[][DIM], int valence[],
    float ctl_pt[][DIM], float knots[])
{
    float    t0,c[4];
    int m;
    double     mypi = 4*atan(1.0);
    float    myknots[10] = { -3,-3,-3,-3,0,0,3,3,3,3 };

    c[1] = cos(2*mypi/valence[0]);
    c[3] = cos(2*mypi/valence[1]);

    // previous bug, c[0], c[2] not initialized. corrected May 1st. 2000
    c[0] = c[2] =0;

    for (m=0; m<10;m++) knots[m] = myknots[m];
    for (m=0; m<DIM; m++) {
      t0 = -7.0/24.0*Ci[0][m]-Ci[1][m]/8+Ci[2][m]/6+5.0/8.0*Ci[3][m]+Ci[6][m]
/24+Ci[8][m]/6+2.0/3.0*Ci[4][m]-Ci[5][m]/4+5.0/96.0*c[3]*Ci[2][m]+c[1]*Ci[0][m]
/12+13.0/96.0*c[1]*Ci[3][m]-13.0/96.0*c[1]*Ci[1][m]-13.0/96.0*c[1]*Ci[2][m]-5.0
/96.0*c[1]*Ci[5][m]+5.0/96.0*c[1]*Ci[6][m]+5.0/96.0*c[1]*Ci[7][m]-c[3]*Ci[0][m]
/24+c[3]*Ci[7][m]/96-c[3]*Ci[5][m]/96-c[3]*Ci[6][m]/96-5.0/96.0*c[3]*Ci[1][m]+
5.0/96.0*c[3]*Ci[3][m];
	
	ctl_pt[   0   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/24-Ci[1][m]/24+5.0/24.0*Ci[2][m]+5.0/8.0*Ci[3][m]+Ci[8][m]
/12+Ci[4][m]/12-c[1]*Ci[2][m]/16+c[1]*Ci[0][m]/16+c[1]*Ci[3][m]/16-c[1]*Ci[1]
[m]/16+c[3]*Ci[3][m]/48-c[3]*Ci[0][m]/48-c[3]*Ci[1][m]/48+c[3]*Ci[2][m]/48;
	
	ctl_pt[   1   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/4+Ci[1][m]/12+Ci[2][m]/4+5.0/12.0*Ci[3][m];
	
	ctl_pt[   2   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/4+25.0/60.0*Ci[1][m]+Ci[2][m]/4+Ci[3][m]/12;
	
	ctl_pt[   3   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/24+5.0/8.0*Ci[1][m]+5.0/24.0*Ci[2][m]-Ci[3][m]/24+Ci[8][m]
/12+Ci[4][m]/12-c[1]*Ci[0][m]/48-c[1]*Ci[3][m]/48+c[1]*Ci[1][m]/48+c[1]*Ci[2]
[m]/48+c[3]*Ci[0][m]/16+c[3]*Ci[1][m]/16-c[3]*Ci[2][m]/16-c[3]*Ci[3][m]/16;
	
	ctl_pt[   4   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -7.0/24.0*Ci[0][m]+5.0/8.0*Ci[1][m]+Ci[2][m]/6-Ci[3][m]/8+Ci[6][m]
/24-Ci[7][m]/4+2.0/3.0*Ci[8][m]+Ci[4][m]/6-c[1]*Ci[0][m]/24-5.0/96.0*c[1]*Ci[3]
[m]+5.0/96.0*c[1]*Ci[1][m]+5.0/96.0*c[1]*Ci[2][m]+c[1]*Ci[5][m]/96-c[1]*Ci[6]
[m]/96-c[1]*Ci[7][m]/96+c[3]*Ci[0][m]/12-5.0/96.0*c[3]*Ci[7][m]+5.0/96.0*c[3]*
Ci[5][m]+5.0/96.0*c[3]*Ci[6][m]+13.0/96.0*c[3]*Ci[1][m]-13.0/96.0*c[3]*Ci[2][m]
-13.0/96.0*c[3]*Ci[3][m];
	
	ctl_pt[   5   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -Ci[0][m]/8+Ci[1][m]/48+Ci[2][m]/48+7.0/16.0*Ci[3][m]+Ci[6][m]/48+Ci
[7][m]/48+5.0/6.0*Ci[4][m]-11.0/48.0*Ci[5][m]-c[1]*Ci[2][m]/16+c[1]*Ci[0][m]/24
+c[1]*Ci[3][m]/16-c[1]*Ci[1][m]/16-c[1]*Ci[5][m]/48+c[1]*Ci[6][m]/48+c[1]*Ci[7]
[m]/48;
	
	ctl_pt[   6   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 5.0/24.0*Ci[0][m]+Ci[1][m]/16+7.0/144.0*Ci[2][m]+73.0/144.0*Ci[3][m]
-Ci[6][m]/144+Ci[7][m]/144-Ci[8][m]/36+7.0/36.0*Ci[4][m]+Ci[5][m]/144-c[3]*Ci
[2][m]/96+5.0/144.0*c[1]*Ci[0][m]+7.0/288.0*c[1]*Ci[3][m]-7.0/288.0*c[1]*Ci[1]
[m]-7.0/288.0*c[1]*Ci[2][m]+c[1]*Ci[5][m]/96-c[1]*Ci[6][m]/96-c[1]*Ci[7][m]/96+
c[3]*Ci[0][m]/144-c[3]*Ci[7][m]/288+c[3]*Ci[5][m]/288+c[3]*Ci[6][m]/288+c[3]*Ci
[1][m]/96-c[3]*Ci[3][m]/96;
	
	ctl_pt[   7   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 5.0/12.0*Ci[0][m]+5.0/36.0*Ci[1][m]+Ci[2][m]/12+13.0/36.0*Ci[3][m]-
Ci[8][m]/18+Ci[4][m]/18+c[1]*Ci[2][m]/72-c[1]*Ci[0][m]/72-c[1]*Ci[3][m]/72+c[1]
*Ci[1][m]/72-c[3]*Ci[3][m]/72+c[3]*Ci[0][m]/72+c[3]*Ci[1][m]/72-c[3]*Ci[2][m]
/72;
	
	ctl_pt[   8   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 5.0/12.0*Ci[0][m]+13.0/36.0*Ci[1][m]+Ci[2][m]/12+5.0/36.0*Ci[3][m]+
Ci[8][m]/18-Ci[4][m]/18+c[1]*Ci[0][m]/72+c[1]*Ci[3][m]/72-c[1]*Ci[1][m]/72-c[1]
*Ci[2][m]/72-c[3]*Ci[0][m]/72-c[3]*Ci[1][m]/72+c[3]*Ci[2][m]/72+c[3]*Ci[3][m]
/72;
	
	ctl_pt[   9   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 5.0/24.0*Ci[0][m]+73.0/144.0*Ci[1][m]+7.0/144.0*Ci[2][m]+Ci[3][m]/16
-Ci[6][m]/144+Ci[7][m]/144+7.0/36.0*Ci[8][m]-Ci[4][m]/36+Ci[5][m]/144+c[1]*Ci
[0][m]/144+c[1]*Ci[3][m]/96-c[1]*Ci[1][m]/96-c[1]*Ci[2][m]/96-c[1]*Ci[5][m]/288
+c[1]*Ci[6][m]/288+c[1]*Ci[7][m]/288+5.0/144.0*c[3]*Ci[0][m]+c[3]*Ci[7][m]/96-c
[3]*Ci[5][m]/96-c[3]*Ci[6][m]/96+7.0/288.0*c[3]*Ci[1][m]-7.0/288.0*c[3]*Ci[2]
[m]-7.0/288.0*c[3]*Ci[3][m];
	
	ctl_pt[   10   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -Ci[0][m]/8+7.0/16.0*Ci[1][m]+Ci[2][m]/48+Ci[3][m]/48+Ci[6][m]/48
-11.0/48.0*Ci[7][m]+5.0/6.0*Ci[8][m]+Ci[5][m]/48+c[3]*Ci[0][m]/24-c[3]*Ci[7][m]
/48+c[3]*Ci[5][m]/48+c[3]*Ci[6][m]/48+c[3]*Ci[1][m]/16-c[3]*Ci[2][m]/16-c[3]*Ci
[3][m]/16;
	
	ctl_pt[   11   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[3][m]/6+Ci[4][m]-Ci[5][m]/6;
	
	ctl_pt[   12   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/3+Ci[1][m]/72+Ci[2][m]/72+7.0/24.0*Ci[3][m]-Ci[6][m]/72-Ci
[7][m]/72+Ci[4][m]/3+Ci[5][m]/24+c[1]*Ci[5][m]/72-c[1]*Ci[3][m]/72+c[1]*Ci[1]
[m]/72+c[1]*Ci[2][m]/72-c[1]*Ci[6][m]/72-c[1]*Ci[7][m]/72;
	
	ctl_pt[   13   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 13.0/24.0*Ci[0][m]+Ci[1][m]/18+Ci[2][m]/24+7.0/36.0*Ci[3][m]-Ci[7]
[m]/72+Ci[4][m]/6+Ci[5][m]/72+c[3]*Ci[3][m]/288-c[1]*Ci[0][m]/24-11.0/288.0*c
[1]*Ci[3][m]+11.0/288.0*c[1]*Ci[1][m]+11.0/288.0*c[1]*Ci[2][m]-c[1]*Ci[5][m]
/288+c[1]*Ci[6][m]/288+c[1]*Ci[7][m]/288+c[3]*Ci[7][m]/288-c[3]*Ci[5][m]/288-c
[3]*Ci[6][m]/288-c[3]*Ci[1][m]/288+c[3]*Ci[2][m]/288;
	
	ctl_pt[   14   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 13.0/24.0*Ci[0][m]+7.0/36.0*Ci[1][m]+Ci[2][m]/24+Ci[3][m]/18+Ci[7]
[m]/72+Ci[8][m]/6-Ci[5][m]/72-c[1]*Ci[3][m]/288+c[1]*Ci[1][m]/288+c[1]*Ci[2][m]
/288+c[1]*Ci[5][m]/288-c[1]*Ci[6][m]/288-c[1]*Ci[7][m]/288-c[3]*Ci[0][m]/24-c
[3]*Ci[7][m]/288+c[3]*Ci[5][m]/288+c[3]*Ci[6][m]/288-11.0/288.0*c[3]*Ci[1][m]+
11.0/288.0*c[3]*Ci[2][m]+11.0/288.0*c[3]*Ci[3][m];
	
	ctl_pt[   15   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/3+7.0/24.0*Ci[1][m]+Ci[2][m]/72+Ci[3][m]/72-Ci[6][m]/72+Ci
[7][m]/24+Ci[8][m]/3-Ci[5][m]/72+c[3]*Ci[7][m]/72-c[3]*Ci[5][m]/72-c[3]*Ci[6]
[m]/72-c[3]*Ci[1][m]/72+c[3]*Ci[2][m]/72+c[3]*Ci[3][m]/72;
	
	ctl_pt[   16   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[1][m]/6-Ci[7][m]/6+Ci[8][m];
	
	ctl_pt[   17   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -Ci[3][m]/6+Ci[4][m]+Ci[5][m]/6;
	
	ctl_pt[   18   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/3-Ci[1][m]/72-Ci[2][m]/72+Ci[3][m]/24+Ci[6][m]/72+Ci[7][m]
/72+Ci[4][m]/3+7.0/24.0*Ci[5][m]+c[1]*Ci[3][m]/72-c[1]*Ci[1][m]/72-c[1]*Ci[2]
[m]/72-c[1]*Ci[5][m]/72+c[1]*Ci[6][m]/72+c[1]*Ci[7][m]/72;
	
	ctl_pt[   19   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 13.0/24.0*Ci[0][m]-Ci[1][m]/72+Ci[3][m]/72+Ci[6][m]/24+Ci[7][m]/18+
Ci[4][m]/6+7.0/36.0*Ci[5][m]-c[1]*Ci[0][m]/24-c[1]*Ci[3][m]/288+c[1]*Ci[1][m]
/288+c[1]*Ci[2][m]/288-11.0/288.0*c[1]*Ci[5][m]+11.0/288.0*c[1]*Ci[6][m]+11.0/
288.0*c[1]*Ci[7][m]-c[3]*Ci[7][m]/288+c[3]*Ci[5][m]/288+c[3]*Ci[6][m]/288+c[3]*
Ci[1][m]/288-c[3]*Ci[2][m]/288-c[3]*Ci[3][m]/288;
	
	ctl_pt[   20   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = c[1]*Ci[7][m]/288+13.0/24.0*Ci[0][m]+Ci[1][m]/72-Ci[3][m]/72+Ci[6]
[m]/24+7.0/36.0*Ci[7][m]+Ci[8][m]/6+Ci[5][m]/18+c[1]*Ci[3][m]/288-c[1]*Ci[1][m]
/288-c[1]*Ci[2][m]/288-c[1]*Ci[5][m]/288+c[1]*Ci[6][m]/288+11.0/288.0*c[3]*Ci
[6][m]-c[3]*Ci[0][m]/24-11.0/288.0*c[3]*Ci[7][m]+11.0/288.0*c[3]*Ci[5][m]-c[3]*
Ci[1][m]/288+c[3]*Ci[2][m]/288+c[3]*Ci[3][m]/288;
	
	ctl_pt[   21   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -c[3]*Ci[2][m]/72+Ci[0][m]/3+Ci[1][m]/24-Ci[2][m]/72-Ci[3][m]/72+Ci
[6][m]/72+7.0/24.0*Ci[7][m]+Ci[8][m]/3+Ci[5][m]/72-c[3]*Ci[7][m]/72+c[3]*Ci[5]
[m]/72+c[3]*Ci[6][m]/72+c[3]*Ci[1][m]/72-c[3]*Ci[3][m]/72;
	
	ctl_pt[   22   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -Ci[1][m]/6+Ci[7][m]/6+Ci[8][m];
	
	ctl_pt[   23   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -Ci[0][m]/8+Ci[1][m]/48+Ci[2][m]/48-11.0/48.0*Ci[3][m]+Ci[6][m]/48+
Ci[7][m]/48+5.0/6.0*Ci[4][m]+7.0/16.0*Ci[5][m]+c[1]*Ci[0][m]/24-c[1]*Ci[3][m]
/48+c[1]*Ci[1][m]/48+c[1]*Ci[2][m]/48+c[1]*Ci[5][m]/16-c[1]*Ci[6][m]/16-c[1]*Ci
[7][m]/16;
	
	ctl_pt[   24   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 5.0/24.0*Ci[0][m]+Ci[1][m]/144-Ci[2][m]/144+Ci[3][m]/144+7.0/144.0*
Ci[6][m]+Ci[7][m]/16-Ci[8][m]/36+7.0/36.0*Ci[4][m]+73.0/144.0*Ci[5][m]+5.0/
144.0*c[1]*Ci[0][m]+c[1]*Ci[3][m]/96-c[1]*Ci[1][m]/96-c[1]*Ci[2][m]/96+7.0/
288.0*c[1]*Ci[5][m]-7.0/288.0*c[1]*Ci[6][m]-7.0/288.0*c[1]*Ci[7][m]+c[3]*Ci[0]
[m]/144+c[3]*Ci[7][m]/96-c[3]*Ci[5][m]/96-c[3]*Ci[6][m]/96-c[3]*Ci[1][m]/288+c
[3]*Ci[2][m]/288+c[3]*Ci[3][m]/288;
	
	ctl_pt[   25   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = 5.0/12.0*Ci[0][m]+Ci[6][m]/12+5.0/36.0*Ci[7][m]-Ci[8][m]/18+Ci[4][m]
/18+13.0/36.0*Ci[5][m]-c[1]*Ci[0][m]/72-c[1]*Ci[5][m]/72+c[1]*Ci[6][m]/72+c[1]*
Ci[7][m]/72+c[3]*Ci[0][m]/72+c[3]*Ci[7][m]/72-c[3]*Ci[5][m]/72-c[3]*Ci[6][m]/72
;
	
	ctl_pt[   26   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = c[3]*Ci[5][m]/72+5.0/12.0*Ci[0][m]+Ci[6][m]/12+13.0/36.0*Ci[7][m]+Ci
[8][m]/18-Ci[4][m]/18+5.0/36.0*Ci[5][m]+c[1]*Ci[0][m]/72+c[1]*Ci[5][m]/72-c[1]*
Ci[6][m]/72-c[1]*Ci[7][m]/72-c[3]*Ci[0][m]/72-c[3]*Ci[7][m]/72+c[3]*Ci[6][m]/72
;
	
	ctl_pt[   27   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = c[1]*Ci[2][m]/288+5.0/24.0*Ci[0][m]+Ci[1][m]/144-Ci[2][m]/144+Ci[3]
[m]/144+7.0/144.0*Ci[6][m]+73.0/144.0*Ci[7][m]+7.0/36.0*Ci[8][m]-Ci[4][m]/36+Ci
[5][m]/16+c[1]*Ci[0][m]/144-c[1]*Ci[3][m]/288+c[1]*Ci[1][m]/288+5.0/144.0*c[3]*
Ci[0][m]+c[1]*Ci[5][m]/96-c[1]*Ci[6][m]/96-c[1]*Ci[7][m]/96+c[3]*Ci[1][m]/96+
7.0/288.0*c[3]*Ci[7][m]-7.0/288.0*c[3]*Ci[5][m]-7.0/288.0*c[3]*Ci[6][m]-c[3]*Ci
[2][m]/96-c[3]*Ci[3][m]/96;
	
	ctl_pt[   28   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -c[3]*Ci[1][m]/48-Ci[0][m]/8-11.0/48.0*Ci[1][m]+Ci[2][m]/48+Ci[3][m]
/48+Ci[6][m]/48+7.0/16.0*Ci[7][m]+5.0/6.0*Ci[8][m]+Ci[5][m]/48+c[3]*Ci[0][m]/24
+c[3]*Ci[7][m]/16-c[3]*Ci[5][m]/16-c[3]*Ci[6][m]/16+c[3]*Ci[2][m]/48+c[3]*Ci[3]
[m]/48;
	
	ctl_pt[   29   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -7.0/24.0*Ci[0][m]+Ci[2][m]/24-Ci[3][m]/4+Ci[6][m]/6-Ci[7][m]/8+Ci
[8][m]/6+2.0/3.0*Ci[4][m]+5.0/8.0*Ci[5][m]+c[1]*Ci[0][m]/12-5.0/96.0*c[1]*Ci[3]
[m]+5.0/96.0*c[1]*Ci[1][m]+5.0/96.0*c[1]*Ci[2][m]+13.0/96.0*c[1]*Ci[5][m]-13.0/
96.0*c[1]*Ci[6][m]-13.0/96.0*c[1]*Ci[7][m]-c[3]*Ci[0][m]/24-5.0/96.0*c[3]*Ci[7]
[m]+5.0/96.0*c[3]*Ci[5][m]+5.0/96.0*c[3]*Ci[6][m]+c[3]*Ci[1][m]/96-c[3]*Ci[2]
[m]/96-c[3]*Ci[3][m]/96;
	
	ctl_pt[   30   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/24+5.0/24.0*Ci[6][m]-Ci[7][m]/24+Ci[8][m]/12+Ci[4][m]/12+
5.0/8.0*Ci[5][m]+c[1]*Ci[0][m]/16+c[1]*Ci[5][m]/16-c[1]*Ci[6][m]/16-c[1]*Ci[7]
[m]/16-c[3]*Ci[0][m]/48-c[3]*Ci[7][m]/48+c[3]*Ci[5][m]/48+c[3]*Ci[6][m]/48;
	
	ctl_pt[   31   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[0][m]/4+Ci[6][m]/4+Ci[7][m]/12+5.0/12.0*Ci[5][m];
	
	ctl_pt[   32   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = Ci[5][m]/12+Ci[0][m]/4+Ci[6][m]/4+25.0/60.0*Ci[7][m];
	
	ctl_pt[   33   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -c[3]*Ci[5][m]/16+Ci[0][m]/24+5.0/24.0*Ci[6][m]+5.0/8.0*Ci[7][m]+Ci
[8][m]/12+Ci[4][m]/12-Ci[5][m]/24-c[1]*Ci[0][m]/48-c[1]*Ci[5][m]/48+c[1]*Ci[6]
[m]/48+c[1]*Ci[7][m]/48+c[3]*Ci[0][m]/16+c[3]*Ci[7][m]/16-c[3]*Ci[6][m]/16;
	
	ctl_pt[   34   ][   m   ] = t0;
    }

    for (m=0; m<DIM; m++) {
      t0 = -5.0/96.0*c[1]*Ci[5][m]-7.0/24.0*Ci[0][m]-Ci[1][m]/4+Ci[2][m]/24+Ci
[6][m]/6+5.0/8.0*Ci[7][m]+2.0/3.0*Ci[8][m]+Ci[4][m]/6-Ci[5][m]/8-c[1]*Ci[0][m]
/24+c[1]*Ci[3][m]/96-c[1]*Ci[1][m]/96-c[1]*Ci[2][m]/96+13.0/96.0*c[3]*Ci[7][m]+
5.0/96.0*c[1]*Ci[6][m]+5.0/96.0*c[1]*Ci[7][m]+c[3]*Ci[0][m]/12+5.0/96.0*c[3]*Ci
[2][m]-13.0/96.0*c[3]*Ci[5][m]-13.0/96.0*c[3]*Ci[6][m]-5.0/96.0*c[3]*Ci[1][m]+
5.0/96.0*c[3]*Ci[3][m];
	
	ctl_pt[   35   ][   m   ] = t0;
    }

}
/* Pcp2Nurb:
 * convert submesh of a planar cut polyhedron into a nurbs patch 
 *
 * generated by jorg@cs.purdue.edu Jan 1996
 * all rights reserved!
 */
