/*  Driver routine for Pcp2Nurb
 *  
 *	generates a twisted cube from a planar cut polyhedron (pcp)
 *      to illustrate surface of medium complexity
 *
 *      HINT: if confused read the ACM TOMS paper
 *
 *	     Pcp2Nurb: smooth free-form surfacing
 *	     with linearly-trimmed bicubic B-splines
 *				by Jorg Peters
 *
 *  input: file with (72) pcp subnets
 *  output: Open Inventor NurbsSurfaces
 *
 *  jorg@cs.purdue.edu Jan 1996
 *  IRIX 5.3
 *		COMPILATION and EXECUTION sequence
 *
 *  modify makefile according to your platform
 *  make 
 *  nurb_iv2 in2
 */
#include <stdio.h>
#include <stdlib.h>
#include "glut.h"

#define PTS	6	/* number of B-spline control points per row */
#define DEG	3	/* degree of B-spline patch */
#define PFL	4	/* number of points in profile (trimming) curve */
#define KTS	DEG+PTS+1	/* number of knots per row */
#define DIM	3	
#define SUB	9	/* size of subnet */
#define FIN     4	/* fineness: points evaluated per side */

void  Pcp2Nurb( float Ci[][DIM], int valence[],
		float ctl_pt[][DIM], float knots[]);

/* opengl initialize */
void windowinit(int argc, char *argv[]);

/* glut callbacks */
void display();
void mouseButton(int button, int state, int x, int y);
void mouseMotion(int x, int y);
void keyboard(unsigned char key, int x, int y);
void spin();

/* patch drawing functions */
void define_patch(float ctl_pt[][DIM]);
void define();
void DrawTriangle(GLfloat ctrl_pt[][DIM], int Deg, int fineness, int orient);

/*--------------------------VARS------------------------------------*/
    /* nine-point subnet */
    float	Pcp[SUB][DIM];  
    int		valence[2];

    /* profile (trimming) curve and control points of NURBS */
    float	profil[4][2] = {
		    {0,3},
		    {-3,0},
		    {0,-3},
		    {3,0} };
    float	ctl_pt[40][DIM], knots[KTS];

    int		i,j, no_subnets;

    FILE* datafile;
    
main(int argc, char *argv[])
{


/*------------------------------------------------------------------*/
    if (argc != 2) {
            printf("Usage: %s <filename>\n", argv[0]);
            exit(-1);
    }

    if((datafile = fopen(argv[1], "r")) <=0 ) {
            printf("input file  %s not found\n", argv[0]);
            exit(-1);
    }    

    /* openGL initializations */
    windowinit(argc, argv);
    
    /*  generate the object */
    define();

    
    glutMainLoop(); /* enter event loop */
    return 0;

}


void define()
{
    glNewList(1, GL_COMPILE); 

    /* read the data from input file */
    fseek(datafile, 0, SEEK_SET);
    fscanf(datafile, "%d", &no_subnets);
    for (i=0; i< no_subnets; i++) {
	fscanf(datafile, "%d %d",&valence[0], &valence[1]);
	for (j=0; j< SUB; j++) 
	    fscanf(datafile, "%f %f %f",&Pcp[j][0], &Pcp[j][1], &Pcp[j][2]);

	/* transform subset Pcp of planar cut polyhedron to a B-spline patch */
	Pcp2Bezier(Pcp,valence, ctl_pt);

	define_patch(ctl_pt);
    }
    glEndList();

    fclose(datafile);

}

/*------------------------------------------------------------------*/


/*  OPENGL output  */

void windowinit(int argc, char *argv[])
{
    GLfloat light_ambient[]={0.5, 0.5, 0.5, 1.0};
    GLfloat light_diffuse[]={1.0, 1.0, 1.0, 0.1};
    GLfloat light_specular[]={0.4, 0.4, 0.4, 1.0};
    GLfloat light_position[]={1.0,1.0,1.0,0.0};

    GLfloat mat_specular[]={1.0, 1.0, 1.0, 1.0};
    GLfloat mat_diffuse[] ={0.2, 0.2, 0.2, 0.1};
    GLfloat mat_ambient[]  ={0.7,0.5,1.0,1.0};
    GLfloat mat_shininess ={20.0};


    glutInit(&argc,argv);
    glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);

    glutInitWindowSize(500,500);  /* 500 x 500 pixel window */
    glutInitWindowPosition(0,0);  /* place window top left on display */
    glutCreateWindow(argv[0]); /* window title */

    glutKeyboardFunc(keyboard);   /* set glut callback functions */
    glutIdleFunc(spin);
    glutMouseFunc(mouseButton);
    glutMotionFunc(mouseMotion);
    glutDisplayFunc(display); 

    /* set up ambient, diffuse, and specular components for light 0 */

    glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
    glLightfv(GL_LIGHT0, GL_POSITION, light_position);

    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
    glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
    glMaterialf(GL_FRONT, GL_SHININESS, mat_shininess);

    glEnable(GL_LIGHTING);   /* enable lighting */
    glEnable(GL_LIGHT0);     /* enable light 0 */
    glEnable(GL_DEPTH_TEST); /* enable z buffer */

    /* attributes */
    glClearColor(0.0, 0.0, 0.0, 0.0); /* black background */

    /* set up viewing */
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(45.0, 1, 1.0, 20.0);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glTranslated(0, 0, -9);    /* Sets the view point of projection */

    glShadeModel(GL_SMOOTH); /*enable smooth shading */
    glEnable(GL_NORMALIZE);

}

void define_patch(float ctl_pt[][DIM])
{
	DrawTriangle(ctl_pt, 3, FIN, 0);
	DrawTriangle(&(ctl_pt[10]), 3, FIN, 0);
	DrawTriangle(&(ctl_pt[20]), 3, FIN, 0);
	DrawTriangle(&(ctl_pt[30]), 3, FIN, 0);
}


void draw(void)
{

	/* clears the screen */
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	// show one patch
	glCallList(1);

	glFlush(); /* clear buffers */
}

void keyboard(unsigned char key, int x, int y)
{
    GLfloat mat_ambient[3][4] = {
	{0.7,0.5,1.0,1.0},
	{0.2,0.7,0.7,1.0},
	{0.8,0.8,0.8,1.0}
    };

    static int F_Color=0;
    
    switch(key)
    {
    case 27:  /* press ESC to quit */
        exit(0);
        break;
    case 'Z':
	glScaled(0.7, 0.7, 0.7);
        display();
    	break;
    case 'z':
	glScaled(1.4, 1.4, 1.4);
        display();
    	break;
    case 'c':
        F_Color = (F_Color+1)%3;
        glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, mat_ambient[F_Color]);
        display();
    	break;
    }
}

