/*  Driver routine for Pcp2Nurb
 *	generates three Inventor NurbsSurface s
 *      that smooth the corner of a cube with vertices +/-2 
 *	using an adjustable blend ratio with default 0.5.
 *	A fourth biquadratic patch is C1 attached.
 *
 *      HINT: if confused read the ACM TOMS paper
 *
 *	     Pcp2Nurb: smooth free-form surfacing
 *	     with linearly-trimmed bicubic B-splines
 *				by Jorg Peters
 *
 *
 *  jorg@cs.purdue.edu Jan 1996
 *  IRIX 5.3
 *		COMPILATION and EXECUTION sequence
 *  make
 *  nurb_iv1 0.5
 *  (try also other values in [0,1]: 0.35 0.65 0.0 1.0 --
 *   10 and -10 are not  legal values -- so try them, too)
 */
#include <stdio.h>
#include <stdlib.h>
#include "glut.h"

#define DEG	3	/* degree of B-spline patch */
#define DIM	3	

#define FIN     10	/* fineness: points evaluated per side */

void Pcp2Bezier(float Ci[][DIM], int valence[],
    float ctl_pt[][DIM]);

/* opengl initialize */
void windowinit(int argc, char *argv[]);

/* glut callbacks */
void display();
void mouseButton(int button, int state, int x, int y);
void mouseMotion(int x, int y);
void keyboard(unsigned char key, int x, int y);
void spin();

/* patch drawing functions */
void define_patch(float ctl_pt[][DIM]);
void define_bi2( float ctl_pt[][DIM], float more_ctl_pt[][DIM]);
void DrawTriangle(GLfloat ctrl_pt[][DIM], int Deg, int fineness, int orient);

main( int argc, char *argv[])
{
/*--------------------------VARS------------------------------------*/
    /* profile (trimming) curve and control points of NURBS */

    float ctl_pt[40][DIM];  /* bezier control points */

    /*  9 (control and vertex) points of a planar-cut polyhedron
     *          arranged in the order (cf. Figure 5 of the accompanying paper)
     *      8 7 6
     *      1 0 5    4,8 are vertex points
     *      2 3 4
     */
    float	Pcp[9][DIM] = {
		      {  1, 1, 2},	
		      { -1, 1, 2},
		      { -1, 2, 1},
		      {  1, 2, 1},
		      {  1.333333, 1.333333, 1.333333},
			  /* V_1 = avg of 1,1,2; 1,2,1; 2,1,1 */
		      {  2, 1, 1},
		      {  2,-1, 1},
		      {  1,-1, 2},
		      {  0, 0, 2}};
    int		valence[2] = {3,4};	/* V_1 has 3 neighbors */

    /* control points for an abutting biquadratic patch:
     * reuse Pcp[3], Pcp[0], Pcp[7], Pcp[1], Pcp[2], and
     * specify 4 additional vertices: */
    float       Cbi2[4][DIM] = {
			{ -1, -1, 2},	
			{ -3, -1, 2}, { -3, 1, 2}, { -3, 2, 1}};

    double	blend_ratio = 0.5;
    int		i;

/*------------------------------------------------------------------*/
    if (argc > 1)
    {
	blend_ratio = atof((const char*) argv[1]) ;		
    }

    /* to minimize the amount of code describing the planar cut polyhedron
       I take advantage of symmetry:     */
    Pcp[0][0] = Pcp[0][1] = Pcp[3][0] = Pcp[3][2] = Pcp[5][1] = Pcp[5][2] = 
	2*(1-blend_ratio)+ 0*blend_ratio;
    /* adjust centroids in case the blend ratio has been changed */
    for (i=0; i< 3; i++) {
	Pcp[4][i] = (Pcp[0][i]+Pcp[3][i]+Pcp[5][i])/3;
	Pcp[8][i] = (Pcp[1][i]+Pcp[0][i]+Pcp[7][i]+Cbi2[0][i])/4;
    }


    /* transform subset Pcp of planar cut polyhedron to a B-spline patch */
    Pcp2Bezier(Pcp,valence, ctl_pt);

    /* glut initialiation */
    windowinit(argc, argv);
    
    define_patch(ctl_pt);	  /* define the patches */
    define_bi2(Pcp, Cbi2);

    glutMainLoop(); /* enter event loop */
    return 0;

/*------------------------------------------------------------------*/
}


/*  OPENGL output  */

void windowinit(int argc, char *argv[])
{
    GLfloat light_ambient[]={0.5, 0.5, 0.5, 1.0};
    GLfloat light_diffuse[]={1.0, 1.0, 1.0, 0.1};
    GLfloat light_specular[]={0.4, 0.4, 0.4, 1.0};
    GLfloat light_position[]={1.0,1.0,1.0,0.0};

    GLfloat mat_specular[]={1.0, 1.0, 1.0, 1.0};
    GLfloat mat_diffuse[]={0.2, 0.2, 0.2, 0.1};
    GLfloat mat_shininess={20.0};

    glutInit(&argc,argv);
    glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);

    glutInitWindowSize(500,500);  /* 500 x 500 pixel window */
    glutInitWindowPosition(0,0);  /* place window top left on display */
    glutCreateWindow(argv[0]); /* window title */

    glutKeyboardFunc(keyboard);   /* set glut callback functions */
    glutIdleFunc(spin);
    glutMouseFunc(mouseButton);
    glutMotionFunc(mouseMotion);
    glutDisplayFunc(display); 

    /* set up ambient, diffuse, and specular components for light 0 */

    glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
    glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
    glLightfv(GL_LIGHT0, GL_POSITION, light_position);

    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
    glMaterialf(GL_FRONT, GL_SHININESS, mat_shininess);

    glEnable(GL_LIGHTING);   /* enable lighting */
    glEnable(GL_LIGHT0);     /* enable light 0 */
    glEnable(GL_DEPTH_TEST); /* enable z buffer */

    /* attributes */
    glClearColor(0.0, 0.0, 0.0, 0.0); /* black background */

    /* set up viewing */
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(45.0, 1, 1.0, 20.0);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glTranslated(0, 0, -9);    /* Sets the view point of projection */

    glShadeModel(GL_SMOOTH); /*enable smooth shading */

}

void define_patch(float ctl_pt[][DIM])
{

    glNewList(1, GL_COMPILE);

	DrawTriangle(ctl_pt, 3, FIN, 0);
	DrawTriangle(&(ctl_pt[10]), 3, FIN, 0);
	DrawTriangle(&(ctl_pt[20]), 3, FIN, 0);
	DrawTriangle(&(ctl_pt[30]), 3, FIN, 0);
	
    glEndList();

}

void draw_cube()
{
    glColor3f(1.0,1.0,1.0);
    glBegin(GL_LINE_STRIP);
	glVertex3f(2.0,2.0,2.0);
	glVertex3f(2.0,-2.0,2.0);
	glVertex3f(-2.0,-2.0,2.0);
	glVertex3f(-2.0,2.0,2.0);
	glVertex3f(2.0,2.0,2.0);
	glVertex3f(2.0,2.0,-2.0);
	glVertex3f(2.0,-2.0,-2.0);
	glVertex3f(-2.0,-2.0,-2.0);
	glVertex3f(-2.0,2.0,-2.0);
	glVertex3f(2.0,2.0,-2.0);

	glVertex3f(2.0,-2.0,-2.0);
	glVertex3f(2.0,-2.0,2.0);
	glVertex3f(-2.0,-2.0,2.0);
	glVertex3f(-2.0,-2.0,-2.0);
	glVertex3f(-2.0,2.0,-2.0);
	glVertex3f(-2.0,2.0,2.0);
	
    glEnd();

}

	
void define_bi2(float ctl_pt[][DIM], float more_ctl_pt[][DIM])
{
    GLfloat point[9][DIM];
    GLfloat knots [6]=
		 {0.0, 1.0, 2.0, 3.0, 4.0, 5.0};
    GLfloat ambient[] = {0.3,0.3,1.0,1.0};

    GLUnurbsObj* patch;
    GLint i, j, k;
    
    i = 7;
    for(k=0; k<DIM; k++)
	point[0][k] = ctl_pt[i][k];
    i = 0;
    for(k=0; k<DIM; k++)
	point[1][k] = ctl_pt[i][k];
    i = 3;
    for(k=0; k<DIM; k++)
	point[2][k] = ctl_pt[i][k];
    j = 0;
    for(k=0; k<DIM; k++)
	point[3][k] = more_ctl_pt[j][k];
    i = 1;
    for(k=0; k<DIM; k++)
	point[4][k] = ctl_pt[i][k];
    i = 2;
    for(k=0; k<DIM; k++)
	point[5][k] = ctl_pt[i][k];
    j = 1;
    for(k=0; k<DIM; k++)
	point[6][k] = more_ctl_pt[j][k];
    j = 2;
    for(k=0; k<DIM; k++)
	point[7][k] = more_ctl_pt[j][k];
    j = 3;
    for(k=0; k<DIM; k++)
	point[8][k] = more_ctl_pt[j][k];

    patch = gluNewNurbsRenderer();
    gluNurbsProperty(patch, GLU_SAMPLING_TOLERANCE, 300/FIN);
    gluNurbsProperty(patch, GLU_DISPLAY_MODE, GLU_FILL);
    glNewList(2, GL_COMPILE);
        glEnable(GL_AUTO_NORMAL);
        glEnable(GL_NORMALIZE);
	glMaterialfv(GL_FRONT, GL_AMBIENT, ambient);
        gluBeginSurface(patch);
	gluNurbsSurface (patch, 6, knots, 6, knots, 3, 3*3, 
	    &point[0][0], 3,3, GL_MAP2_VERTEX_3);
	gluEndSurface(patch);
    glEndList();

}


void draw(void)
{
	GLfloat axis[3] = {1.0,1.0,1.0};

	GLfloat ambient1[] = {0.8,0.8,0.8,1.0};
	GLfloat ambient2[] = {1.0,0.5,0.7,1.0};
	GLfloat ambient3[] = {0.7,0.5,1.0,1.0};

	/* clears the screen */
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	//show box
	draw_cube();

	//show bi2
	glCallList(2);

	// show one patch
	glMaterialfv(GL_FRONT, GL_AMBIENT, ambient1);
	glCallList(1);

	// rotate it by 120.
	glPushMatrix();
	glMaterialfv(GL_FRONT, GL_AMBIENT, ambient2);
	glRotatef(120.0, axis[0],axis[1],axis[2]);
	glCallList(1);
	glPopMatrix();

	// rotate it by 240.
	glPushMatrix(); 
	glMaterialfv(GL_FRONT, GL_AMBIENT, ambient3);
	glRotatef(240.0, axis[0],axis[1],axis[2]);
	glCallList(1);
	glPopMatrix();

	glFlush(); /* clear buffers */
}

void keyboard(unsigned char key, int x, int y)
{
        if (key ==27)  /* press ESC to quit */
        {
            exit(0);
        }
}
