/* OpenGL routine for drawing a triangle Patch 
 *    Control point coordinates entered in order:
 *                9
 *	         7 8
 *		4 5 6
 *	       0 1 2 3
 *
 *		by Jorg Peters
 *		jorg@cise.ufl.edu
 *
 *  All rights reserved
 */
#include <stdio.h>
#include <string.h>
#include "glut.h"

#define DIM 3
#define MAXDEG 20 
#define MAXFIN 400

/* ----------------------------------------------------
   DrawTriangle : Draw the triangle bezier patch.

   Parameter:
	ctrl_pt[][]: the bezier control points
        DEG:	     degree of the bezier patch
	fineness:    points eveluated per side
        oreint:      patch orientation
 ------------------------------------------------------ */

void DrawTriangle(GLfloat ctrl_pt[][DIM], int DEG, int fineness, int orient)
{

    GLint   i,j,k;
    GLint   d,m;
    GLfloat u, v, w; /* parameter of patch */ 

    GLfloat DeCastel[(MAXDEG+1)*(MAXDEG+2)/2][DIM];

    /* ------
       For every step of evaluation in De Casteljau algorithm,
       to calculate the ith point, we need to know which triangle
       it comes from the former layer. Because we are using
       one dimension array, so we can save the vertices' index
       use two int arrays:
       
                           /\ <- Pyramid_Up[i]
                          /  \
                         / i  \
      Pyramid_Down[i]-> -------- <-Pyramid_Down[i]+1

		for each layer of de Casteljau algorithm
    */

    GLint   Pyramid_Down[MAXDEG][MAXDEG*(MAXDEG+1)/2];
    GLint   Pyramid_Up[MAXDEG][MAXDEG*(MAXDEG+1)/2];

    /* indexes of vertices on left side used to 
       calculate the above pyramid */
    GLint   old_left[MAXDEG], new_left[MAXDEG]; 

    /* Array used to save one line's Points and Normals */
    GLfloat Point_Save[MAXFIN][DIM];
    GLfloat Normal_Save[MAXFIN][DIM];

/* -----------  Code starts here ------------ */

    /* Step 1. Caculate the index Pyramid */
    for(d=DEG-1; d>=1; d--)
    {
	old_left[0]=0;
	for(i =1;i <= d+1; i++) 
	    old_left[i] = old_left[i-1]+(d+3-i);

	new_left[0]=0;
	for(i=1; i<=d; i++)
	    new_left[i] = new_left[i-1]+(d+2-i);

	i=0;
	j=0;
	for(k=0;k<(d+1)*(d+2)/2; k++)
	{
	    if(k>=new_left[i+1]) {
		i++;
		j=0;
	    }
	    Pyramid_Down[d][k] = old_left[i]+j;
	    Pyramid_Up[d][k] = old_left[i+1]+j;
	    j++;
	}
    }


    /* Step 2. Draw triangle patch */
    for (i=0; i<=fineness; i++)
    {
	glBegin(GL_TRIANGLE_STRIP);

	for (j=0;j<=fineness-i;j++)
	{
	    GLfloat Point[3];
	    GLfloat Normal[3];
	    GLfloat v1[3], v2[3];

	    u = (float)i/fineness;
	    v = (float)j/fineness;
	    w = 1-u-v;

	    /* initialize the DeCastel Array */
	    for (k = 0; k <(DEG+1)*(DEG+2)/2; k++)
		for( m = 0; m <DIM; m++)
	            DeCastel[k][m] = ctrl_pt[k][m]; 

	    /* de Casteljau algorithm */

	    /* stop at degree 2 to calculate the normal */
	    for (d = DEG ; d >=2; d--)
		for( k = 0; k < d*(d+1)/2 ; k++)
		{
		    DeCastel[k][0] = u* DeCastel[Pyramid_Down[d-1][k]][0] +
			v* DeCastel[Pyramid_Down[d-1][k]+1][0] +
			w* DeCastel[Pyramid_Up[d-1][k]][0];
		    DeCastel[k][1] = u* DeCastel[Pyramid_Down[d-1][k]][1] +
			v* DeCastel[Pyramid_Down[d-1][k]+1][1] +
			w* DeCastel[Pyramid_Up[d-1][k]][1];
		    DeCastel[k][2] = u* DeCastel[Pyramid_Down[d-1][k]][2] +
			v* DeCastel[Pyramid_Down[d-1][k]+1][2] +
			w* DeCastel[Pyramid_Up[d-1][k]][2];
	        }

	    /* calculate the degree 2 triangle's vectors */
	    v1[0]=DeCastel[1][0]-DeCastel[0][0];
	    v1[1]=DeCastel[1][1]-DeCastel[0][1];
	    v1[2]=DeCastel[1][2]-DeCastel[0][2];

	    v2[0]=DeCastel[2][0]-DeCastel[0][0];
	    v2[1]=DeCastel[2][1]-DeCastel[0][1];
	    v2[2]=DeCastel[2][2]-DeCastel[0][2];

	    /* Normal = v1 X v2 */
	    Normal[0]= -(v2[2]*v1[1]-v1[2]*v2[1]);
	    Normal[1]= -(v2[0]*v1[2]-v1[0]*v2[2]);
	    Normal[2]= -(v2[1]*v1[0]-v1[1]*v2[0]);

	    if (orient)
	    {
		Normal[0] = - Normal[0];
		Normal[1] = - Normal[1];
		Normal[2] = - Normal[2];
	    }


	    /* Last step of de Casteljau algorithm */

	    Point[0] = u* DeCastel[0][0] + v* DeCastel[1][0] +
		w* DeCastel[2][0];
	    Point[1] = u* DeCastel[0][1] + v* DeCastel[1][1] +
		w* DeCastel[2][1];
	    Point[2] = u* DeCastel[0][2] + v* DeCastel[1][2] +
		w* DeCastel[2][2];

	    if (i>0) /* no draw action for first line */
	    {
	        /* make use of the former line to form a triangle strip */
		glNormal3fv(Normal_Save[j]);
	    	glVertex3fv(Point_Save[j]);

	        glNormal3fv(Normal); /* Current Normal and Point */
		glVertex3fv(Point);
	    }

	    /* Save the point for being used when drawing next line */
	    for ( m=0; m<DIM; m++)
	    {
	        Point_Save[j][m] = Point[m];
	        Normal_Save[j][m] = Normal[m];
	    }
	}

	/* last point the complete the triangle strip */
	if(i>0)
	{
	    glNormal3fv(Normal_Save[j]);
    	    glVertex3fv(Point_Save[j]);
	}
	glEnd();
    }
}

