function sps = tpsplit(sp, rn, knotlocal)
% sps = tpsplit(sp, rn, knotlocal)
% TPSPLIT -- Split a TP spline SP into RN subpatches by knot
% insertion. RN is a vector with RN(i) determining into how many
% subpatches SP is to be split along the i-th variable direction. If RN
% is omitted it defaults to the degrees of SP. The split is always uniform.
%
% SPS is a cell array of dimensions RN, with SPS{i,j} containing a
% B-spline structure as returned by SPMAK. If KNOTLOCAL is true, the knot
% sequences of the splines in SPS are the ones being inserted into SP. If
% KNOTLOCAL is false, they are the standard knot sequence for a Bernstein
% polynomial.
%
% This routine works only for bivariate TP Bernstein polynomials, though
% it would be trivial to make it work for general B-splines.

% Created by lutter on Fri May 21 15:48:06 1999

% The whole premise of this routine is broken: the redistributing of
% coefficients into a cell array of splines might be convenient, but is a
% real performance killer. It makes routines that use TPSPLIT easier to
% understand, though.

d = sp.order - 1;
nvar = length(d);
if nargin < 2
  rn = sp.order - 1;
end
if nargin < 3 | strcmp(knotlocal, 'local')
  knotlocal = 1;
else
  knotlocal = 0;
end

% Produce the (local) refined knot sequences and the global knot sequence
% of a degree D TP Bernstein polynomial
knots = cell(1, nvar);
refknots = cell(1, nvar);
for i=1:nvar
  knots{i} = [ zeros(1, d(i)+1) ones(1, d(i)+1) ];
  refknots{i} = brk2knt((1:rn(i)-1)/rn(i) , (d(i)+1)*ones(1, rn(i)-1));
  refknots{i} = (1-refknots{i}) * sp.knots{i}(1) ...
		+ refknots{i}*sp.knots{i}(end);
end

spr = sprfn(sp, refknots);

% Split the refined patches into a cell array of splines
sps = cell(rn(1), rn(2));
for i=1:rn(1)
  for j=1:rn(2)
    if ~ knotlocal
      sps{i,j} = spmak({ subknots(spr.knots{1}, i, d(1))  , ...
			 subknots(spr.knots{2}, j, d(2)) }, ...
		       spr.coefs(:, subctrl(i, d(1)), subctrl(j, d(2))));
    else
      sps{i,j} = spmak(knots, ...
		       spr.coefs(:, subctrl(i, d(1)), subctrl(j, d(2))));
    end
  end
end
    
function ind = subctrl(i, d)
  ind = (i-1)*(d+1)+1:i*(d+1);
  
function sk = subknots(knots, i, d)
  sk = knots((i-1)*(d+1)+1:(i+1)*(d+1));
