function [P, M] = tprange(D, psi, psiP, psiM, niter, ndepth)
% [P, M] = tprange(D, psiP, psiM, niter, ndepth)
% TPRANGE -- Generate new estimates P and M for the range of the
% antidifference basis PSI for the difference operators D from initial
% estimates PSIP and PSIM. Iterate the estimation NITER times based on a
% subdivision of PSI into NDEPTH*ORDER pieces where ORDER is the order of
% the functions PSI.
% 
% If PSIP and PSIM are omitted or the empty matrix, they are initialised
% by using the minmax criterion. NITER and NDEPTH default to 1.

% Created by lutter on Mon Sep 27 11:45:49 1999

nop = D.k;
m = D.m;
n = D.n;
if ~ exist('psiP', 'var') | isempty(psiP)
  fprintf('Initializing range estimates\n');
  s = size(psi.coefs);
  coefs = reshape(psi.coefs, nop, m*n);
  psiP = repmat(max(coefs, [], 2), 1, m*n) - coefs;
  psiM = repmat(min(coefs, [], 2), 1, m*n) - coefs;
end
if ~ exist('niter', 'var')  niter = 1; end;
if ~ exist('ndepth','var')  ndepth = 1; end;

if prod(size(ndepth)) == 1
  ndepth = ndepth*[1 1];
end

rn = ndepth.*(psi.order-1);
split = tpsplit(psi, rn);

P = psiP;
M = psiM;

m1 = (m-1)*ndepth(1)+1;
n1 = (n-1)*ndepth(2)+1;

istrip = strips(m, ndepth(1));
jstrip = strips(n, ndepth(2));

x = zeros(1, m1);
y = zeros(1, n1);
for i1=1:ndepth(1)
  sp = split{i1, 1};
  x(istrip(i1,:)) = bary(sp.knots{1}, sp.order(1), i1, ndepth(1));
end
for j1=1:ndepth(2)
  sp = split{1, j1};
  y(jstrip(j1,:)) = bary(sp.knots{2}, sp.order(2), j1, ndepth(2));
end

f = cell(m-1, n-1);
g = cell(m-1, n-1);
for iter=1:niter
  fprintf('Iteration %d:', iter);
  for k=1:nop
    fprintf(' %d',k);
    for i=1:m-1
      for j=1:n-1
	f{i,j} = -inf*ones(m1, n1);
	g{i,j} = -inf*ones(m1, n1);
	for i1=1:ndepth(1)
	  for j1=1:ndepth(2)
	    sp = split{(i-1)*ndepth(1)+i1, (j-1)*ndepth(2)+j1};
	    [upper, lower] = tpenv(sp.coefs(k,:,:), D, psiP, psiM);
	    is = istrip(i1,:);
	    js = jstrip(j1,:);
	    f{i,j}(is, js) = max(f{i,j}(is, js), upper);
	    g{i,j}(is, js) = max(g{i,j}(is, js), -lower);
	  end
	end
      end
    end
    P(k,:) = bilinroof(x, y, f, m, n, psiP(k,:));
    M(k,:) = - bilinroof(x, y, g, m, n, - psiM(k,:));
  end
  fprintf('\n');
  coefs = reshape(psi.coefs, [k m*n]);
  P = P - coefs;
  M = M - coefs;
  dP = psiP-P;
  dM = M - psiM;
  [ max(max(dP)) min(min(dP)); max(max(dM)) min(min(dM)) ]
  psiP = P;
  psiM = M;
end

function xi = bary(knots, order, k, n)
  xi = aveknt(knots, order);
  xi = (1-xi)*(k-1)/n + xi*k/n;

function ind = strips(m,nstrips)
  ind = zeros(nstrips, m);
  ind(1,:) = 1:m;
  for i=2:nstrips
    ind(i,:) = (m-1)*(i-1)+1:(m-1)*i+1;
  end
  
function q = bilinroof(u, v, f, m, n, q0)
  %% Given function values at x and f (as (m-1)x(n-1) cell arrays)
  %% compute a pcw. bilinear function of size mxn that stays above the
  %% pcw. bilinear interpolant of f
  nvar = m*n;
  [m1 n1] = size(f{1,1});
  nconstr = prod([m-1 n-1 4 n1]); % Assume all cells in f are same
				  % size
  A = sparse(nconstr, nvar);
  c = zeros(1,nvar);
  b = zeros(nconstr,1);
  ub = knt2brk(u);
  vb = knt2brk(v);
  S = [ sum(sum((1-ub')*(1-vb))) sum(sum(ub'*(1-vb))), ...
	sum(sum(ub'*vb)) sum(sum((1-ub')*vb)) ];
  constr = 1;
  for i=1:m-1
    for j=1:n-1
      vars = sub2ind([m n], [i i+1 i+1 i], [j j j+1 j+1]);
      c(vars) = c(vars) + S;
      %% Optimize constraints: for each column, only the constraints
      %% corresponding to the two largest f-values can ever be active
      active = zeros(m1, n1);
      for i1 = 1:m1
	active(i1, linenv(v, f{i,j}(i1,:))) = 1;
      end
      for j1 = 1:n1
	ind = find(active(:,j1));
	if ~isempty(ind)
	  active(:,j1) = 0;
	  active(ind(linenv(u(ind), f{i,j}(ind, j1)')), j1) = 1;
	end
      end
      for i1=1:m1
	for j1=find(active(i1,:))
	  A(constr, vars) = -[ (1-u(i1))*(1-v(j1)), u(i1)*(1-v(j1)), ...
			       u(i1)*v(j1), (1-u(i1))*v(j1) ];
	  b(constr) = -f{i,j}(i1,j1);
	  constr = constr+1;
	end
      end
    end
  end
  A(constr:nconstr,:) = [];
  b(constr:nconstr,:) = [];
  q = lp(c, full(A), b, [], [], q0)';
  
function ind = linenv(x, y)
  % LINENV -- Find the subset of vertices of the polygon given by [x; y]
  % that supports the upper linear envelope of the polygon. In other
  % words: find all the nonconcave corners of the polygon and return
  % their indices.
  %
  % The implementation is pure crud.

  % Created by lutter on Mon Oct  4 22:10:10 1999

  n = length(x);
  ind = zeros(1,n);
  ind(1)=1;
  i=1;
  while i<n
    phi = angle(y-y(i)+(x-x(i))*sqrt(-1));
    [dum j] = min(phi(n:-1:i+1));
    i = n+1-j;
    ind(i)=1;
  end
  ind = find(ind);
