function [upper, lower] = tpenv(sp, D, psiP, psiM)
% [upper, lower] = tpenv(sp, D, psiP, psiM)
% TPENV -- Compute the upper and lower envelope (M) of the tensor product
%          spline SP with respect to difference operators D and
%          corresponding psi constants PSIP and PSIM.
%          D has to be a struct returned by MKNETS and PSIP and PSIM have
%          to be matrices: each row corresponds to the operator in the
%          same row of D and the columns correspond to vertices of the
%          control nets. 
%
%          If SP is a function (sp.dim == 1), then UPPER and LOWER are
%          matrices the same size as the control polygon of SP; piecewise
%          bilinear interpolation of the entries in UPPER and LOWER
%          yields functions that are above resp. below all of SP.
%
%          If SP is vector valued (sp.dim > 1), then UPPER(i,:,:) and
%          LOWER(i,:,:) have the same interpretation for the i-th
%          coordinate function of SP, i=1:sp.dim


% Created by lutter on Wed Jun  9 17:06:10 1999
%
%

if isstruct(sp)
  spdim = sp.dim;
  coefs = sp.coefs;
else
  coefs = sp;
  if length(size(coefs)) == 2
    coefs = reshape(coefs, [1 size(coefs)]);
  end
  spdim = size(coefs,1);
end
m = D.m;
n = D.n;


z = zeros(m*n-4, 1);
P = psiP';
M = psiM';
if spdim > 1
  upper = zeros([spdim, m, n]);		% Upper envelope
  lower = zeros([spdim, m, n]);		% Lower envelope

  for dim = 1:spdim
    net = coefs(dim, :, :);
    % Compute the differences of the control net of the dim-th coordinate
    % function of SP and split them into the positive and negative part
    diff2 = D.nets * net(:);
    diff2p = max(z, diff2);
    diff2m = min(z, diff2);
    upper(dim, :, :) = net + reshape(P*diff2p + M*diff2m, [1 m n]);
    lower(dim, :, :) = net + reshape(P*diff2m + M*diff2p, [1 m n]);
  end
else
  upper = zeros(m, n);
  lower = zeros(m, n);

  net = squeeze(coefs);
  % Compute the differences of the control net of the dim-th coordinate
  % function of SP and split them into the positive and negative part
  diff2 = D.nets * net(:);
  diff2p = max(z, diff2);
  diff2m = min(z, diff2);
  upper = net + reshape(P*diff2p + M*diff2m, [m n]);
  lower = net + reshape(P*diff2m + M*diff2p, [m n]);
end
