/******************************************************************************
 *
 *   Author: Rajit Manohar (rajit@vlsi.cs.caltech.edu)
 *   Started On: Fri Jul 22 21:25:14 PDT 1994
 *
 *   Note:
 *     The purpose of this file is to provide a simple portable method for
 *   file i/o. The files created by this program will be portable across 
 *   architechtures. This program assumes very little about the architechture
 *   of the machine it is on and the tools provided and so should compile 
 *   without modification on any machine with an ANSI C compliant compiler.
 *
 *   This file provides the following functions:
 *
 *      pioopenread (int *file_handle, char *filename, int *len)
 *        This function returns a unique file descriptor which can be used
 *      to refer to the file in other function calls. It opens the file
 *      specified by "filename" for reading. If the file doesn't exist, it
 *      returns a value of -1 in "file_handle".
 *
 *      pioopenwrite (int *file_handle, char *filename, int *len)
 *        This function returns a unique file descriptor which can be used
 *      to refer to the file in other function calls. It opens the file
 *      specified by "filename" for writing. If the file exists, then the
 *      existing contents of the file are lost.
 *
 *      pioclose (int *file_handle)
 *        It closes the file specified by "file_handle".
 * 
 *      piowritereal (int *file_handle, float *array, int *size)
 *        This function writes out "size" floating-point values stored in 
 *      "array" to the file specified by "file_handle".
 *
 *      piowriteint (int *file_handle, int *array, int *size)
 *        This function writes out "size" floating-point values stored in 
 *      "array" to the file specified by "file_handle".
 *
 *      pioreadreal (int *file_handle, float *array, int *size)
 *        This function reads in the next "size" floating-point values stored
 *      in "array" from the file specified by "file_handle".
 *
 *      pioreadint (int *file_handle, int *array, int *size)
 *        This function reads in the next "size" floating-point values stored 
 *      in "array" from the file specified by "file_handle".
 *
 *   On reaching the end of file or on encountering an error, any pio operation
 *   will convert "file_handle" to -1.
 *
 *   The same functions with an underscore appended to their names are also
 *   available, and have exactly the same functionality.
 *
 * FORTRAN USERS:
 *
 *   Subroutines with the following names are provided:
 *        PIOOPENREAD (FILEHANDLE, FILENAME)
 *        PIOOPENWRITE (FILEHANDLE, FILENAME)
 *        PIOCLOSE (FILEHANDLE)
 *        PIOWRITEREAL (FILEHANDLE, ARRAY, SIZE)
 *        PIOWRITEINT (FILEHANDLE, ARRAY, SIZE)
 *        PIOREADREAL (FILEHANDLE, ARRAY, SIZE)
 *        PIOREADINT (FILEHANDLE, ARRAY, SIZE)
 *
 *
 *****************************************************************************/
/*===========================================================================*/
/*===========================================================================*/
#include <stdio.h>

#include "pio.h"

typedef unsigned char byte;	/* C defines sizeof(char) to be 1. */
				/* So, this should be the same no matter */
				/* no matter what the architechture of the */
				/* machine you are using is. */

typedef enum { False = 0, True = 1 } boolean;
				/* Booleans used all over the place */


/***** File Table *****/

typedef struct {
  boolean used;			/* true if the entry is being used. */
  boolean read;		        /* true if the file was opened for reading */
  byte size_long;		/* size of long integer in bytes */
  FILE *fp;			/* file descriptor pointer */
  byte divisor;	                /* the divisor */
  unsigned int actual;	        /* 2^divisor */
} File;

static File fbuf[MAX_OPEN_FILES]; /* used for storing information about */
				  /* the file. */

#define assert(A)    do{if(!(A)){printf("Assertion FAILED: file %s, line %d\n",__FILE__,__LINE__);exit(1);}}while(0)
#define Assert(A,B)  do{if(!(A)){printf("Warning: "B"\n");return;}}while(0)

/*========================================================================
  All functions return "True" if they succeed, and "False" if they fail.
 ========================================================================*/

/*------------------------------------------------------------------------
 - 
 - Write out the header. This header contains the architechture information:
 -  0. a magic number.
 -  1. size of an integer on the machine.
 -  2. floating-point resolution. (divisor)
 -
 -      fd : The fbuf location of the file.
 ------------------------------------------------------------------------*/
static boolean create_header (int fd)
{
  byte b[10] = { 'r', 'm', 0, 12, 21, 'f', 'm', 't', 48, 29 };
  int i, j;

  if (fwrite (b, 1, 10, fbuf[fd].fp) != 10) return False;
  b[0] = sizeof(int);
  b[1] = DIVISOR;
  fbuf[fd].size_long = b[0];
  fbuf[fd].divisor = DIVISOR;
  if (fwrite(b, 1, 2, fbuf[fd].fp) != 2) return False;
  return True;
}


/*------------------------------------------------------------------------
 - Reads in the header into the file structure pointed to by "fd"
 ------------------------------------------------------------------------*/
static boolean read_header (int fd)
{
  byte c[10] = { 'r', 'm', 0, 12, 21, 'f', 'm', 't', 48, 29 };
  byte b[10];
  int i;

  assert ((fd >= 0) && (fd < MAX_OPEN_FILES));
  assert (fbuf[fd].used);
  if (fread (b, 1, 10, fbuf[fd].fp) != 10) return False;
  for (i=0; i < 10; i++)
    if (b[i] != c[i]) return False;
  if (fread (&fbuf[fd].size_long, 1, 1, fbuf[fd].fp) != 1) return False;
  if (fread (&fbuf[fd].divisor, 1, 1, fbuf[fd].fp) != 1) return False;
  return True;
}

/*------------------------------------------------------------------------
 - Returns the first free file entry. -1 if none exist.
 ------------------------------------------------------------------------*/
static int new_slot (void)
{
  int i;

  for (i=0; i < MAX_OPEN_FILES; i++)
    if (!fbuf[i].used)
      return i;
  return -1;
}

/*------------------------------------------------------------------------
  initializes the table. once.
  ------------------------------------------------------------------------*/
static void initialize_func (void)
{
  static int init = 0;
  int i;
  if (init) return;
  init = 1;
  for (i=0; i < MAX_OPEN_FILES; i++)
    fbuf[i].used = False;
  if (DIVISOR >= 8*sizeof(int)) {
    printf ("Please recompile the PIO package so that DIVISOR satisfies:\n");
    printf ("    2^DIVISOR <= max value of an integer.\n");
    exit (1);
  }
}

/*------------------------------------------------------------------------
  writes out a int array to an open file 
 ------------------------------------------------------------------------*/
static boolean writeint (int fd, int *buf, int size)
{
  byte *b;
  int i, j, ival;
  unsigned long div;
  int lsize;

  lsize = fbuf[fd].size_long + 1;

  if (!(b = (byte *)malloc(lsize*size))) return False;

  for (i=0; i < size; i++) {
    ival = buf[i];
    b[i*lsize] = (ival >= 0) ? 0 : 1;
    ival = (ival < 0)? -ival:ival;
    for (div = 1, j=0; j < fbuf[fd].size_long; j++, div *= 256)
      b[i*lsize+j+1] = (ival/div)%256;
  }
  if (fwrite (b, 1, lsize*size, fbuf[fd].fp) != lsize*size) {
    free (b);
    return False;
  }
  free (b);
  return True;
}

/*------------------------------------------------------------------------
  reads in an int array from an open file 
 ------------------------------------------------------------------------*/
static boolean readint (int fd, int *buf, int size)
{
  byte *b;
  int i, j, lsize;
  
  lsize = fbuf[fd].size_long+1;

  if (!(b = (byte *)malloc(lsize*size))) return False;
  if (fread (b, 1, lsize*size, fbuf[fd].fp) != lsize*size) {
    free (b);
    return False;
  }

  for (i=0; i < size; i++) {
    for (buf[i] = 0, j=0; j < fbuf[fd].size_long; j++) {
      buf[i] *= 256;
      buf[i] += b[i*lsize+lsize-j-1];
    }
    if (b[i*lsize]) buf[i] = -buf[i];
  }
  free (b);
  return True;
}

/*------------------------------------------------------------------------
  Writes out typecheck info. Simple minded checking for stupid mistakes.
 ------------------------------------------------------------------------*/
static boolean writetypeinfo (int fd, boolean real, int size)
{
  byte head[2] = { 'c', 197 };
  byte b;
  b = real;
  if (fwrite (head, 1, 2, fbuf[fd].fp) != 2) return False;
  if (fwrite (&b, 1, 1, fbuf[fd].fp) != 1) return False;
  return writeint (fd, &size, 1);
}

/*------------------------------------------------------------------------
  Verifies typecheck info. Simple minded checking for stupid mistakes.
 ------------------------------------------------------------------------*/
static boolean verifytypeinfo (int fd, boolean real, int size)
{
  byte head[2] = { 'c', 197 };
  byte file[3];
  int rsize;
  byte b;
  b = real;
  if (fread (file, 1, 3, fbuf[fd].fp) != 3) return False;
  if ((file[0] != head[0]) || (file[1] != head[1]) || (file[2] != b)) 
    return False;
  if (!readint (fd, &rsize, 1)) return False;
  if (size != rsize) return False;
  return True;
}

/*------------------------------------------------------------------------
  writes out a float array. Written as follows:
   SIGN  INTEGER PART, FLOATING_PART*(2^DIVISOR)
 ------------------------------------------------------------------------*/
static boolean writereal (int fd, float *buf, int size)
{
  byte *b;
  int i, j, ival;
  unsigned int floatpart;
  float f;
  unsigned long div;
  int lsize;

  lsize = fbuf[fd].size_long*2+1;
  if (!(b = (byte*)malloc(size*lsize))) return False;
  
  for (i=0; i < size; i++) {
    f = buf[i];
    ival = (int)f;
    b[i*lsize] = (f >= 0) ? 0 : 1;
    ival = (ival < 0)? -ival:ival;
    for (div = 1, j=0; j < fbuf[fd].size_long; j++, div *= 256)
      b[i*lsize+1+j] = (ival/div)%256;
    f = (f < 0) ? -f : f;
    floatpart = (f-(int)f)*fbuf[fd].actual;
    for (div = 1, j = 0; j < fbuf[fd].size_long; j++, div *= 256)
      b[i*lsize+j+1+fbuf[fd].size_long] = (floatpart/div)%256;
  }
  if (fwrite (b, 1, lsize*size, fbuf[fd].fp) != lsize*size) {
    free (b);
    return False;
  }
  free (b);
  return True;
}

static boolean readreal (int fd, float *buf, int size)
{
  byte *b;
  int i, j, ival;
  unsigned int floatpart;
  float f;
  unsigned long div;
  int lsize;

  lsize = fbuf[fd].size_long*2+1;
  if (!(b = (byte*)malloc(size*lsize))) return False;
  if (fread (b, 1, lsize*size, fbuf[fd].fp) != lsize*size) {
    free (b);
    return False;
  }
  
  for (i=0; i < size; i++) {
    for (ival = 0, j=0; j < fbuf[fd].size_long; j++) {
      ival *= 256;
      ival += b[i*lsize+1+(fbuf[fd].size_long-1-j)];
    }
    for (floatpart = 0, j=0; j < fbuf[fd].size_long; j++) {
      floatpart *= 256;
      floatpart += b[i*lsize+lsize-j-1];
    }
    buf[i] = ival + (float)floatpart/(float)fbuf[fd].actual;
    if (b[i*lsize]) buf[i] = -buf[i];
  }
  free (b);
  return True;
}

#include <fcntl.h>

/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
void pioopenread (int *file_handle, char *myfile, int *len)
{
  int i, j;
  FILE *fp;
  char *filename;
 
  filename = (char*)malloc(*len+1);
  if (!filename) {
    printf ("MALLOC FAILED. size=%d\n", *len+1);
    exit (1);
  }
  strncpy (filename, myfile, *len);
  for (i = *len-1; i >= 0 && filename[i] == ' '; i--) ;
  filename[i+1] = '\0';

  initialize_func();
  if ((i = new_slot ()) == -1) {
    printf ("Warning: pioopenread: Too many open files. Maximum allowed: %d\n",
	    MAX_OPEN_FILES);
    *file_handle = -1;
    return;
  }
  fbuf[i].used = True;
  if (!(fp = fopen(filename, "rb"))) {
    printf ("Warning: pioopenread: Could not open file [%s] for reading.\n", 
	    filename);
    *file_handle = -1;
    return;
  }
  fbuf[i].fp = fp;
  if (!read_header (i)) {
    printf ("Warning: pioopenread: File [%s] was not saved using pio.\n", 
	    filename);
    *file_handle = -1;
    return;
  }
  if (fbuf[i].divisor >= 8*sizeof(int)) {
    printf ("Warning: pioopenread: File [%s].\n", filename);
    printf ("\tFloating-point precision on this machine is less than\n");
    printf ("\tthat on the machine on which the file was created.\n");
    *file_handle = -1;
    return;
  }
  for (fbuf[i].actual = 1, j = fbuf[i].divisor; j > 0; j--, 
       fbuf[i].actual *= 2);
  *file_handle = i;
  free (filename);
  fbuf[i].read = True;
}
void pioopenread_ (int *x, char *y, int *z) { pioopenread (x,y,z); }

/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
void pioopenwrite (int *file_handle, char *myfile, int *len)
{
  int i, j;
  FILE *fp;
  char *filename;
 
  filename = (char*)malloc(*len+1);
  if (!filename) {
    printf ("MALLOC FAILED. size=%d\n", *len+1);
    exit (1);
  }
  strncpy (filename, myfile, *len);
  for (i = *len-1; i >= 0 && filename[i] == ' '; i--) ;
  filename[i+1] = '\0';

  initialize_func();
  if ((i = new_slot ()) == -1) {
    printf ("Warning: pioopenwrite: Too many open files. Maximum allowed: %d\n"
	    , MAX_OPEN_FILES);
    *file_handle = -1;
    return;
  }
  fbuf[i].used = True;
  if (!(fp = fopen(filename, "wb"))) {
    printf ("Warning: pioopenwrite: Could not open file [%s] for writing.\n", 
	    filename);
    *file_handle = -1;
    return;
  }
  fbuf[i].fp = fp;
  if (!create_header (i)) {
    printf ("Warning: pioopenwrite: Error writing to file [%s].\n", filename);
    *file_handle = -1;
    return;
  }
  for (fbuf[i].actual = 1, j = fbuf[i].divisor; j > 0; j--,
       fbuf[i].actual *= 2);
  fbuf[i].read = False;
  free (filename);
  *file_handle = i;
}
void pioopenwrite_ (int *x, char *y, int *z) { pioopenwrite(x,y,z); }

/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
void  pioclose (int *file_handle) 
{
  Assert (file_handle, "pioclose: Invalid file handle.");
  Assert ((*file_handle >= 0) && (*file_handle < MAX_OPEN_FILES), 
	  "pioclose: Invalid file handle.");
  Assert (fbuf[*file_handle].used, "pioclose: Invalid file handle.");
  fbuf[*file_handle].used = False;
  fclose (fbuf[*file_handle].fp);
}
void pioclose_ (int *x) { pioclose(x); }

/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
void piowritereal (int *file_handle, float *array, int *size) 
{ 
  Assert (file_handle, "piowritereal: Invalid file handle.");
  Assert ((*file_handle >= 0) && (*file_handle < MAX_OPEN_FILES), 
	  "piowritereal: Invalid file handle.");
  Assert (fbuf[*file_handle].used, "piowritereal: Invalid file handle.");
  Assert (fbuf[*file_handle].read == False, 
	  "piowritereal: Specified file was opened for reading.");
  initialize_func();
  if (!writetypeinfo (*file_handle, True, *size)) {
    printf ("piowritereal: Error writing to file.\n");
    *file_handle = -1;
    return;
  }
  if (!writereal (*file_handle, array, *size)) {
    printf ("piowritereal: Error writing to file.\n");
    *file_handle = -1;
    return;
  }
}
void piowritereal_ (int *x, float *y, int *z) { piowritereal(x,y,z); }

/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
void piowriteint (int *file_handle, int *array, int *size) 
{
  Assert (file_handle, "piowriteint: Invalid file handle.");
  Assert ((*file_handle >= 0) && (*file_handle < MAX_OPEN_FILES), 
	  "piowriteint: Invalid file handle.");
  Assert (fbuf[*file_handle].used, "piowriteint: Invalid file handle.");
  Assert (fbuf[*file_handle].read == False, 
	  "piowriteint: Specified file was opened for reading.");
  initialize_func ();
  if (!writetypeinfo (*file_handle, False, *size)) {
    printf ("piowriteint: Error writing to file.\n");
    *file_handle = -1;
    return;
  }
  if (!writeint (*file_handle, array, *size)) {
    printf ("piowriteint: Error writing to file.\n");
    *file_handle = -1;
    return;
  }
}
void piowriteint_ (int *x, int *y, int *z)  { piowriteint(x,y,z); }

/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
void pioreadreal (int *file_handle, float *array, int *size) 
{
  Assert (file_handle, "pioreadreal: Invalid file handle.");
  Assert ((*file_handle >= 0) && (*file_handle < MAX_OPEN_FILES), 
	  "pioreadreal: Invalid file handle.");
  Assert (fbuf[*file_handle].used, "pioreadreal: Invalid file handle.");
  Assert (fbuf[*file_handle].read == True, 
	  "pioreadreal: Specified file was opened for writing only.");
  initialize_func ();
  if (!verifytypeinfo (*file_handle, True, *size)) {
    *file_handle = -1;
    return;
  }
  if (!readreal (*file_handle, array, *size))
    *file_handle = -1;
}
void pioreadreal_ (int *x, float *y, int *z) { pioreadreal (x,y,z); }

/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
void pioreadint (int *file_handle, int *array, int *size) 
{
  Assert (file_handle, "pioreadint: Invalid file handle.");
  Assert ((*file_handle >= 0) && (*file_handle < MAX_OPEN_FILES), 
	  "pioreadint: Invalid file handle.");
  Assert (fbuf[*file_handle].used, "pioreadint: Invalid file handle.");
  Assert (fbuf[*file_handle].read == True, 
	  "pioreadint: Specified file was opened for writing only.");
  initialize_func ();
  if (!verifytypeinfo (*file_handle, False, *size)) {
    *file_handle = -1;
    return;
  }
  if (!readint (*file_handle, array, *size))
    *file_handle = -1;
}
void pioreadint_ (int *x, int *y, int *z) { pioreadint (x,y,z); }
