/******************************************************
* Class rational                                      *
*                                                     *
* Implements a class of rational numbers              *
* First version started September, 2004               *
* Authors:  Alejandro Pauly                           *
*           Brian E. Weinrich                         *
*                                                     *
*******************************************************/

#include <cstdlib>
#include <string>
#include <iostream>
#include <cmath>
#include "rational.h"

namespace stu{
	using namespace stu;
    using namespace std;
	
	/* macro used for debugging, first line should be commented out
	* when in production */
	#define debug(x) cout << x << endl;
	#ifndef debug
	#define debug(x)
	#endif
	
	//utility macros
	#define max(x,y) (x>y?x:y)
	
	//standard constructors
	
		rational::rational():ri(){}
	
	/*copy constructors, deeps copy, no shared data*/
		rational::rational(const rational&arg):ri(arg.ri){}
		rational::rational(const rationalImpl&arg):ri(arg){}
	
	//deserializing constructor
	rational::rational(unsigned char* data, uint size):ri(data,size){
	}

	
	/*prints directly to standard output, same as cout << rational*/
		void rational::print()
	{ ri.print(); }
	
		rational::rational(signed char nschar, signed char dschar) throw( ZeroDenominatorException )
			:ri(nschar,dschar){}
	
		rational::rational(unsigned char nuchar, unsigned char duchar) throw( ZeroDenominatorException )
			:ri(nuchar,duchar){}
	
		rational::rational(signed short int nssint, signed short int dssint) throw( ZeroDenominatorException )
			:ri(nssint,dssint){}
	
		rational::rational(unsigned short int nusint, unsigned short int dusint) throw( ZeroDenominatorException )
			:ri(nusint,dusint){}
	
		rational::rational(signed int nsint, signed int dsint) throw( ZeroDenominatorException )
			:ri(nsint,dsint){}
	
		rational::rational(unsigned int nuint, unsigned int duint) throw( ZeroDenominatorException )
			:ri(nuint,duint){}
	
		rational::rational(signed long nslong, signed long dslong) throw( ZeroDenominatorException )
			:ri(nslong,dslong){}
	
		rational::rational(unsigned long nulong, unsigned long dulong) throw( ZeroDenominatorException )
			:ri(nulong,dulong){}
	
		rational::rational(float ntheFloat, float dtheFloat) throw( ZeroDenominatorException )
			:ri(ntheFloat,dtheFloat){}
	
		rational::rational(double ntheDouble, double dtheDouble) throw( ZeroDenominatorException )
			:ri(ntheDouble,dtheDouble){}
	
		rational::rational(const string &nstr, const string &dstr) throw( ZeroDenominatorException )
			:ri(nstr,dstr){}
	
	// Assignment Operators
	/* assignment makes a deep copy, no shared data left */
		rational& rational::operator = (const rational &rhs)
	{
		ri=rhs.ri;
		return *this;
	}
	
	// Arithmetic operators '+', '-', '*', '/', '%', '^'
		rational rational::operator + (const rational &rhs) const
	{
		rational temp(ri+rhs.ri);
		return temp;
	} // Overload binary +
	
	
		rational rational::operator - (const rational &rhs) const
	{
		rational temp(ri-rhs.ri);
		return temp;
	} // Overload binary -
	
		rational rational::operator * (const rational &rhs) const
	{
		rational temp(ri*rhs.ri);
		return temp;
	} // Overload *
	
		rational rational::operator / (const rational &rhs) const  throw (DivisionByZeroException)
	{
		rational temp(ri/rhs.ri);
		return temp;
	} // Overload /
	
		rational rational:: operator ^ (const integer &rhs) const
	{
		rational temp(ri^rhs);
		return temp;
	} // Overload ^
	
	// Arithmetic assignment operators '+=', '-=', '*=', '/='
		rational& rational::operator += (const rational &rhs)
	{
		(*this)=(*this) + rhs;
		return *this;   
	}
	
		rational& rational::operator -= (const rational &rhs)
	{  
		(*this)=(*this) - rhs;
		return *this;   
	}
	
		rational& rational::operator *= (const rational &rhs)
	{
		(*this)=(*this) * rhs;
		return *this;   
	}
	
		rational& rational::operator /= (const rational &rhs)  throw (DivisionByZeroException)
	{
		(*this)=(*this) / rhs;
		return *this;   
	}
	
	// Unary operators '+', '-'
		rational rational::operator + () const
	{
		return (*this);
	}
	
		rational rational::operator - () const
	{
		return (*this).neg();
	}
	
	// Increment/decrement operators '++', '--'
		rational& rational::operator ++()
	{
		rational one(1,1);
		(*this)=(*this) + one;
		return (*this);
	} // prefix
	
		rational rational::operator ++(int)
	{
		rational temp(ri);
		rational one(1,1);
		(*this)=(*this) + one;
		return temp;
	} // postfix
	
		rational& rational::operator --()
	{
		rational one(1,1);
		(*this)=(*this) - one;
		return (*this);
	} // prefix
	
		rational rational::operator -- (int)
	{
		rational temp(ri);
		rational one(1,1);
		(*this)=(*this) - one;
		return temp;
	} // postfix
	
	// Comparison operators
		bool rational::operator == (const rational &rhs) const
	{
		return (ri==rhs.ri);
	}
	
		bool rational::operator != (const rational &rhs) const
	{
		return (ri!=rhs.ri);
	}
	
		bool rational::operator > (const rational &rhs) const
	{
		return (ri>rhs.ri);
	}
	
		bool rational::operator < (const rational &rhs) const
	{
		return (ri<rhs.ri);
	}
	
		bool rational::operator >= (const rational &rhs) const
	{
		return (ri>=rhs.ri);
	}
	
		bool rational::operator <=(const rational &rhs) const
	{
		return (ri<=rhs.ri);
	}
	
	// Stream input/output operators
	integer rational::toInteger() const //returns floor integer
	{
		return ri.toInteger();
	}	

	double rational::toDouble() const{
		return ri.toDouble();	
	}
			
	istream& operator >> (istream &istr, rational &rhs)
	{
		return istr >> rhs.ri;
	}         
	
	ostream& operator << (ostream &ostr, const rational &rhs)
	{
		ostr << rhs.ri;
		return ostr;
	}
	
	rational rational::abs() const
	{
		rational temp(ri.abs());
		return temp;
	} // absolute value
	
	rational rational::neg() const
	{
		rational temp(ri.neg());
		return temp;
	} // negate value
	
	rational rational::sqrt() const
	{
		return rational(ri.sqrt());
	} //square root
	rational rational::sin() const
	{
		return ri.sin();
	} //sinus
	rational rational::cos() const
	{
		return ri.cos();
	} //cosin
	rational rational::tan() const throw (DivisionByZeroException)
	{
		return ri.tan();
	} //tangent
	rational rational::cot() const throw (DivisionByZeroException)
	{
		return ri.cot();
	} //cotangent
	
	uint rational::serialize(unsigned char* data) const
	{
		return ri.serialize(data);
	}
	
	uint rational::serialSize() const
	{
		return ri.serialSize();
	}
}	
