/*****************************************************
* Class integerImpl                                     *
*                                                   *
* Implements a class of "arbitrarily long" integers *
*                                                   *
* Author: Markus Schneider                          *
*                                                   *
*****************************************************/

#ifndef INTEGERIMPL_H
#define INTEGERIMPL_H

#include <string>
#include <iostream>
#include <istream>
#include "stuVector.h"
#include "user_types.h"
#include "divisionbyzeroexception.h"


namespace stu{
	using namespace stu;
	//using namespace std;
	
	//integerImpl class interface declaration
	
	class integerImpl
	{
	private:
	stuVector<uint16> biginteger; //hold the big number, [n] position=1 for positive, 0 for negative
	void compress(); //eliminate "most significant" zeros
	
	public:
	// Standard constructors
	integerImpl(); // default constructor initializing to zero
	integerImpl(const integerImpl &arg); // copy constructor
	integerImpl(unsigned char* data, uint size); //deserializing constructor
	~integerImpl(); // destructor
	void print(); ///for debugging
	
	//actual definition allows us to by-pass user defined types
	
	integerImpl(signed char schar);
	integerImpl(unsigned char uchar);
	integerImpl(signed short int ssint);
	integerImpl(unsigned short int usint);
	integerImpl(signed int sint);
	integerImpl(unsigned int uint);
	integerImpl(signed long slong);
	integerImpl(unsigned long ulong);
	integerImpl(float theFloat);
	integerImpl(double theDouble);
	
	integerImpl(const std::string &str);
	
	// Special ructor for positioning a representation pointer of an
	// integerImpl object to the internal representation of another, already
	// existing integerImpl object
	
	integerImpl(const stuVector<uint16> & _biginteger);
	
	// Assignment operators
	
	integerImpl& operator = (const integerImpl &rhs);
	
	// Arithmetic operators '+', '-', '*', '/', '%', '^'
	
	integerImpl operator + (const integerImpl &rhs) const; 
	integerImpl operator - (const integerImpl &rhs) const;
	integerImpl operator * (const integerImpl &rhs) const;  
	integerImpl operator / (const integerImpl &rhs) const throw (DivisionByZeroException);
	integerImpl operator % (const integerImpl &rhs) const;
	integerImpl operator ^ (const integerImpl &rhs) const;
	
	// Arithmetic assignment operators '+=', '-=', '*=', '/='
	
	integerImpl &operator += (const integerImpl &rhs);
	integerImpl &operator -= (const integerImpl &rhs);
	integerImpl &operator *= (const integerImpl &rhs);
	integerImpl &operator /= (const integerImpl &rhs) throw (DivisionByZeroException);
			
	// Unary operators '+', '-'
	
	integerImpl operator + () const;
	integerImpl operator - () const;
	
	// Increment/decrement operators '++', '--'
	
	integerImpl& operator ++ ();    // prefix
	integerImpl& operator ++ (int); // postfix
	integerImpl& operator -- ();    // prefix
	integerImpl& operator -- (int); // postfix
	
	// Comparison operators
	
	bool operator == (const integerImpl &rhs) const;
	bool operator != (const integerImpl &rhs) const;
	bool operator >  (const integerImpl &rhs) const;
	bool operator <  (const integerImpl &rhs) const;
	bool operator >= (const integerImpl &rhs) const;
	bool operator <= (const integerImpl &rhs) const;
		
	// Other functions
	
	integerImpl gcd(const integerImpl &rhs) const; // greatest common divisor
	integerImpl lcm(const integerImpl &rhs) const; // least common multiple
	integerImpl abs() const; // absolute value
	integerImpl neg() const; // negate value
	
    //serializer
	uint serialize(unsigned char* data) const;
	uint serialSize() const; //returns number of bytes this object takes when serialized

	unsigned long asULong() const;


	// Stream input/output operators
	
	friend std::istream &operator >> (std::istream &istr, integerImpl &rhs);
	friend std::ostream &operator << (std::ostream &ostr, const integerImpl &rhs);
	
	};
}
#endif // integerImpl_h 
