
/******************************************************
* Class integerImpl                                       *
*                                                     *
* Implements a class of "arbitrarily long" integers   *
* First version finished in July, 2004                *
*                                                     *
* Authors:  Alejandro Pauly, Brian E. Weinrich        *
*                                                     *
*                                                     *
* Modified: 08/14/2004 (AP)                           *
*           Clean up code, wrote a few comments       *
*                                                     *
*           09/06/2004 (AP)                           *
*           -Remove "positive"(sign) property, put as *
*            part of stuVector in position n             *
*           -fixed unary +,- definitions              *
*                                                     *
*           09/15/2004 (AP)                           *
*           Some documentation done                   *
*                                                     *
*           09/17/2004 (BEW)                          *
*           Extensive documentation and               *
*           reformatting done                         *
*                                                     *
*******************************************************/

#include <cstdlib>
#include <string>
#include <iostream>
#include <cmath>
#include "user_types.h"
#include "integerImpl.h"
#include <limits.h>

namespace stu{
	using namespace stu;
	//using namespace std;

	/*
	* macro used for debugging, first line should be commented out
	* when in production
	*/
	//#define debug(x) cout << x << endl;
	#ifndef debug
	#define debug(x)
	#endif
	
	//utility macros
	#define BASEEXP 16
	#define max(x,y) (x>y?x:y)
	
	/*
	*  Eliminate "most significant" zeros    *
	*  necessary because algorithms can      *
	*  only estimate size of result          *
	*/
	void integerImpl::compress()
	{  
	/**************LOG*******************
	
	Purpose:  To eliminate trailing zeros in the most
				significant part of number.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	int i,d;
	i=biginteger.length()-2;
	d=0;
	while (biginteger[i] == 0 && i > 0) {
		i--;
		d++;
	} // while
		
	biginteger[i+1]=biginteger[biginteger.length()-1];  //save sign
	biginteger.resize(i+2,0); //one for index starting at 0, the other for the sign
	} // function compress
	
	/*default constructor, initializes to number 0*/
	integerImpl::integerImpl()
	{   
	/**************LOG*******************
	
	Purpose:  Default constructor to initialize integerImpl to 0.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	
	stuVector<uint16> bigInt(2);
	bigInt[0]=0;
	bigInt[1]=1; //its positive by default
	biginteger=bigInt;
	} // Default constructor
	
	/*copy constructor, deeps copy, no shared data*/
	integerImpl::integerImpl(const integerImpl&arg)
	{
	/**************LOG*******************
	
	Purpose:  Copy constructor to create copy of integerImpl.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	biginteger=arg.biginteger;
	} // Copy constructor
	
	
	
	
	//deserializing constructor
	integerImpl::integerImpl(unsigned char* data, uint size):biginteger((uint16*)data,(uint16)(size/sizeof(uint16)))
	//:biginteger(size/sizeof(uint16))
	{
		/**************LOG*******************
	
		Purpose:  Deserializes data into integer
	
		Created by:  Alejandro Pauly - October, 2006
		************************************/
/*		uint16 ele;
		for (uint i=0; i<size; i+=sizeof(uint16))
		{
			ele=*((uint16*)(data+i));
			biginteger[i/sizeof(uint16)]=ele;
		}*/
// 		std::cout << "This integer: [" << (uint16)data[0] << "," << (uint16)data[2] << "]" << std::endl;
// 		std::cout << "Stored as: [" << biginteger[0] << "," << biginteger[1] << "]" << std::endl;
	}
	
	
	
	/*destructor, delete repr pointer*/
	integerImpl::~integerImpl()
	{
	/**************LOG*******************
	
	Purpose:  Destructor to garbage collect unneeded integerImpl.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	//delete biginteger;
	} // Destructor
	
	/*prints directly to standard output, same as cout << integerImpl*/
	void integerImpl::print()
	{
	/**************LOG*******************
	
	Purpose:  To output integerImpl to standard output.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
		for (uint i=0; i< biginteger.length()-1; i++) /*** Mark changed "i" to uint type from int type on 10/26/2006 ***/
		std::cout << biginteger[i] << ".";
	std::cout << std::endl;
	} // function print
	
	integerImpl::integerImpl(signed char schar)
	{
	/**************LOG*******************
	
	Purpose:  Constructor for signed character values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(2);
	theInt[0] = schar;
	theInt[1] = (schar>=0?1:-1);
	biginteger = theInt;
	} // Signed char constructor
	
	/* 
	* Parametrized constructors start                                   *
	* for constructors with                                             *
	* numeric params                                                    *
	*   maxvalue(param)<base:                                           *
	*     one limb integerImpl is constructed                               *
	*   maxvalue(param>                                                 *
	*     initialized to two limbs, resized as necessary                *
	* non-numeric params (i.e.- string)                                 *
	*     string assumed to be base 10. Each digit is multiplied by     *
	*        10^(n-i) where n is the size of the string and i (1..n) is *
	*        the position of the base 10 digit in the string            *
	*/
	
	integerImpl::integerImpl(unsigned char uchar)
	{
	/**************LOG*******************
	
	Purpose:  Constructor for unsigned character values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(2);
	theInt[0] = uchar;
	theInt[1] = 1; //unsigned, always positive
	
	biginteger = theInt;
	} // Unsigned char constructor
	
	integerImpl::integerImpl(signed short int ssint)
	{
	/**************LOG*******************
	
	Purpose:  Constructor for signed short int values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(2);
	theInt[0] = ssint;
	theInt[1] = (ssint>=0?1:0);
	
	biginteger = theInt;
	} // Signed short int constructor
	
	integerImpl::integerImpl(unsigned short int usint)
	{
	/**************LOG*******************
	
	Purpose:  Constructor for unsigned short int values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(2);
	theInt[0]=usint;
	theInt[1]=1; //unsigned always positive
	
	biginteger = theInt;
	} // Unsigned short int constructor 
	
	integerImpl::integerImpl(signed int sint)
	{
	/**************LOG*******************
	
	Purpose:  Constructor for unsigned int values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(3,0);
	uint i=0; /*** Mark changed "i" to uint type from int type on 10/26/2006 ***/
	uint32 base=(uint32)pow(2.0,BASEEXP);
	int tmp = (int)fabs(sint);
	while (tmp>0) {
		theInt[i]=tmp % base;
		tmp=(signed int)floor((double)tmp / base);
		i++;
		if (i>=theInt.length()-1) //always leave space at the end for sign
			theInt.resize(i*2,0);
	} // while
	
	theInt[theInt.length()-1]=(sint>=0?1:0);
	
	biginteger = theInt;
	compress();
	} // Signed int constructor
	
	integerImpl::integerImpl(unsigned int unt)
	{
	/**************LOG*******************
	
	Purpose:  Constructor for unsigned int values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(3,0);
	uint i=0; /*** Mark changed "i" to uint type from int type on 10/26/2006 ***/
	uint32 base=(uint32)pow(2.0,BASEEXP); 
	while (unt>0) {
		theInt[i]=unt % base;
		unt=(unsigned int)floor((double)unt / base);
		i++;
		if (i>=theInt.length()-1) //always leave space at the end for sign
			theInt.resize(i*2,0);
	} // while
	
	theInt[theInt.length()-1]=1; //unsigned always positive
	
	biginteger = theInt;
	compress();
	} // Unsigned int constructor
	
	integerImpl::integerImpl(unsigned long longNumber)
	{  
	/**************LOG*******************
	
	Purpose:  Constructor for unsigned long values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(3,0);
	uint i=0; /*** Mark changed "first" and "z" to uint type from int type on 10/26/2006 ***/
	//std::cout << "Constructing " << longNumber << " here. " << std::endl;
	uint32 base=(uint32)pow(2.0,BASEEXP);
	while (longNumber>0) {
		theInt[i]=(unsigned long)longNumber % (unsigned long)base;
		//longNumber=(unsigned long)floor((double)longNumber / base);
		longNumber=(unsigned long)longNumber / (unsigned long)base;
		i++;
		if (i>=theInt.length()-1)//always leave space at the end for sign
			theInt.resize(i*2,0);
	} // while
	
	theInt[theInt.length()-1]=1; //unsigned always positive
	
	biginteger = theInt;
	compress();
	} // Unsigned long constructor
	
	integerImpl::integerImpl(signed long slong)
	{  
	/**************LOG*******************
	
	Purpose:  Constructor for signed long values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(3,0);
	uint i=0; /*** Mark changed "first" and "z" to uint type from int type on 10/26/2006 ***/
	uint32 base=(uint32)pow(2.0,BASEEXP);
	long tmp = (long)fabs(slong);
	while (tmp>0) {
		theInt[i]=tmp % base;
		slong=(signed long)floor((double)tmp / base);
		i++;
		if (i>=theInt.length()-1) //always leave space at the end for sign
		theInt.resize(i*2,0);
	} // while
	
	theInt[theInt.length()-1]=(slong>=0?1:0);
	
	biginteger = theInt;
	compress();
	} // Signed long constructor
	
	integerImpl::integerImpl(float theFloat)
	{   
	/**************LOG*******************
	
	Purpose:  Constructor for float values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(2,0);
	uint i=0; /*** Mark changed "first" and "z" to uint type from int type on 10/26/2006 ***/
	uint32 base=(uint32)pow(2.0,BASEEXP);
	float tmp = fabs(theFloat);
	while (tmp!=0) {
		theInt[i]=(uint32)tmp % base;
		tmp=(float)floor((double)tmp / base);
		i++;
		if (i>=theInt.length()-1)//always leave space at the end for sign
			theInt.resize(i*2,0);
	} // while
	
	theInt[theInt.length()-1]=(theFloat>=0?1:0);
	
	biginteger = theInt;
	compress();
	} // Float constructor
	
	integerImpl::integerImpl(double theDouble)
	{      
	/**************LOG*******************
	
	Purpose:  Constructor for double values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	stuVector<uint16> theInt(2,0);
	uint i=0; /*** Mark changed "first" and "z" to uint type from int type on 10/26/2006 ***/
	uint32 base=(uint32)pow(2.0,BASEEXP); //assuming baseexp constant
	double tmp = fabs(theDouble);
	while (tmp>0) {
		theInt[i]=(uint32)tmp % base;
		tmp=(double)floor((double)tmp / base);
		i++;
		if (i>=theInt.length())//always leave space at the end for sign
			theInt.resize(i*2);
	} // while
	
	theInt[theInt.length()-1]=(theDouble>=0?1:0);
	
	biginteger = theInt;
	compress();
	} // Double constructor
	
	integerImpl::integerImpl(const std::string &str)
	{
	/**************LOG*******************
	
	Purpose:  Constructor for string values.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
		
		Changes: C1. Alejandro Pauly - November 8, 2004
				-fixed to eliminate any leading zeros
	************************************/
	/*  
	Construct base 216=65536 stuVector<uint16> from string in base 10.
	*/
	bool positive;
	std::string strt;
	
	/*** Mark changed "first" and "z" to uint type from int type on 10/26/2006 ***/
	
	uint first; //signals position of first digit in case of sign present
	if (str[0]=='-') {
		positive=false;
		first=1;
	} // if
	else { 
		first=0;
		if (str[0]=='+') first=1;
		positive=true;
	} // else
	
	//C1. eliminate any leading zeros
	uint z;
	for (z=first; z<str.length() && str[z]==0; z++);
		
	if (z>0) //at least one zero, cut
		strt=str.substr(z);
	else
		strt=str.substr(first);
	
	uint32 exp; //exponent for the base 10 digit
	integerImpl* temp;
	integerImpl  ten((short int)10); //will be (pow to exp) * current digit = temp
	int i=strt.length()-1;
	
	integerImpl final((unsigned short int)(strt[i]-'1'+1)%10); //start with LEAST Sig digit, then accumulate
	i--;
	exp=1;
	while (i>=0) {
		if (strt[i]=='0'){
			i--;
			exp++;
		}else{
			temp=new integerImpl((short int)1);
			for (uint j=1; j<=exp; j++){  /*** Mark changed "j" to uint type from int type on 10/26/2006 ***/
				(*temp)=(*temp) * ten;
			} // for
			(*temp)=(*temp) * integerImpl((unsigned short int)((strt[i]-'1'+1)%10));
			final=final + (*temp);
			i--;
			exp++;
		}
	} // while
	
	biginteger = final.biginteger;
	if (positive)
		biginteger[biginteger.length()-1]=1;
	else
		biginteger[biginteger.length()-1]=0;
	compress();
	} // String constructor
	
	integerImpl::integerImpl(const stuVector<uint16> & _biginteger)
	{
	/**************LOG*******************
	
	Purpose:  Constructor for integerImpl record values.
	
		Created by:  Alejandro Pauly and Brian E. Weinrich - July, 2004
		Changes:
	************************************/
	biginteger=_biginteger;
	} // IntegerRepr constructor
	
	// Assignment Operator
	/* assignment makes a deep copy, no shared data left */
	integerImpl& integerImpl::operator = (const integerImpl &rhs)
	{
	/**************LOG*******************
	
	Purpose:  To assign a base 2^n number.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
		if (this!=&rhs) {
			//delete biginteger;
			biginteger=rhs.biginteger;
		} // if
		return *this;
	} // Overload =
	
	// Arithmetic operators '+', '-', '*', '/', '%', '^'
	
	integerImpl integerImpl::operator + (const integerImpl &rhs) const
	{ 
	/**************LOG*******************
	
	Purpose:  To add two base 2^n numbers.
	
		Created by:  Brian E. Weinrich and Alejandro Pauly
		Changes:  C1. 08/12/2004, Alejandro Pauly
						-correction of semantics errors
					
					C2. 09/07/2004, Alejandro Pauly
						-change of sign representation
	************************************/
	//
	// Algorithm following Knuth's in TAOCP.
	//
	uint32 j, carry; //C1. int -> uint32
	uint32 base, digit1, digit2; //C1. int64 -> uint32
	const integerImpl* big; //C1
	const integerImpl* small; //C1
		//C1. stuVector<unsigned int> sum(10, 0); -> (wrong size restriction on result)
	uint32 n=biginteger.length()-1;
	uint32 m=rhs.biginteger.length()-1;
	//
	// Cases of operands with different signs.
	//
	if (biginteger[n] != rhs.biginteger[m]) {
		if (biginteger[n]==1) {
			integerImpl rhstemp = rhs;
			rhstemp.biginteger[m] = 1; //change sign
			return ((*this) - rhstemp); 
		} // if
		else {
			integerImpl thistemp = (*this);
			thistemp.biginteger[n] = 1;
			thistemp= (thistemp - rhs);
	//C1. switch sign
			return thistemp.neg();
		} // else
	} // if
	//  
	// Cases of operands with the same sign.
	//
	else {
		if (n>m) { //C1. changes for implementation simplicity, original logic correct
			big=this;
			small=&rhs; 
		} // if
		else{
			small=this;
			big=&rhs;  
		} // else
		stuVector<uint16> sum((*big).biginteger);
		sum.resize(sum.length()+1,0);
		sum[sum.length()-1]=sum[sum.length()-2]; //move sign to end of stuVector
		sum[sum.length()-2]=0;
	//
	// Add digits for lower order digits.
	//
		base = (uint32)pow(2.0, (double)BASEEXP);
		carry = 0;
		for (j = 0; j < (*small).biginteger.length()-1; j++) {
			digit1 = (*big).biginteger[j];
			digit2 = (*small).biginteger[j];
			sum[j] = (digit1 + digit2 + carry) % base;
			carry = (digit1 + digit2 + carry) / base;
		} // for
	//
	// Left over digits from bigger operand.
	//
		if (rhs.biginteger.length() == biginteger.length())
			sum[(*small).biginteger.length()-1] = carry;
		else {
			for (j = (*small).biginteger.length()-1; j < (*big).biginteger.length()-1; j++){
	//C1. changed as wrong assumption made of "carry" being carried only to first digit
				sum[j] = ((*big).biginteger[j] + carry) % base;
				carry = ((*big).biginteger[j] + carry) / base;
			} // for
			sum[(*big).biginteger.length()-1]=carry;
		} // else
	//
	// Instantiate and return sum.
	//
		integerImpl sumrec(sum);
		sumrec.compress();
		return sumrec; 
	} // else same sign
	} // Overload binary +
	
	integerImpl integerImpl::operator - (const integerImpl &rhs) const
	{  
	/**************LOG*******************
	
	Purpose:  To subtract two base 2^n numbers.
	
		Created by: Brian E. Weinrich and Alejandro Pauly
		Changes:  C1. 08/12/2004, Alejandro Pauly
						-correction of semantics errors
					
					C2. 09/07/2004, Alejandro Pauly
						-change of sign representation
	************************************/
	//
	// Algorithm is a modification of Knuth's from TAOCP.
	//
	uint32 j, t; //C1. int -> uint32
	int32 carry;
	uint32 base;
	int32 digit1, digit2;
	const integerImpl* big;
	const integerImpl* small;
	bool secondbigger,bothnegative;
	//C1. stuVector<unsigned int> difference(10, 0); -> (wrong size restriction on result)
	uint32 n=biginteger.length()-1;
	uint32 m=rhs.biginteger.length()-1;
	//
	// Cases of operands with different signs.
	//
	if (biginteger[n]==1) {
		if (rhs.biginteger[m]==0){
			integerImpl rhstemp=rhs;
			rhstemp.biginteger[m] = 1;
			return ((*this) + rhstemp); 
		} // if
	} // if
		
	if (biginteger[n]==0){
		if (rhs.biginteger[m]==0) { //both negative
			integerImpl rhstemp=rhs;
			rhstemp.biginteger[m] = 1;
			return ((*this) + rhstemp);
		} // if
		else {
			integerImpl thistemp = (*this);
			thistemp.biginteger[n] = 1;
			thistemp=rhs + thistemp;
	//switch sign
			return thistemp.neg();
		} // else
	} // if
	//
	// Case when both operands are positive.
	//
	if (biginteger[n]==0) bothnegative=true; //never getting here
	else bothnegative=false; //always here
	
	if ((*this)>=rhs) { //C1. changes for implementation simplicity, original logic correct
		secondbigger=false;
		big=this;
		small=&rhs; 
	} // if
	else {
		secondbigger=true;
		small=this;
		big=&rhs;  
	} // else
	stuVector<uint16> dif((*big).biginteger.length(),0); 
	for (j=0; j<(*small).biginteger.length()-1; j++)
		dif[j]=(*small).biginteger[j];
	
	base = (uint32)pow(2.0, (double)BASEEXP);
	carry = 0;
	//C1. for loop completely reimplemented, original semantics partially incorrect
	for (j = 0; j < (*big).biginteger.length()-1; j++) {
		digit1 = (*big).biginteger[j];
		digit2 = dif[j];
		t=(digit1-digit2 + (carry+1) + (base-1)) % base;
		dif[j]=t;
		if ((digit1+carry)<digit2)
			carry=-1;
		else
			carry=0;
	} // for
	
	/*C1. removed loops taking care leftover digits, already
	taken care of in previous loop. Original loops were incorrect as
	an assumption was made that carry would not propagate.
	*/
	//
	// Instantiate and return difference.
	//
	if (secondbigger)
		dif[dif.length()-1] = 0;
	else
		dif[dif.length()-1] = 1;
	if (bothnegative)
		dif[dif.length()-1] = (dif[dif.length()-1]==1?0:1);
	integerImpl differrec(dif);
	differrec.compress();
	return differrec;
	} // Overload binary -
	
	integerImpl integerImpl::operator * (const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To multiply two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	//
	// Heavily based on Knuth's multiplication algorithm O(n2).
	//
	uint32 i,j;
	uint32 t,k,base,d;
	uint32 m,n;
	
	base=(uint32)pow(2.0,BASEEXP);
	//
	// Pointers set for easier implementation.
	//
	const stuVector<uint16> * u=&(biginteger);
	const stuVector<uint16> * v=&(rhs.biginteger);
	stuVector<uint16> * w; //result
	n=u->length()-1; //the length of this integerImpl(u)
	m=v->length()-1; //the length of the parameter(v)
	//
	//first take special case where one of the factors has only one limb.
	//
	if (n==1) {
		w=new stuVector<uint16> ((*v));
		w->resize(w->length()+1,0);
		(*w)[w->length()-1]=(*w)[w->length()-2]; //move sign to the end
		(*w)[w->length()-2]=0;
		d=(*u)[0];
		k=0;
	//
	//u*d
	//
		for (i=0; i<m; i++) {
			t=(*w)[i]*d+k;
			(*w)[i]=t % base;
			k=(uint32)floor((double)t / base);
		} // for
		(*w)[m]=k; //m==w.length()-2
	} // if 
	else if (m==1) {
		w=new stuVector<uint16> ((*u));
		w->resize(w->length()+1,0);
		(*w)[w->length()-1]=(*w)[w->length()-2]; //move sign to the end
		(*w)[w->length()-2]=0;
		d=(*v)[0];
		k=0;
	//
	//u*d
	//
		for (i=0; i<n; i++) {
			t=(*w)[i]*d+k;
			(*w)[i]=t % base;
			k=(uint32)floor((double)t / base);
		} // for
		(*w)[n]=k; //n==w.length()-2
	} // else if
	else {
		w=new stuVector<uint16> (m+n+1,0); //1 added for sign
	//
	//M1. Initialize.
	//
		j=0;
		k=0;
	
		while (j<m) { // M6. Loop on j
						//M2. Zero Multiplier?
			if ((*v)[j]==0){
				(*w)[j]=0;
			} // if
			else {
				i=0;
				k=0; //M3. intialize i
				do{ //M5. loop on i
					//M4. Multiply and Add
				t=(*u)[i]*(*v)[j]+(*w)[i+j]+k;
				(*w)[i+j]=t % base;
				k=(uint32)floor((double)t/base);
				i=i+1;
				} while (i<n);
				(*w)[j+n]=k;
			} // else
			j=j+1;
		} // while
	} // else
	//
	// Instantiate and return product.
	//  
	(*w)[w->length()-1]=(rhs.biginteger[m]== biginteger[n]?1:0);
	integerImpl productrec(*w);
	productrec.compress();
	return productrec;
	} // Overload *
	
	integerImpl integerImpl::operator / (const integerImpl &rhs) const throw (DivisionByZeroException)
	{ 
	/**************LOG*******************
	
	Purpose:  To divide two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	
		Modified by: Alejandro Pauly - November 29, 2005 (Added DivisionByZeroException)
	************************************/
	/*
	* Based on Knuth's with implementation tips taken from previous   *
	* Modula implementation by Andreas Rohrbach, Markus Schneider.    *
	*/
	int32 j;  /*** Mark changed "first" and "z" to uint type from int type on 10/26/2006 ***/
	uint32 i, qt,d,t,k,base;
	uint32 m,n,mn;
	integerImpl zero(0);
	//
	// Do tests for special situations.
	//
	if (rhs == zero)
			throw DivisionByZeroException();
	
	if (rhs.abs()>(*this).abs())
		return integerImpl(0);
		
	if (biginteger.length()==2){ //dividend of length one then divisor of length one too
		int sign=(biginteger[biginteger.length()-1]==rhs.biginteger[rhs.biginteger.length()-1]?1:0);    
		stuVector<uint16> q(2,0);
		q[0]=biginteger[0]/rhs.biginteger[0];
		q[1]=sign;
		integerImpl quotient(q);
		return quotient;
	}
	base=(uint32)pow(2.0,BASEEXP);
	//
	// Make copies to work on.
	//
	stuVector<uint16> u(biginteger); 
	stuVector<uint16> v(rhs.biginteger);
	mn=u.length()-1; //the length of this integerImpl(u)
	n=v.length()-1; //the length of the parameter(v)
	m=mn-n+1;
	
	stuVector<uint16> q(m+1,0); //used for result,add one for sign 
	//
	// We can assume dividend has at least 2 limbs, else would have been caught previously.
	//
	if (rhs.biginteger.length()==2) { //then proceed with following simplified algorithm
		uint32 rem=0; //to hold temporary remainder
		uint32 dividend=0;
		j=mn-1;
		k=1;
		while (j>=0) {
			dividend=dividend +  u[j];
			if (dividend>=v[0]){
				qt=(uint32)floor((double)dividend / v[0]);
				q[m-k]=qt;
				k++;
				rem=dividend % v[0];
				dividend=rem * base;
			} // if
			else {
				q[m-k]=0;
				k++;
				dividend=dividend * base;
			}
			j--;
		} // while
		if (k==m) { //resize quotient
			for (i=1; i<m; i++) {
				q[i-1]=q[i];
			} // for
			q.resize(mn,0); //add one for sign
		} // if
	} // if  end of simplified 1 limb divisor algorithm
	//D1.Normalize
	else {
		d=(uint32)floor((double)(base/(v[n-1]+1)));
		u.resize(mn+1,0);
		if (d==1)
			u[mn]=0;
		else {
			k=0;
			for (i=0; i<mn; i++) {
				t=u[i]*d+k;
				u[i]=t % base;
				k=(uint32)floor((double)t / base);
			} // for
			u[mn]=k;
	
			k=0;
			for (i=0; i<n; i++) {
				t=v[i]*d+k;
				v[i]=t % base;
				k=(uint32)floor((double)t / base);
			} // for
			if (k>0)
				debug("problem with normalization"); 
		} // else
		
	//D2. Initialize j
		j=mn-n;
	
		while (j>=0) {
	//D3. Calculate q
			if (u[j+n]==v[n-1])
				qt=base-1;
			else
				qt=(uint32)floor((double)(u[j+n]*base + u[j+n-1])/v[n-1]);
			
			bool qcorrect=false;
			uint32 v2,qt2;
			do {
				qt2=u[j+n]*base + u[j+n-1] - qt * v[n-1];
				v2=(n==1?0:v[n-2]);
				if (qt2>=base)
				qcorrect=true;
				else {
				qcorrect=((v2 * qt) <= (qt2 * base + u[j+n-2]));
				if (!qcorrect)
					qt--;
				} // else
			} while (!qcorrect);
				
	//D4. Multiply and Subtract
			uint32 carry,z,zlow,zhigh;
		
			if (qt==0)
				q[j]=0;
			else {
				carry=0;
			
				k=0;
				stuVector<uint16> vtemp(v);
	
				for (i=0; i<n; i++) {
				z=v[i] * qt + carry;
				zlow=z % base;
				zhigh=(uint32)((double)(z / base));          
				if (u[i+j]>=zlow) {
					u[i+j] = u[i+j] - zlow;
					carry=zhigh;
				} // if
				else {
					u[i+j]=base + u[i+j] - zlow;
					carry = zhigh + 1;
				} // else
				} // for
				if (carry<=u[j+n]) {
				carry=0;
				u[j+n]=u[j+n]-carry;
				} // if
				q[j]=qt;
	
	//D5. test remainder
				if (carry>0) { //its negative...add back
				q[j]--;
	// Add back
				k=0;
				carry=0;
				for (i=0; i<n; i++) {
					z=v[i] + u[i+j] + carry;
					carry = (uint32)floor((double)z / base);
					u[i+j]=z % base;
				} // for
				} // if
			} // else D4
			j--;
		} // while
	} // else D1
	//
	// Instantiate and return quotient.
	//    
	q[q.length()-1]=(rhs.biginteger[rhs.biginteger.length()-1]
						== biginteger[biginteger.length()-1]?1:0);
	integerImpl quotient(q);
	quotient.compress();
	return quotient;
	} // Overload /
	
	integerImpl integerImpl::operator % (const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To find the remainder of dividing two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	/* initial version for correctness, should be optimized */
	integerImpl q((signed short int)1);
	integerImpl p((signed short int)1);
	integerImpl m((signed short int)1);
	
	q=(*this).abs() / rhs.abs();
	p=rhs.abs() * q.abs();
	q=(*this).abs();
	p=p.abs();
	m=q - p;
	if ((biginteger[biginteger.length()-1]==0) || 
			(rhs.biginteger[rhs.biginteger.length()-1]==0))
		m.biginteger[m.biginteger.length()-1]=0;
	m.compress();
	return m;
	} // Overload %
	
	integerImpl integerImpl:: operator ^ (const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To perform exponentiation of base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	integerImpl temp((*this));
	integerImpl counter(rhs);
	
	while(counter>1){
		temp=temp * (*this);
		counter--;
	} // while
	return temp;
	} // Overload ^
	
	// Arithmetic assignment operators '+=', '-=', '*=', '/='
	/* these all make calls to above implementations, study idea of *
	* reimplementing for efficiency                                */
	
	integerImpl& integerImpl::operator += (const integerImpl &rhs)
	{
	/**************LOG*******************
	
	Purpose:  To add and assign two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
		Changes: C1. Alejandro Pauly - 10/16/2004
					added return statement
	************************************/
	(*this)=(*this) + rhs;
	return *this;
	} // Overload +=
	
	integerImpl& integerImpl::operator -= (const integerImpl &rhs)
	{  
	/**************LOG*******************
	
	Purpose:  To subtract and assign two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
		Changes: C1. Alejandro Pauly - 10/16/2004
					added return statement
	************************************/
	(*this)=(*this) - rhs;  
	return *this;   
	} // Overload -=
	
	integerImpl& integerImpl::operator *= (const integerImpl &rhs)
	{
	/**************LOG*******************
	
	Purpose:  To multiply and assign two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
		Changes: C1. Alejandro Pauly - 10/16/2004
					added return statement
	************************************/
	(*this)=(*this) * rhs;
	return *this;   
	} // Overload *=
	
	integerImpl& integerImpl::operator /= (const integerImpl &rhs) throw (DivisionByZeroException)
	{
	/**************LOG*******************
	
	Purpose:  To divide and assign two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
		Changes: C1. Alejandro Pauly - 10/16/2004
					added return statement
	Alejandro Pauly - November 29, 2005 (Added DivisionByZeroException)
	************************************/
	(*this)=(*this) / rhs;
	return *this;   
	} // Overload /=
	
	// Unary operators '+', '-'
	
	integerImpl integerImpl::operator + () const
	{
	/**************LOG*******************
	
	Purpose:  To apply unary plus to a base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	return (*this);
	} // Overload unary +
	
	integerImpl integerImpl::operator - () const
	{
	/**************LOG*******************
	
	Purpose:  To negate two base 2^n numbers.
	
		Created by:  Brian E. Weinrich - July, 2004
		Contributor: Alejandro Pauly
		Changes:
	************************************/
	return (*this).neg();
	} // Overload unary -
	
	// Increment/decrement operators '++', '--'
	integerImpl& integerImpl::operator ++()
	{
	/**************LOG*******************
	
	Purpose:  To increment a base 2^n number with prefix style.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	integerImpl one(1);
	(*this)=(*this) + one;
	return (*this);
	} // Overload prefix ++
	
	integerImpl& integerImpl::operator ++(int)
	{
	/**************LOG*******************
	
	Purpose:  To increment a two base 2^n number with postfix style.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	integerImpl * temp=new integerImpl(biginteger);
	integerImpl one(1);
	(*this)=(*this) + one;
	return *temp;
	} // Overload postfix ++
	
	integerImpl& integerImpl::operator --()
	{
	/**************LOG*******************
	
	Purpose:  To decrement a base 2^n number with prefix style.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	integerImpl one(1);
	(*this)=(*this) - one;
	return (*this);
	} // Overload prefix ++
	
	integerImpl& integerImpl::operator --(int)
	{
	/**************LOG*******************
	
	Purpose:  To decrement a base 2^n number with postfix style.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	integerImpl * temp=new integerImpl(biginteger);
	integerImpl one(1);
	(*this)=(*this) - one;
	return *temp;
	} // Overload postfix ++
	
	// Comparison operators
	/* Basic idea of comparison operators is to find first distinction *
	* between operands and analyze the distinction to determine the   *
	* what result it should yield                                     */
	bool integerImpl::operator ==(const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To test for equality of two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	if (rhs.biginteger.length()!=biginteger.length() ||
			rhs.biginteger[rhs.biginteger.length()-1]!=
			biginteger[biginteger.length()-1])
		return false;
	
	for (uint i=0; i<biginteger.length()-1; i++) /*** Mark changed "i" to uint type from int type on 10/26/2006 ***/
		if (rhs.biginteger[i]!=biginteger[i])
			return false;       
	//
	// Everything is the same.
	//
	return true;
	} // Overload ==
	
	bool integerImpl::operator !=(const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To test for inequality of two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	if (rhs.biginteger.length()!=biginteger.length() ||
			rhs.biginteger[rhs.biginteger.length()-1]!=
			biginteger[biginteger.length()-1])
		return true;
	
	for (uint i=0; i<biginteger.length()-1; i++) /*** Mark changed "i" to uint type from int type on 10/26/2006 ***/
		if (rhs.biginteger[i]!=biginteger[i])
			return true;
	//
	// Everything is the same.
	//
	return false;
	} // Overload !=
	
	bool integerImpl::operator >(const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To do greater than comparison of two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	if (rhs.biginteger[rhs.biginteger.length()-1]!=
			biginteger[biginteger.length()-1])
		return (biginteger[biginteger.length()-1]==1);
	
	if (rhs.biginteger.length()>biginteger.length())
		return (biginteger[biginteger.length()-1]!=1);
	
	if (rhs.biginteger.length()<biginteger.length())
		return (biginteger[biginteger.length()-1]==1);
	
	int i;
	bool ueq1=false,ueq2=false;
	for (i=biginteger.length()-2; i>=0; i--) {
		ueq1=(rhs.biginteger[i]!=biginteger[i]);
		if (ueq1) break;
	} // for
	if (i<0) return false;  //all digits are the same
	
	ueq1=(rhs.biginteger[i]>biginteger[i]);
	ueq2=(rhs.biginteger[i]<biginteger[i]);
	
	return  ((biginteger[biginteger.length()-1]==0 && ueq1) || 
				(biginteger[biginteger.length()-1]==1 && ueq2));
	} // Overload >
	
	bool integerImpl::operator <(const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To do less than comparison of two base 2^n numbers.
	
		Created by:  Brian E. Weinrich - July, 2004
		Changes:  Alejandro Pauly - method completely reimplemented.
	************************************/
	if (rhs.biginteger[rhs.biginteger.length()-1]!=
			biginteger[biginteger.length()-1])
		return (biginteger[biginteger.length()-1]!=1);
	
	if (rhs.biginteger.length()>biginteger.length())
		return (biginteger[biginteger.length()-1]==1);
	
	if (rhs.biginteger.length()<biginteger.length())
		return (biginteger[biginteger.length()-1]!=1);
	
	int i;
	bool ueq1=false,ueq2=false;
	for (i=biginteger.length()-2; i>=0; i--) {
		ueq1=(rhs.biginteger[i]!=biginteger[i]);
		if (ueq1) break;
	} // for
	
	if (i<0) return false; //all digits are the same
	
	ueq1=(rhs.biginteger[i]<biginteger[i]);
	ueq2=(rhs.biginteger[i]>biginteger[i]);
	
	return  ((biginteger[biginteger.length()-1]==0 && ueq1) || 
				(biginteger[biginteger.length()-1]==1 && ueq2));
	} // Overload <
	
	bool integerImpl::operator >=(const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To do greater than or equals comparison of two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	if (rhs.biginteger[rhs.biginteger.length()-1]!=
			biginteger[biginteger.length()-1])
		return (biginteger[biginteger.length()-1]==1);
	
	if (rhs.biginteger.length()>biginteger.length())
		return (biginteger[biginteger.length()-1]!=1);
	
	if (rhs.biginteger.length()<biginteger.length())
		return (biginteger[biginteger.length()-1]==1);
	
	int i;
	bool ueq1=false,ueq2=false;
	for (i=biginteger.length()-2; i>=0; i--) {
		ueq1=(rhs.biginteger[i]!=biginteger[i]);
		if (ueq1) break;       
	} // for
		
	if (i<0) //all digits are the same
		return true; //they are equal
			
	ueq1=(rhs.biginteger[i]>biginteger[i]);
	ueq2=(rhs.biginteger[i]<biginteger[i]);
		
	return  ((biginteger[biginteger.length()-1]==0 && ueq1) || 
				(biginteger[biginteger.length()-1]==1 && ueq2));
	} // Overload >=
	
	bool integerImpl::operator <=(const integerImpl &rhs) const
	{
	/**************LOG*******************
	
	Purpose:  To do less than or equals comparison of two base 2^n numbers.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	if (rhs.biginteger[rhs.biginteger.length()-1]!=
			biginteger[biginteger.length()-1])
		return (biginteger[biginteger.length()-1]!=1);
	
	if (rhs.biginteger.length()>biginteger.length())
		return (biginteger[biginteger.length()-1]==1);
	
	if (rhs.biginteger.length()<biginteger.length())
		return (biginteger[biginteger.length()-1]!=1);
	
	int i;
	bool ueq1=false,ueq2=false;
	for (i=biginteger.length()-2; i>=0; i--) {
		ueq1=(rhs.biginteger[i]!=biginteger[i]);
		if (ueq1) break;
	} // for
		
	if (i<0) //all digits are the same
		return true; //they are equal
			
	ueq1=(rhs.biginteger[i]<biginteger[i]);
	ueq2=(rhs.biginteger[i]>biginteger[i]);
		
	return  ((biginteger[biginteger.length()-1]==0 && ueq1) || 
		(biginteger[biginteger.length()-1]==1 && ueq2));
	} // Overload <=
	
	// Stream input/output operators
	/* currently there is a problem, semantics for input need to be studied */
	std::istream& operator >> (std::istream &istr, integerImpl &rhs)
	{
	/**************LOG*******************
	
	Purpose:  To provide for input in base 10 of a base 2^n number.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	char c=' ';
	bool first=true;
	std::string temp="";
	do {
		istr.get(c);
		if (c!='\n'){
			if (c=='0' || c=='1' || c=='2' || c=='3' || c=='4' || 
				c=='5' || c=='6' || c=='7' || c=='8' || c=='9' ||
				((c=='-' || c=='+') && first==true)){
				temp=temp + c;
				first=false;
			} // if
			else {
				std::cout << "ERROR: Invalid Input:" << c << ":" << std::endl;
				exit(0);
			} // else
		} // if
	} while (c!='\n');
	if (temp!="") rhs=integerImpl(temp);
	return istr;
	} // Overload >>
	
	/*overloaded output op just makes call to recursive function and adds the sign */
	std::ostream& operator << (std::ostream &ostr, const integerImpl &rhs)
	{
	/**************LOG*******************
	
	Purpose:  To provide for output in base 10 of a base 2^n number.
	
		Created by:  Alejandro Pauly and Brian E. Weinrich - July, 2004
		Changes:
	************************************/
		std::string minus="";
	if (rhs.biginteger[rhs.biginteger.length()-1]==0) {
		minus="-";
		integerImpl cpy(rhs.biginteger);
		cpy=cpy.neg(); //signed already taken care of
		return (ostr << minus << cpy);
	} // if
	//
	// Not negative, so just go straight into output digits.
	//
	integerImpl ten((signed int)10);
	std::string str="";
	if (rhs.biginteger.length()==2) {
		if (rhs.biginteger[0]==0)
			str="0";
		else {
			uint32 temp=rhs.biginteger[0];
			uint32 modulus;
			while (temp>0) {
				modulus=temp % 10;
				if (modulus==0)
				str="0" + str;
				else
				str=((char)(modulus + '1' - 1)) + str;
				temp=temp / 10;
			} // while
		} // else
		//cout << "Returning: " << str << endl;
		return (ostr << str);
	} // if
	else {
		integerImpl zero((signed int)0);
		integerImpl cpy(rhs.biginteger);
		integerImpl U((signed int)1);
		if (cpy<zero)
			cpy=cpy.neg(); //signed will be taken care of later
		while (cpy>zero) {
			U=cpy % ten;
			cpy=cpy / ten;
			return (ostr << cpy << U); //lines simulates "post-order" recursion
			//cout << "Size: " << U.biginteger.length() << " First Value: " << U.biginteger[0] << endl;
		} // if
		//else
		//   return ostr;
	} // else
	//
	// Should never reach
	//
	return ostr;
	} // Overload <<
	
	// Other functions
	integerImpl integerImpl::gcd(const integerImpl &rhs) const
	{  
	/**************LOG*******************
	
	Purpose:  To find the greatest common divisor a base 2^n number.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	/* 
	* Can be optimized for long integers,        *
	* algorithm to be used needs to be studied.  *
	*/
	integerImpl u(1),v(1),r(1);
	integerImpl zero(0);
	
	if ((*this).abs()>rhs.abs()) {
		u=(*this).abs();
		v=rhs.abs();
	} // if
	else {
		v=(*this).abs();
		u=rhs.abs();
	} // else
	
	while (v>zero){
		r=u%v;
		u=v;
		v=r;
	} ;
	return u;
	} // function gcd (greatest common divisor)
	
	integerImpl integerImpl::lcm(const integerImpl &rhs) const
	{ // Needs to be implemented
		return *this;
	} // least common multiple
	
	integerImpl integerImpl::abs() const
	{
	/**************LOG*******************
	
	Purpose:  To find the absolute value of a base 2^n number.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	integerImpl temp(biginteger);
	temp.biginteger[temp.biginteger.length()-1]=1;
	return temp;
	} // function abs (absolute value)
	
	integerImpl integerImpl::neg() const
	{
	/**************LOG*******************
	
	Purpose:  To perform negation.
	
		Created by:  Alejandro Pauly - July, 2004
		Contributor:  Brian E. Weinrich
	************************************/
	integerImpl temp(biginteger);
	temp.biginteger[temp.biginteger.length()-1]=
		(temp.biginteger[temp.biginteger.length()-1]==1?0:1);
	return temp;
	} // function neg (negate value)
	
	uint integerImpl::serialize(unsigned char* data) const
	{
		uint16 temp;
		unsigned char* tempc;
		
		for (uint i=0; i<biginteger.length(); i++){
			temp=biginteger[i];
			tempc=(unsigned char*)&temp;
			for (uint j=0; j<sizeof(uint16); j++)
				data[(i*sizeof(uint16)+j)]=tempc[j];
		}
		
		return (biginteger.length()* sizeof(uint16));
	}
	
	uint integerImpl::serialSize() const //returns number of bytes this object takes when serialized
	{
		return (biginteger.length()* sizeof(uint16));
	}
	
	
	unsigned long integerImpl::asULong() const{
		unsigned long temp=0;
		uint32 base=(uint32)pow(2.0,BASEEXP);
		for (uint i=0; i<biginteger.length()-1; i++){
			temp+=(biginteger[i]*(unsigned long)pow(base,i));
		}
		
		return temp;
	}


}
