#ifndef MSLOB_H
#define MSLOB_H

#include <iostream>
#include "SequenceIndex.h"
#include "imSlobstream.h"
#include "memSlob.h"
#include "Locator.h"
#include "LocatorTable.h"
#include "constants.h"

/** class definition for mSlob.
*/

using namespace std;

//const uint ADDRESS_SIZE=4; //already defined in SequenceIndex.h
	
class Locator;
class imSlobstream;
class memSlob;

class mSlob{
	public:
		friend class imSlobstream;
		friend class memSlob;
		friend class Locator;
		friend class LocatorNode;
		
		//Constructors & destructor
		mSlob(); /**< Simply initializes necessary structures to default values */
		mSlob(void* b); //construct mSlob given storage structure handle b (ie Blob, File)
		mSlob(const mSlob& s); //copy constructor
		~mSlob();/**< Default destructor, releases all references*/

		//return Locator to the root of the hierarchy of the mSLOB
		Locator locateGlobal(); 
		
		
		//Restructuring data
		void resequence();
		void resequence( const Locator& loc );
		uint fragmentation(); //gives the percentage of fragmentation in the mSlob. A fully sequenced mSlob should return 0
		
		//"batch" handling methods to somewhat increase performance and reduce fragmentation
		bool beginBatch();
		bool commitBatch();
		
		//move to private after debugging
		SequenceIndex* sinx;
		
	private:
		void* store;
		Locator *global;
		bool openBatch; //indicates whether a batch is already open.
		bool readonly; //indicates whether the supplied storage is read only or not
		
		//oci function wrappers
		bool openStore() const; //opens storage. attempts read/write first, and if unsuccesful goes to readonly
		uint appendData(uint dsize, unsigned char* buf, uint bufsize); //returns status returned by OCI call
		uint readData(unsigned char* buf, uint bufsize, uint adr); //where to read, how much, from where
		uint readLength();
		bool closeStore() const;
		
		//utility functions
		void writeSinx(); //append the sequence index to the underlying storage
		void resequence(const Locator& loc, void* newStore, uint currWritePos ); //recursive function
		Locator locateRef(const Locator& loc, uint idx);
		Locator DeReference(const Locator& loc);
		Locator insertPRIVATE(unsigned char data[], uint size, Locator& loc, uint idx, int oType=BASE_LEVEL); //insert object at position idx of object loc, returns locator to inserted base object
		bool weakRemove(Locator& loc, uint idx); //remove reference to object idx at level loc, update loc's offset count		
		bool strongRemove(Locator& loc, uint idx); //remove whole object and all its links
		void weakRemoveUtil(Locator & loc, Locator & obj); //locate obj or any of its subobjects in loc and remove
		
		memSlob* mem; //used if we are actually working with an in-mem mSlob
		
		
		//locate subobject idx of loc,
		Locator locate(const Locator& loc, uint idx); 
		
		//get stream for l. streams can only be retrieved for structures and objects below
		imSlobstream getStream(Locator& l); 
		
		//insert object at position idx of object loc, returns locator to inserted base object. loc must be a structure or below
		Locator insert(unsigned char data[], uint size, Locator& loc, uint idx); 
		
		//insert empty object into position idx of loc. to be used for insertion of non-base objects or collection objects
		Locator insert(Locator& loc, uint idx, int oType = OBJECT_LEVEL); 
		
		//insert reference to loc1 into spot idx of loc2, returns locator to inserted object. only works when loc1 and loc2 belong to the same global mslob
		Locator insert(const Locator& loc1, Locator& loc2, uint idx); 
		
		//remove item at index idx of loc. If loc is in secondary structure, then it simply removes the reference and not the object it references
		bool remove(Locator& loc, uint idx);
		
		//size of object at loc, in bytes
		uint length(const Locator& loc); 
		
		//number of subobjects of loc
		uint count(const Locator& loc); 
		
		
};
#endif


/*
Notes on mSlob handling:

1. Count of subobjects in object is located after "end" dummy of object.
2. When looking for data values: first look on "start" values of sequence index, then if the address is within a range. Last, look at value being at the end of a range which would mean it was substituted.
3. The start of the first (sub)object,is computed differently from the rest because there is no "previous" offset to compute it so it is computed by: start of object + (count * offset_size)L
4. Steps to create a Locator, where S is the start of the parent object P, E is the end, C is the count and idx is the index of the object to locate within P:
		
		NOTE: value: address x length -> byte[]
		
		a) start=value(START + (idx*ADDRESS_SIZE)L, ADDRESS_SIZE);
		b) end=value(START + ((idx+1)*ADDRESS_SIZE)L, ADDRESS_SIZE);
		c) count=value(end, (ADDRESS_SIZE)L)


*/
