#ifndef SEQUENCEINDEX_H
#define SEQUENCEINDEX_H

typedef unsigned int uint;
const uint ADDRESS_SIZE=sizeof(uint);

#include <cstdio>
#include <cstring>

using namespace std;

class SequenceIndex{

	public:
		SequenceIndex();
		
		/*constructor to build a seqidx object loaded from a sequence of bytes
			structured as an array where even index values are start of range and
			odd index values are end of range
		-size refers to the length of data (in bytes|unsigned char), so actual number of range indicators= size/offset_size
		*/
		SequenceIndex(unsigned char *data, uint size);
		
		//get the logical address offset bytes from the end of the sequence
		uint logicalFromBack(uint offset); 

		//get the logical address offset bytes from the start of the sequence
		uint logicalFromFront(uint offset); 
		
		//return the physical address of the byte (logically) located offset bytes after phaddr
		uint logicalAdd(uint phaddr, uint offset); 
		
		uint logicalSubtract(uint phaddr, uint offset); 
		
		/*return physical address of "next" byte. 
			given seqidx ranges [a-b[, [c-d[:
			nextPhysical(a) = a
			nextPhysical(b) = c
			nextPhysical(x) = x, where x somewhere in the a-b or c-d range 
		*/
		uint nextPhysical(uint ph);
		
		/*return physical address of "previous" byte. 
			given seqidx ranges [a-b[, [c-d[:
		prevPhysical(b) = b
		prevPhysical(c) = c|b
		nextPhysical(x) = x, where x somewhere in the a-b or c-d range 
		*/
		uint prevPhysical(uint ph);
			
		//insert at logstart, size number of bytes actually stored at phstart
		bool insert(uint logstart, uint phstart, uint size);
		
		//append size number of bytes actually stored at phstart
		bool append(uint phstart, uint size);
		
		//remove size number of bytes start at phstart, go logically
		bool remove(uint phstart, uint size); 
		
		//remove all logical references to data starting at phstart till phend.
		bool removeLogical(uint phstart, uint phend); 
		
		//replace whatever is starting at logstart, size number of bytes with what is now stored at phstart
		bool replace(uint logstart, uint phstart, uint size);
		
		//returns true if x is logically inside the range between phstart and phend
		bool inRange(uint x, uint phstart, uint phend);
		
		//sets bytes to the array of bytes representing Seqidx, returns size of byte array
		uint getBytes(unsigned char bytes[]); 
				
		//actual number of bytes between addresses (might be more than phend-phstart cause of shifting)
		uint byteCount(uint phstart, uint phend);
		
		uint byteCount(); //length of sinx in bytes, sum of differences of ranges
		
		uint rangeCount(); //return number of ranges in seqidx
		
		void debug();
	private:
		uint* ranges; //pairs are starts, odds are ends(exclusive ends), so actual range includes start to (end-1)
		uint length; //length of ranges array
		
};

#endif
