#include <oci.h>
#include "mSlob.h"
#include <stdexcept>

#define DEBUG(A,B) cout << A << ": " << B << endl;

/**	This mSlob implementation is specific to Oracle DBMS
	\author Alejandro Pauly, Mark McKenney
	\date 2006-02-17
	\version 0.3A - original SLOB implementation
 	\date 2006-03-05
	\version 0.4AM - Semi-stable version
    \date 2006-03-09
    \version 0.5AM - adding dummy byte idea
 	\date 2006-06-29
 	\version 0.51 - modifying to fit mSLOB
 	\date 2006-09-25
 	\version 0.52 - finalizing mSLOB functionality for Spal2D
 */

//using namespace oracle::occi;
using namespace std;

typedef struct OCILobWrapper{
	OCILobLocator* lob;
	OCISvcCtx*	cntxt;
	OCIError* errhp;
} OCILobWrapper;

unsigned char dummy[DUMMY_SIZE]={'X'};
int *COLLECTION_DUMMY;
int *OBJECT_DUMMY;
int *BASE_DUMMY;
int *REFERENCE_DUMMY;

void initializeDummies(){
	COLLECTION_DUMMY=new int;
	OBJECT_DUMMY=new int;
	BASE_DUMMY=new int;
	REFERENCE_DUMMY=new int;
	
	*COLLECTION_DUMMY=COLLECTION_LEVEL;
	*OBJECT_DUMMY=OBJECT_LEVEL;
	*BASE_DUMMY=BASE_LEVEL;
	*REFERENCE_DUMMY=REFERENCE_LEVEL;
}

static void checkerr(OCIError *errhp, sword status)
{
	text errbuf[512];
/*	ub4 buflen;*/
	sb4 errcode;
	string er;
	
	if (status == OCI_SUCCESS) return;

	switch (status)
	{
		case OCI_SUCCESS_WITH_INFO:
			cout << "Error - OCI_SUCCESS_WITH_INFO" << endl;
			OCIErrorGet ((dvoid *) errhp, (ub4) 1, (text *) NULL, &errcode,
						  errbuf, (ub4) sizeof(errbuf), (ub4) OCI_HTYPE_ERROR);
			er=(char*)errbuf;
			cout << "Error -" << er << endl;
			break;
		case OCI_NEED_DATA:
			cout << "Error - OCI_NEED_DATA" << endl;
			break;
		case OCI_NO_DATA:
			cout << "Error - OCI_NO_DATA" << endl;
			break;
		case OCI_ERROR:
			OCIErrorGet ((dvoid *) errhp, (ub4) 1, (text *) NULL, &errcode,
						  errbuf, (ub4) sizeof(errbuf), (ub4) OCI_HTYPE_ERROR);
			er=(char*)errbuf;
			cout << "Error -" << er << endl;
			break;
		case OCI_INVALID_HANDLE:
			cout << "Error - OCI_INVALID_HANDLE" << endl;
			break;
		case OCI_STILL_EXECUTING:
			cout << "Error - OCI_STILL_EXECUTING" << endl;
			break;
		case OCI_CONTINUE:
			cout << "Error - OCI_CONTINUE" << endl;
			break;
		default:
			cout << "Error - " << status << endl;
			break;
	}
}

bool mSlob::openStore() const{
	OCILobWrapper* blb=(OCILobWrapper*)store;
	boolean b;
	text errbuf[512];
/*	ub4 buflen;*/
	sb4 errcode;	
	OCILobIsOpen(blb->cntxt, blb->errhp, blb->lob, &b );
	
	if( !b ){ //not open, then open
		if (OCILobOpen(blb->cntxt, blb->errhp, blb->lob, OCI_LOB_READWRITE )!= OCI_SUCCESS)
			if (OCILobOpen(blb->cntxt, blb->errhp, blb->lob, OCI_LOB_READONLY )!= OCI_SUCCESS){
				OCIErrorGet (blb->errhp, (ub4) 1, (text *) NULL, &errcode,
							errbuf, (ub4) sizeof(errbuf), (ub4) OCI_HTYPE_ERROR);
			
				string er=(char*)errbuf;
				cout << "Error: " << er << endl;
				throw runtime_error("Unable to open apparently unopened BLOB.");
			}
			else
				return true;
		else
			return false; //not readonly, thus readwrite
	}
	//ambiguous return because if opened externally, at this point it may be unknown whether it is readonly or not.
	return readonly;
}

bool mSlob::closeStore() const {
	OCILobWrapper* blb=(OCILobWrapper*)store;
	boolean b;
	
	OCILobIsOpen(blb->cntxt, blb->errhp, blb->lob, &b );
	
	if( b ){ //open, then clse
		if (OCILobClose(blb->cntxt, blb->errhp, blb->lob)== OCI_SUCCESS)
			return true;
		else
			return false; //couldnt close
	}else
		return true; //already unopened
}

uint mSlob::appendData(uint dsize, unsigned char* buf, uint bufsize){ //returns number of byte written
	//assume store is open already
	OCILobWrapper* blb=(OCILobWrapper*)store;
	
	oraub8 bsize=bufsize;
	oraub8 dufsize=dsize;
	oraub8 x=0;
	
/*	cout << "BufSize: " << bufsize << " Dsize: " << static_cast<ub4>(dsize) << endl;*/
	checkerr(blb->errhp, OCILobWriteAppend2(blb->cntxt, blb->errhp, blb->lob, &dufsize, &x,  (dvoid*)buf, bsize, 
		OCI_ONE_PIECE, NULL, NULL, (ub2) 0, (ub1) SQLCS_IMPLICIT));
	
	dsize=static_cast<uint>(dufsize);
	
	return dsize;
}

uint mSlob::readData(unsigned char* buf, uint bufsize, uint adr){
	OCILobWrapper* blb=(OCILobWrapper*)store;

	oraub8 bsize=bufsize;
	
	uint x=0; //ignored parameter
	checkerr(blb->errhp, OCILobRead2(blb->cntxt, blb->errhp, blb->lob, &bsize, (oraub8*)&x, adr, (dvoid*)buf, bufsize, 
			 OCI_ONE_PIECE, NULL, NULL, 0, 0));
	
	return bufsize;
	/*NOTE: should check on the last two parameters or writeappend (0,0), not sure how they can behave*/
}

uint mSlob::readLength(){
	OCILobWrapper* blb=(OCILobWrapper*)store;
	text errbuf[512];
/*	ub4 buflen;*/
	sb4 errcode;
	
	uint blength=0;
	sword retval=OCILobGetLength2(blb->cntxt, blb->errhp, blb->lob, (oraub8*)&blength);
	
	if (retval!=OCI_SUCCESS && retval!=OCI_SUCCESS_WITH_INFO){
		OCIErrorGet (blb->errhp, (ub4) 1, (text *) NULL, &errcode,
					 errbuf, (ub4) sizeof(errbuf), (ub4) OCI_HTYPE_ERROR);
		
		string er=(char*)errbuf;
		cout << "Error: " << er << endl;
		throw runtime_error("Unable to read length of BLOB.");
	}
	
	return blength;
}

//default constructor to an empty mSLOB
mSlob::mSlob()
{
	initializeDummies();
	
	global = NULL;
	store = NULL;
	sinx = NULL;
	openBatch=false;
	mem=new memSlob(this);
		
}

//main constructor, given a pointer to a Blob
mSlob::mSlob(void* b):mem(NULL){
	
	initializeDummies();
	
	uint currPos; //saves information about current writing position
	
	openBatch=false;
	global = NULL;

	//store =(void*) new Blob(*((Blob*)b));
	store = (OCILobWrapper*)b;
	
	readonly=openStore();
	
	uint blength=readLength();
		
	//load sequence index
	
	if (blength==0) //then mSlob is completely empty
	{
		if (readonly) 
			throw runtime_error("Unable to open Blob for writing for initialization!");
			
		//cout << "Started from empty mSLOB" << endl;
		//initialize primary structure to empty
		
		//calculate address of end dummy for first object
/*		uint address=1+(4*DUMMY_SIZE)+(3*ADDRESS_SIZE);*/
		uint cnt=0;
		
 		currPos=1;
		
		//write initial dummy for whole mSLOB
/*		cout << "Will append: " << DUMMY_SIZE << endl;
		OCILobWrapper* blb=(OCILobWrapper*)store;
		ub4 toWrite=DUMMY_SIZE;
		checkerr(blb->errhp, OCILobWrite(blb->cntxt, blb->errhp, blb->lob, &toWrite, 1, (dvoid *) dummy,
				 (ub4) DUMMY_SIZE, OCI_ONE_PIECE, (dvoid *)0,
					(sb4 (*)(dvoid *, dvoid *, ub4 *, ub1 *)) 0,
					(ub2) 0, (ub1) SQLCS_IMPLICIT));
		cout << "Wrote something" << endl;*/
		appendData(DUMMY_SIZE,(unsigned char*)COLLECTION_DUMMY,DUMMY_SIZE);
		/*blb->writeChunk(DUMMY_SIZE,dummy,DUMMY_SIZE,currPos);*/
		currPos+=DUMMY_SIZE; 
		
		//append address of first object to the blob
/*		appendData(ADDRESS_SIZE,(unsigned char*)(&address),ADDRESS_SIZE);*/
		/*		blb->writeChunk(ADDRESS_SIZE,(unsigned char*)(&address),ADDRESS_SIZE,currPos);*/
/*		currPos+=ADDRESS_SIZE; */
		
		//append begin dummy for first object
/*		appendData(DUMMY_SIZE,(unsigned char*)OBJECT_DUMMY,DUMMY_SIZE);*/
		/*		blb->writeChunk(DUMMY_SIZE,dummy,DUMMY_SIZE,currPos);*/
/*		currPos+=DUMMY_SIZE;*/
		
		//calculate address of primary structure of first object
/*		address=1+(3*DUMMY_SIZE)+(2*ADDRESS_SIZE);*/
		
		//append address of primary structure for first object
/*		appendData(ADDRESS_SIZE,(unsigned char*)(&address),ADDRESS_SIZE);*/
		/*		blb->writeChunk(ADDRESS_SIZE,(unsigned char*)(&address),ADDRESS_SIZE,currPos);*/
/*		currPos+=ADDRESS_SIZE; */
		
		//append begin dummy for primary structure of first object
/*		appendData(DUMMY_SIZE,(unsigned char*)(new int(NONE_LEVEL)),DUMMY_SIZE);*/
		/*		blb->writeChunk(DUMMY_SIZE,dummy,DUMMY_SIZE,currPos);*/
/*		currPos+=DUMMY_SIZE;*/
		
		//append end dummy for primary structure of first object
/*		appendData(DUMMY_SIZE,dummy,DUMMY_SIZE);*/
		/*		blb->writeChunk(DUMMY_SIZE,dummy,DUMMY_SIZE,currPos);*/
/*		currPos+=DUMMY_SIZE;*/
		
		//append count of subobjects in primary struct (0)
/*		appendData(ADDRESS_SIZE, (unsigned char*)&cnt, ADDRESS_SIZE);*/
		/* 		blb->writeChunk(ADDRESS_SIZE, (unsigned char*)&cnt, ADDRESS_SIZE, currPos);*/
//  		currPos+=ADDRESS_SIZE;

		//append end dummy for first object
/*		appendData(DUMMY_SIZE,dummy,DUMMY_SIZE);*/
		/*		blb->writeChunk(DUMMY_SIZE,dummy,DUMMY_SIZE,currPos);*/
/*		currPos+=DUMMY_SIZE;*/
		
/*		cnt=1;*/
		//append count of structs in first object (1)
/*		appendData(ADDRESS_SIZE, (unsigned char*)&cnt, ADDRESS_SIZE);*/
		/*		blb->writeChunk(ADDRESS_SIZE, (unsigned char*)&cnt, ADDRESS_SIZE, currPos);*/
/*		currPos+=ADDRESS_SIZE;*/
		
		//append end dummy for whole mSLOB
		appendData(DUMMY_SIZE,dummy,DUMMY_SIZE);
		/*		blb->writeChunk(DUMMY_SIZE,dummy,DUMMY_SIZE,currPos);*/
		currPos+=DUMMY_SIZE;
		
		//append mSLOB object count of 0, initializing to a simple root collection node with no children
		cnt=0;
		appendData(ADDRESS_SIZE, (unsigned char*)&cnt, ADDRESS_SIZE);
		/*		blb->writeChunk(ADDRESS_SIZE, (unsigned char*)&cnt, ADDRESS_SIZE,currPos);*/
		currPos+=ADDRESS_SIZE;
		
		//create new sequence index, initialize to unfragmented
		sinx=new SequenceIndex();
		sinx->insert(1,1,currPos-1);
		
		//always write initial seqidx, no possible batch here.
		writeSinx(); 
				
		//blb->close();
	}
	else //non-empty mSLOB, retrieve sequence index
	{
		//cout << "Non-EMpty blob found!" << endl;
					
		//step 1: read size of seqidx
		unsigned char* buffer=new unsigned char[ADDRESS_SIZE];
		readData(buffer,ADDRESS_SIZE,(blength+1)-ADDRESS_SIZE);
/*		blb->read(ADDRESS_SIZE, buffer, ADDRESS_SIZE, (blength+1)-ADDRESS_SIZE);*/
		uint s=*((uint*)buffer);
		
		//be clean
		delete buffer;
		
		//step 2: read seqidx data
		buffer=new unsigned char[s];
		readData(buffer,s,((blength+1)-ADDRESS_SIZE)-s);
/*		blb->read(s, buffer, s, ((blength+1)-ADDRESS_SIZE)-s);*/
		sinx=new SequenceIndex(buffer,s);	
		
		//be clean
		delete buffer;
	}
}

//copy constructor
mSlob::mSlob(const mSlob& s)
{
	//create a copy of a mSlob
}

//default destructor
mSlob::~mSlob()
{
	//cerr << "Will destroy mSlob================================================" << endl;
	LocatorTable::unlocateGlobalObject( this );
	if (store!=NULL){
/*		OCILobWrapper* blb=(OCILobWrapper*)store; */
		
		//if there is a running batch commit to ensure seqidx is written.
		if (openBatch) commitBatch();
		
		//if the Blob is open, then close it
		closeStore();
	}
	
	if (mem!=NULL)
		delete mem;
	//cerr << "mSlob destroyed++++++++++++++++++++++++++++++++++++++++++++++++++++" << endl;
}

//build Locator for primary structure
Locator mSlob::locateGlobal() 
{
	//check if the global variable is already set, if it isnt then set it and then find object
	if (global==NULL)
	{
			
		if (mem!=NULL)
		{
			global = LocatorTable::locateGlobalObject( this );
			if( global->isEmpty( ) )
			{
				global->setmemLoc( mem->root );
				global->setObjectType( COLLECTION_LEVEL );
				global->setMySlob( this );
				global->setParent( NULL );
			}
// 			global=new Locator(mem->mslobs->next,-1,this,NULL); //skip head
			
		}
		else
		{
			//locate the locator to locate the mSlob
			global = LocatorTable::locateGlobalObject( this );
			//explicitly cast into Oracle type blob
			//Blob* blb=(Blob*)store;
			if( global->isEmpty( ) ) //the object had not been previouly located, so locate now
			{
				readonly=openStore();
				
				//check whether the blob was initialized to work as mSLOB
				if (readLength()==0){
					cout << "Error: Working on uninitialized mSLOB. " << endl;
					exit(1);
				}
				
				//sinx->debug();
				
				//find the start and end of the whole mSLOB, according to the seqidx
				uint phstart=sinx->logicalFromFront(0);
				uint phend=sinx->logicalFromBack(ADDRESS_SIZE+DUMMY_SIZE); //where does the last object end? right before the stored count (and the dummy)
					
				//read the count of substructures, located at the end of the whole mSLOB
				uint countloc=sinx->nextPhysical(sinx->logicalAdd(phend,DUMMY_SIZE));
				
				//cout << "PhStart: " << phstart << " PhEnd: " << phend << " CountLoc: " << countloc << endl;
				
				//Oracle BLOBs addressing starts at 1, should never go in here, but just in case dummies are empty
				if (countloc==0) countloc=1;
				
				//read the count located at countloc
				unsigned char* buffer=new unsigned char[ADDRESS_SIZE];
				//blb->read(ADDRESS_SIZE, buffer, ADDRESS_SIZE, countloc);
				readData(buffer,ADDRESS_SIZE,countloc);
				
				//store the count in "count", needed to create locator
				uint count=0;
				memcpy(&count,buffer,ADDRESS_SIZE);
				
				//be clean
				delete buffer;
				
				//construct locator into global
// 				global=new Locator(phstart,phend,count,0,-2,this,NULL);
				global->setStart( phstart );
				global->setEnd( phend );
				global->setElements( count );
				global->setMyaddress( 0 );
				global->setObjectType( COLLECTION_LEVEL ); //this is always the case, no need to read actual dummy to see type
				global->setMySlob( this );
				global->setParent( NULL );
			}
		}
	}
	return *global;
}


/*
locatePrivate is used to locate subobjects no matter the struct (primary or secondary).
If the resulting object is the base in a secondary, an extra call makes sure that the actual object locator it points to is returned
*/
Locator mSlob::locateRef(const Locator& loc, uint idx) //locates object from loc level
{
	//Blob* blb=(Blob*)store;
	
	if (mem!=NULL){
		return mem->locateRef(loc,idx);
	}
	
	readonly=openStore();
	
	uint phendaddress, phstartaddress, actualStart;
	uint start, end;
	
	//check index range
	if( idx < 0 || idx >= loc.getElements() ){
		throw runtime_error("Error, LocateRef with invalid index.");
	}
		
	
	//find out start of object
	if (idx==0) //then its the first object, start based on number of elements
	{
		uint datastart=sinx->logicalAdd(loc.getStart(),DUMMY_SIZE);
		start=sinx->logicalAdd(datastart,loc.getElements() * ADDRESS_SIZE);
		//cout << "Start: " << start << endl;
	}
	else
	{
		//start of this object is logically immediate to the end(which is its dummy start) of the previous object + its count (defined by previous address)
		phstartaddress=sinx->logicalAdd(sinx->logicalAdd(loc.getStart( ),DUMMY_SIZE),(idx-1) * ADDRESS_SIZE); //start of 4-byte address (supposedly uint)
		unsigned char* buffer=new unsigned char[ADDRESS_SIZE];
		//blb->read(ADDRESS_SIZE, buffer, ADDRESS_SIZE, phstartaddress);
		readData(buffer, ADDRESS_SIZE, phstartaddress);
		start=*((uint*)buffer);
		start=sinx->logicalAdd(start,ADDRESS_SIZE+DUMMY_SIZE);
		delete buffer;
	}
	//nextPhysical makes sure that position found is not at end of range, if it is, then moves to next physical address in the logical progression
	actualStart=sinx->nextPhysical(start);
/*	cout << "ActualStart: " << actualStart << endl;*/
	
	
	//find out end of object, start by finding location of its own offset
	phendaddress=sinx->logicalAdd(sinx->logicalAdd(loc.getStart( ),DUMMY_SIZE),idx * ADDRESS_SIZE);
	
	//now read offset
	unsigned char* buffer=new unsigned char[ADDRESS_SIZE];
	//blb->read(ADDRESS_SIZE, buffer, ADDRESS_SIZE, phendaddress);
	readData(buffer, ADDRESS_SIZE, phendaddress);
	end=*((uint*)buffer);
	
	uint actualEnd=sinx->nextPhysical(end);
/*	cout << "Actual End: " << actualEnd << endl;*/
	//find out count of object, start by finding location of count
	uint countloc=sinx->nextPhysical(sinx->logicalAdd(actualEnd,DUMMY_SIZE));
	
/*	cout << "Count Loc: " << countloc << endl;*/
	//now read the count
//	blb->read(ADDRESS_SIZE, buffer, ADDRESS_SIZE, countloc);
	readData(buffer, ADDRESS_SIZE, countloc);
	uint count=*((uint*)buffer);
	
/*	cout << "Its Count: " << count << " its located at: " << countloc << endl;*/
	//be clean
	delete buffer;
	
		//now read offset
	buffer=new unsigned char[DUMMY_SIZE];
	//blb->read(ADDRESS_SIZE, buffer, ADDRESS_SIZE, phendaddress);
	readData(buffer, DUMMY_SIZE, actualStart);
	int thisstruct=*((uint*)buffer);

	if (thisstruct==NONE_LEVEL)
		thisstruct=idx; //loading struct element
	
	delete buffer;
	
/*	//determine if locator is for struct or actual object
	if (loc.getObjectType( ) == OBJECT_LEVEL)
		thisstruct=idx; //its for struct
	else if (loc.getObjectType( ) == COLLECTION_LEVEL)
		thisstruct=OBJECT_LEVEL; //its for global
	else
		thisstruct=loc.getObjectType( ); //its for data object*/
	
	//construct and return locator for result
	Locator temp(actualStart,actualEnd,count,phendaddress,thisstruct,loc.getMySlob( ), loc);
	
/*	if (!openBatch){
		cout << "Locating " << idx << " of "; loc.debug();
		cout << "Resulted in "; temp.debug();
		sinx->debug();
	}*/
	
	return  temp;
}

//public locate, given a locator (object or struct), it generates a locator for its idx^th subobject
Locator mSlob::locate(const Locator& loc, uint idx) 
{
	if (mem!=NULL){
		return mem->locate(loc,idx);
	}
	
// 	Blob* blb=(Blob*)store;

	if (loc.getElements( ) <=idx ){ //invalid idx
		cout << "Locate error: Invalid idx. Not enough elements to reach idx provided." << endl;
		exit(1);
	}	
	
	readonly=openStore();
	
	//call private locate
	Locator temp(loc.locateRef(idx));
	
	return temp.DeReference();
	
	//now deal with locator to decide if further read needs to be done for base objects "located" through secondary structures
	 //changed to only go down if base object is of exact size needed to be reference to other object.
	//if (loc.getObjectType() >0 && sinx->byteCount(temp.getStart( ),temp.getEnd( ) )==DUMMY_SIZE+(ADDRESS_SIZE*3) && temp.getElements( ) ==0)	
//	if (loc.structid>0 && sinx->byteCount(temp.start,temp.end)>DUMMY_SIZE && temp.elements==0){
		//then we are dealing with base object through secondary structure. Create actual locator and return

}

Locator mSlob::DeReference(const Locator& l){
	if (mem!=NULL){
		return mem->DeReference(l);
	}
	else if (l.getObjectType() == REFERENCE_LEVEL){
		
		unsigned char addresses[ADDRESS_SIZE*3];
		unsigned char locatorData[ADDRESS_SIZE];
		
		/* construct the locator to the primary struct*/
		
		/* first read the start, end, and offset addresses of the object in the primary struct
		all three addresses are stored together as a base object of a secondary struct*/
		//uint dataRead=blb->read( ADDRESS_SIZE*3, addresses, ADDRESS_SIZE*3, sinx->logicalAdd(temp.start,DUMMY_SIZE) );
		readData(addresses, ADDRESS_SIZE*3, sinx->logicalAdd(l.getStart(),DUMMY_SIZE) );
			
		// get the start and end addresses from the array
		uint primaryStart = *(uint*)(addresses);
		
		//read objectType based on given start
		unsigned char* buffer=new unsigned char[DUMMY_SIZE];
		readData(buffer, DUMMY_SIZE, primaryStart);
		int thisstruct=*((uint*)buffer);
		delete buffer;
		
		uint primaryEnd = *(uint*)(addresses+ADDRESS_SIZE);
		uint myaddress = *(uint*)(addresses+ADDRESS_SIZE+ADDRESS_SIZE);
		
		// now construct all the locator data
		Locator dataLoc(primaryStart,primaryEnd,0,myaddress,thisstruct,const_cast<mSlob*>(l.getMySlob( )), l.getParent( ));  // used to store the locator to the primary index
		//read the count of the object being retrieved
		//dataRead=blb->read( ADDRESS_SIZE, locatorData, ADDRESS_SIZE, sinx->logicalAdd(primaryEnd, DUMMY_SIZE) );
		readData( locatorData, ADDRESS_SIZE, sinx->logicalAdd(primaryEnd, DUMMY_SIZE) );
		dataLoc.setElements(*(uint*)locatorData);
		
		return dataLoc; //return locator to "real" data in primary struct
	}else
		return l; //non-base object or base object through primary structure
}

imSlobstream mSlob::getStream(Locator& l)
{
	if (l.getObjectType( ) == COLLECTION_LEVEL || l.getObjectType( ) == OBJECT_LEVEL || l.getObjectType( ) == REFERENCE_LEVEL){
		throw runtime_error("Attempting to retrieve mSlobStreams for wrong object type.");
	}
	
	if (mem!=NULL){
		return mem->getStream(l);
	}

	readonly=openStore();
	return imSlobstream(this,l);
}


//method for inserting base objects only. required insertion in primary structure
Locator mSlob::insert(unsigned char data[], uint size, Locator& loc, uint idx) //insert object at position idx of object loc
{
	
/*	if (loc.getObjectType()<0){
		throw runtime_error("Cannot insert base data into non-object mSlob");
		exit(1);
	}*/
	
	if (mem!=NULL){
		return mem->insert(data,size,loc,idx);
	}

//the following restriction was removed to allow secondary structure objects to have associate information (such as attributes)
/*	if (loc.structid>0){
		cout << "Invalid StructID: Base objects should only be inserted in primary structure";
		exit(1);	
	}
	else{ //call private|generic insert*/
		//cout << "inserting base data of size " << size << endl;
		return insertPRIVATE(data,size,loc,idx,BASE_LEVEL);
// 	}
}
	
//private insert required to validate insertions before allowing them to occur
Locator mSlob::insertPRIVATE(unsigned char data[], uint size, Locator& loc, uint idx, int oType) //insert object at position idx of object loc	
{
	
// 	Blob* blb=(Blob*)store;
	
	readonly=openStore();

	if (readonly) 
		throw runtime_error("Unable to open Blob for writing!");
	
	//uint curPhy=blb->length()+1; //stores current physical location where new address will be stored, +4 where the new data will start
	uint curPhy=readLength()+1; //stores current physical location where new address will be stored, +4 where the new data will start
	uint currPos=curPhy; //currPos handles current position to write on
	
	uint address, addressloc, bdummy, edummy, countloc, countlocpar;
	
	uint oldcountloc=sinx->nextPhysical(sinx->logicalAdd(loc.getEnd(),DUMMY_SIZE));
// 	if (data!=NULL)
	addressloc=curPhy;
	bdummy=addressloc+ADDRESS_SIZE;
	edummy=bdummy+DUMMY_SIZE+size;
	countloc=edummy+DUMMY_SIZE;
	countlocpar=countloc+ADDRESS_SIZE;
	address=edummy; //address of end dummy
// 	else
// 		address=curPhy+ADDRESS_SIZE+DUMMY_SIZE;
/*	uint addressparent=ADDRESS_SIZE+ADDRESS_SIZE+size+ADDRESS_SIZE;*/
			
	uint writtenSize=0; //,emptycount=0;
	
	//append new address to the blob
/*	cout << "appending address: " << address << endl;*/
	//writtenSize = blb->writeChunk(ADDRESS_SIZE,(unsigned char*)(&address),ADDRESS_SIZE,currPos);
	writtenSize = appendData(ADDRESS_SIZE,(unsigned char*)(&address),ADDRESS_SIZE);
	currPos+=ADDRESS_SIZE;
	
	//cout << "dummy start, written so far: " << writtenSize << endl;
	//writtenSize += blb->writeChunk(DUMMY_SIZE,dummy,DUMMY_SIZE,currPos);
	int thisstruct=BASE_LEVEL;
	if (data!=NULL){
		if (oType==REFERENCE_LEVEL){
			writtenSize += appendData(DUMMY_SIZE,(unsigned char*)REFERENCE_DUMMY,DUMMY_SIZE);
			thisstruct=REFERENCE_LEVEL;
			//cout << "Inserted Reference" << endl;
		}
		else{ //inserting actual base data
			writtenSize += appendData(DUMMY_SIZE,(unsigned char*)BASE_DUMMY,DUMMY_SIZE);
			thisstruct=BASE_LEVEL;
		}
	}else {//writing empty object, internal call
		thisstruct=oType;
		if (loc.getObjectType() == OBJECT_LEVEL) //then adding struct
			thisstruct=idx;
		else if (loc.getObjectType() >= 0)//then adding empty subobject in data context
			thisstruct=loc.getObjectType();

		//value should be either OBJECT_LEVEL (default) or COLLECTION_LEVEL
		//cout << "This struct insert: " << thisstruct << " Otype: " << oType << endl;
		writtenSize += appendData(DUMMY_SIZE,(unsigned char*)&thisstruct,DUMMY_SIZE);
	}
	currPos+=DUMMY_SIZE;
	
	//append data to the blob
	//cout << "appending data of size " << size << endl;
	if (size>0){
		//writtenSize += blb->writeChunk(size, data, size, currPos);
		writtenSize += appendData(size, data, size);
		currPos+=size;
	}
	
	//append dummy end
	//cout << "dummy end, written so far: " << writtenSize << endl;
	//writtenSize += blb->writeChunk(DUMMY_SIZE,dummy,DUMMY_SIZE,currPos);
	writtenSize += appendData(DUMMY_SIZE,dummy,DUMMY_SIZE);
	currPos+=DUMMY_SIZE;
	
	//write zero count of new object, safe to assume its data or empty object
	uint zero=0;
	//cout << "zero count, written so far: " << writtenSize << endl;
	//writtenSize += blb->writeChunk(ADDRESS_SIZE, (unsigned char*)&zero, ADDRESS_SIZE, currPos);
	writtenSize += appendData(ADDRESS_SIZE, (unsigned char*)&zero, ADDRESS_SIZE);
	currPos+=ADDRESS_SIZE;
	
	//create Locator object to be returned. 
	//cout << "creating corresponding locator, written so far: " << writtenSize << endl;
		
	Locator returner(bdummy,address,0,curPhy,thisstruct,loc.getMySlob(),loc);		
	//returner.debug();	
	//write new count of parent object
	uint * newCount=new uint;
	*newCount = (loc.getElements()+1);
	//writtenSize += blb->writeChunk(ADDRESS_SIZE, (unsigned char*)newCount ,ADDRESS_SIZE, currPos);
	countlocpar=(readLength()+1);
/*	cout << "writing new count: " << *newCount << " at " << countlocpar << endl;*/
	writtenSize += appendData(ADDRESS_SIZE, (unsigned char*)newCount ,ADDRESS_SIZE);
	currPos+=ADDRESS_SIZE;
	
	delete newCount;
	
	/*
		now decide where the object and its offset should be logically located
		offset should be located at loc.start+((idx+1)*ADDRESS_SIZE)L 
		object should be located right after the logical end of previous object +4L (because of count).
		special case if idx==0. then object should start at loc.start + (loc.elements*offset_size)L
	*/
	
	//cout << "now deal with seqidx" << endl;
		
	if (idx==0) //offset first in loc, object to be inserted first in loc
	{
		if (loc.getElements()==0){ //its also the first subobject ever for this object
			//insert offset, dummy, data, dummy, and count
			sinx->insert(sinx->logicalAdd(loc.getStart(),DUMMY_SIZE),addressloc,ADDRESS_SIZE+DUMMY_SIZE+size+DUMMY_SIZE+ADDRESS_SIZE); 
		}
		else{
			//insert address
			sinx->insert(sinx->logicalAdd(loc.getStart(),DUMMY_SIZE),addressloc,ADDRESS_SIZE); 
			//insert data at end of offsets
			sinx->insert(sinx->logicalAdd(loc.getStart(),DUMMY_SIZE+((loc.getElements()+1)*ADDRESS_SIZE)),curPhy + ADDRESS_SIZE,DUMMY_SIZE+size+DUMMY_SIZE+ADDRESS_SIZE); 
		}
	}
	else if (idx>loc.getElements()) //then the index is invalid...error
	{
		cout << "Invalid IDX:" << idx << " : Exiting with offset count: " << loc.getElements() << endl;	
		exit(1);
	}
	else
	{
		//locate brother object that will be logically previous to the new
		Locator prev(loc.locateRef(idx-1)); 
		
		//insert address
		sinx->insert(sinx->logicalAdd(sinx->logicalAdd(loc.getStart(),DUMMY_SIZE),idx*ADDRESS_SIZE),curPhy,ADDRESS_SIZE); 
		//insert data at end of prev + bytes storing count (normally ADDRESS_SIZE)
		sinx->insert(sinx->logicalAdd(prev.getEnd(),DUMMY_SIZE+ADDRESS_SIZE),curPhy + ADDRESS_SIZE,DUMMY_SIZE+size+DUMMY_SIZE+ADDRESS_SIZE);
	}
			
	//setting parent count
	sinx->replace(oldcountloc,countlocpar, ADDRESS_SIZE); 
// 	cout << "Replaced:: "; sinx->debug();

	//update parent's count
	//loc.setElements(loc.getElements()+1);	
	
	//write seqidx into storage if no batch is open
	if (!openBatch){ 
		writeSinx(); 
		//cout << "wrote sinx" << endl;
	}
	
	// return whether everything that needed to be written was written, data, offset and count, plus updated count for parent
	//cout << "nice insertion man" << endl;
// 	blb->close();
	return returner;
}

Locator mSlob::insert(Locator& loc, uint idx, int oType) //insert empty object into position idx of loc. non-base object inserter
{	
	if (mem!=NULL){
/*		cout << "Nice mem insertion" << endl;*/
		return mem->insert(loc,idx,oType);
	}

/*	cout << "Insert with oType= " << oType << " Abs: " << abs(oType) << endl;*/
	return insertPRIVATE(NULL,0,loc,idx,oType);
}
	
Locator mSlob::insert(const Locator& loc1, Locator& loc2, uint idx) //insert object in loc1 into spot idx of loc2, returns loc1
{
	if (mem!=NULL){
		return mem->insert(loc1,loc2,idx);
	}

	//create "base" object that points to object in primary struct and then insert
	uint data[3];
	data[0]=loc1.getStart();
	data[1]=loc1.getEnd();
	data[2]=loc1.getMyAddress();
	return insertPRIVATE((unsigned char*)data,sizeof(uint)*3,loc2,idx,REFERENCE_LEVEL);
}

bool mSlob::remove(Locator& loc, uint idx){
	if (!weakRemove(loc,idx))
		return strongRemove(loc,idx);
	else{
// 		cout << "Weak Removed!" << endl;
		return true;
	}
}

bool mSlob::strongRemove(Locator& loc, uint idx)
{
	if (mem!=NULL){
		return mem->strongRemove(loc,idx);
	}
	
	if (loc.getObjectType()==REFERENCE_LEVEL || (loc.getObjectType()==BASE_LEVEL && loc.getParent().getObjectType()>0)){
		//cout << "Remove error: strongRemoves can be performed only on objects through primary structure. To remove references use weakRemove(Locator,uint)." << endl;
		return false;
	}else if (loc.getElements()<=idx){ //invalid idx
		//cout << "Remove error: Invalid idx. Not enough elements to reach idx provided." << endl;
		return false;
	}	
		
// 	cout << "Strong Removing!" << endl;

	uint writtenSize=0;
	
	//entails removing at the sequence index and fixing count of parent object
	//Blob* blb=(Blob*)store;
	
	readonly=openStore();

	if (readonly) 
		throw runtime_error("Unable to open Blob for writing!");
	
	//locate object to be removed
	Locator r(loc.locateRef(idx));

// 	cout << "Object to remove: "; r.debug();

//  	cout << "Locator of object to be removed: "; r.debug();
//  	cout << "Locator of parent: "; loc.debug();
// 	cout << "But its new count was already written: " << *newCount << endl;
	
	// 	
	//save data to remove counts
	uint countloc=sinx->nextPhysical(sinx->logicalAdd(r.getEnd(),DUMMY_SIZE)); 
	uint countlocparent=sinx->nextPhysical(sinx->logicalAdd(loc.getEnd(),DUMMY_SIZE));
		
	//now remove all references to the object to be deleted or any of its children
	//so first find object level parent of loc, in order to find all of its secondary structs
	Locator dad(loc.getParent());
	while (dad.getObjectType()!=OBJECT_LEVEL){
		dad=dad.getParent();
		if (dad.isEmpty()) 
			throw runtime_error("Reached the root, should not have.");
	}
	
	for (uint j=1; j<dad.getElements(); j++){ //for each secondary structure, sister to loc
		Locator structer(dad.locateRef(j)); //locator to secondary structure
// 		cout << "Searching through secondary: "; structer.debug();
		weakRemoveUtil(structer,r);	
	}
	
	//remove address
/*	cout << "Will remove address at " << r.getMyAddress() << endl;*/
	sinx->remove(r.getMyAddress(),ADDRESS_SIZE); 
/*	sinx->debug();*/
	
	//remove data and dummies
/*	cout << "Will remove data from " << r.getStart() << " to " << sinx->logicalAdd(r.getEnd(),DUMMY_SIZE-1) << endl;*/
	sinx->removeLogical(r.getStart(),sinx->logicalAdd(r.getEnd(),DUMMY_SIZE-1)); 
/*	sinx->debug();*/
	
	//remove count
/*	cout << "Will remove childs count at " << countloc << endl;*/
	sinx->remove(countloc,ADDRESS_SIZE);	
/*	sinx->debug();*/
	
	//write new count of parent object
	uint * newCount=new uint;
	*newCount = (loc.getElements()-1);
	//uint blblength=blb->length();
	uint blblength=readLength();
	//writtenSize += blb->writeChunk(ADDRESS_SIZE, (unsigned char*)newCount ,ADDRESS_SIZE, blblength+1);
	writtenSize += appendData(ADDRESS_SIZE, (unsigned char*)newCount ,ADDRESS_SIZE);
	delete newCount;
	
	//replace old with new count of parent
/*	cout << "Will replaces parents count from " << countlocparent << " to " << (blblength+1) << endl;	*/
	sinx->replace(countlocparent,blblength+1,ADDRESS_SIZE);
/*	sinx->debug();*/
	
	//loc.setElements(loc.getElements()-1);
	
	if (!openBatch) writeSinx(); //write seqidx into storage
	
	return true;
}

//locate obj or any of its subobjects in loc and remove
void mSlob::weakRemoveUtil(Locator & loc, Locator & obj){
	
// 	Blob* blb=(Blob*)store;
	//uint deleted=0; //keeps count of siblings remove to perform one single count update per parent
	
	openStore();

	if (readonly) 
		throw runtime_error("Unable to open Blob for writing.");
	
	uint i=0;
	while (i<loc.getElements()){
		Locator obji(loc.locateRef(i));
		
		if (obji.getElements()>0){ //if object not at bottom, keep looking
			weakRemoveUtil(obji,obj);
			i++;
		}
		else if (obji.getObjectType()==REFERENCE_LEVEL) { // obji is now a base object of secondary struct, check reference
			//read data of obji
/*			unsigned char addresses[ADDRESS_SIZE*3];*/
/*			unsigned char locatorData[ADDRESS_SIZE];*/
			
			/* first read the start, end, and offset addresses of the object in the primary struct
			all three addresses are stored together as a base object of a secondary struct*/
			//uint dataRead=blb->read( ADDRESS_SIZE*3, addresses, ADDRESS_SIZE*3, sinx->logicalAdd(obji.start,DUMMY_SIZE) );
/*			uint dataRead=*/
			uint primaryStart=0;
			readData( (unsigned char*)&primaryStart, ADDRESS_SIZE, sinx->logicalAdd(obji.getStart(),DUMMY_SIZE) );
// 			cout << "Is " << primaryStart << " within "; obj.debug();
// 			cout << "Referenced from: "; obji.debug();
			// get the start and end addresses from the array
/*			uint primaryStart = *(uint*)(addresses);*/
/*			uint primaryEnd = *(uint*)(addresses+ADDRESS_SIZE);
			uint myaddress = *(uint*)(addresses+ADDRESS_SIZE+ADDRESS_SIZE);		*/
			
			//now check to see if retrieved object needs to be deleted
			if (sinx->inRange(primaryStart,obj.getStart(),obj.getEnd())) { //then yes delete reference. if the end is not in there then the object is corrupted anyway
				/*manually remove obji*/
// 				cout << "Yes IN RANGE" << endl;
// 				cout << "Loc: "; loc.debug();
				
				if (!loc.remove(i))
					throw runtime_error("Why an error with weakRemoveUtil?");
				//parent count updated at the end!
// 				cout << "LocNow: "; loc.debug();
				
/*				//save count location to be removed
				uint countloc=sinx->nextPhysical(sinx->logicalAdd(obji.getEnd(),DUMMY_SIZE)); 

				//remove count
				sinx->remove(countloc,ADDRESS_SIZE);	

				//remove dummies and data
				sinx->removeLogical(obji.getStart(),sinx->logicalAdd(obji.getEnd(),DUMMY_SIZE)-1);
				deleted++; //another child has been removed
				loc.setElements(loc.getElements()-1);
				
				//delete objects own address reference
				sinx->remove(obji.getMyAddress(),ADDRESS_SIZE);				*/
				
			}else {i++;} //base object referencing non-deleted object, ignore
		}else {i++;} //its empty non-base object of secondary struct, no use for now
	}
// 	cout << "Done for loc" << endl;
	//update parents count based on all removed siblings
// 	if (deleted>0){
// 		uint * newCount=new uint;
// 		*newCount = loc.getElements(); //obj.elements has already been decreased to actual number
// 		cout << "Updating parent count to: " << *newCount << endl;
// //		uint blblength=blb->length();
// //		blb->writeChunk(ADDRESS_SIZE, (unsigned char*)newCount ,ADDRESS_SIZE, blblength+1);
// 		uint blblength=readLength();
// 		appendData(ADDRESS_SIZE, (unsigned char*)newCount ,ADDRESS_SIZE);
// 		uint countlocparent=sinx->nextPhysical(sinx->logicalAdd(loc.getEnd(),DUMMY_SIZE));
// 		cout << "Im gonna call it!!, you ready?" << endl;*/
// 		sinx->replace(countlocparent,blblength+1,ADDRESS_SIZE);
// 		cout << "Kazzaam!" << endl;
// 		cout << "The updated parent: "; loc.debug();
// 	}else
// 		cout << "No, no deletions..." << endl;
}

bool mSlob::weakRemove(Locator& loc, uint idx){
	
	if (mem!=NULL){
		return mem->weakRemove(loc,idx);
	}

	
	if (loc.getObjectType()==0 || loc.getObjectType()==COLLECTION_LEVEL || loc.getObjectType()==OBJECT_LEVEL || (loc.getObjectType()==BASE_LEVEL && loc.getParent().getObjectType()==0)){
		//cout << "Remove error: weakRemoves can be performed only on objects in secondary structures. To remove objects from primary structure use strongRemove." << endl;
		return false;
	}else if (loc.getElements()<=idx){ //invalid idx
		//cout << "Remove error: Invalid idx. Not enough elements to reach idx provided." << endl;
		return false;
	}	
		
// 	cout << "Weak Removing!" << endl;
	//entails removing at the sequence index and fixing count of parent object
// 	Blob* blb=(Blob*)store;
	
	readonly=openStore();

	if (readonly) 
		throw runtime_error("Unable to open Blob for writing!");
		
	//locate object to be removed
	Locator r(loc.locateRef(idx)); //use locatePrivate as to not remove actual object, but just its reference
	
/*	cout << "Object to remove: "; r.debug();*/
	
	//write new count of parent object
	uint * newCount=new uint;
	//loc.setElements(loc.getElements()-1);
	*newCount =loc.getElements()-1;
/*	uint blblength=blb->length();
	blb->writeChunk(ADDRESS_SIZE, (unsigned char*)newCount ,ADDRESS_SIZE, blblength+1);*/
	uint blblength=readLength();
	appendData(ADDRESS_SIZE, (unsigned char*)newCount ,ADDRESS_SIZE);
	
	delete newCount;
	//save data to remove counts
	uint countloc=sinx->nextPhysical(sinx->logicalAdd(r.getEnd(),DUMMY_SIZE)); 
	uint countlocparent=sinx->nextPhysical(sinx->logicalAdd(loc.getEnd(),DUMMY_SIZE));
	
	//remove address
/*	cout << "Will remove address at " << r.getMyAddress() << endl;*/
	sinx->remove(r.getMyAddress(),ADDRESS_SIZE); 
/*	sinx->debug();*/
		
	//remove data and dummies
/*	cout << "Will remove data from " << r.getStart() << " to " << sinx->logicalAdd(r.getEnd(),DUMMY_SIZE-1) << endl;*/
	sinx->removeLogical(r.getStart(),sinx->logicalAdd(r.getEnd(),DUMMY_SIZE-1)); 
/*	sinx->debug();*/
	
	//remove child's count
/*	cout << "Will remove childs count at " << countloc << endl;*/
	sinx->remove(countloc,ADDRESS_SIZE);	
/*	sinx->debug();*/
	
	//replace old with new count of parent
/*	cout << "Will replaces parents count from " << countlocparent << " to " << (blblength+1) << endl;*/
	sinx->replace(countlocparent,blblength+1,ADDRESS_SIZE);
	
	if (!openBatch) writeSinx(); //write seqidx into storage
		//remove address	
	
	return true;
}

void mSlob::resequence()
{
// 	if (mem!=NULL){
// 		mem->resequence();
// 		return; 
// 	}
// 
// 	
// 	Blob* blb=(Blob*)store;
// 
// 	readonly=openStore();
// 
// 	if (readonly) 
// 		cerr << "Unable to open Blob for writing!" << endl;
// 	
// 	//resequence processing
// 	
// 	// create the new Blob object that will have the fully sequenced mSlob
// 	Blob * newBlob = new Blob();
// 	
// 	// recursively resequuence each object within the mSlob
// 	// begin the general mSlob locator
// 	resequence(locatemSlob(), (void*)newBlob, 0);
// 	
// 	//write the sequence index
// 	SequenceIndex *newsinx=new SequenceIndex();
// 	newsinx->insert(1,1,newBlob->length());
// 	
// 	delete sinx;
// 	
// 	sinx=newsinx;
// 	
// 	// delete the old Blob object and set the pointer to the new sequenced blob
// 	blb->setNull();
// 	delete blb;
// 	store = (void*)newBlob;
// 	if( !blb->isOpen( ) )
// 		blb->open();
// 	writeSinx();
// 	
// 	blb->close();
}

void mSlob::resequence(const Locator& loc, void* newStore, uint currWritePos )
{
// 	Blob* blb=(Blob*)store;
// 	Blob* newBlob=(Blob*)newStore;
// 	uint len;	
// 	
// 	// if the object is a base level object, simply copy its data into the new blob
// 	if( loc.elements == 0 &&  length( loc ) > 0 )
// 	{
// 		// get the length of the object
// 		len = length( loc ) + ADDRESS_SIZE;
// 		
// 		// read the data from the blob
// 		unsigned char data[ len ];
// 		blb->read( len - ADDRESS_SIZE, data, len, loc.start );
// 		
// 		//zero out the area left for the offset size
// 		for( int i = len-1; i >= len - ADDRESS_SIZE; i-- )
// 			data[i] &= 0x0;
// 		
// 		// write the data to the new blob at the current write position
// 		newBlob->write( len, data, len, currWritePos );
// 		return;
// 	}
// 	
// 	// else the object is not a base type, so we resequence its elements, then we recursively
// 	// resequence its subobjects!
// 	
// 	//if the object is empty, then it has no elements or data, so we can simply insert
// 	// a zero object count into the new blob;
// 	if( length( loc) == 0 )
// 	{
// 		unsigned char data[ ADDRESS_SIZE ];
// 		for( int i = 0; i < ADDRESS_SIZE; i++ )
// 			data[i] &= 0x0;
// 		
// 		newBlob->write( ADDRESS_SIZE, data, ADDRESS_SIZE, currWritePos );
// 		return;
// 	}
// 	
// 	// if the object is not empty, then first copy over its elements, then add the objcet count, 
// 	// then recursively add each object
// 	
// 	uint start =loc.start;
// 	uint offset, newOffset, newObjectWritePos;
// 	uint size;
// 	uint dataStart = loc.elements * ADDRESS_SIZE + start;
// 	uint prevOffset = loc.elements * ADDRESS_SIZE;
// 	unsigned char* offsetData;
// 		
// 	for( int i = 0; i < loc.elements; i++ )
// 	{
// 		// get the phys address of the current offset
// 		start = sinx->logicalAdd( start, i * ADDRESS_SIZE );
// 		
// 		// read the offset from the DB
// 		blb->read( ADDRESS_SIZE, offsetData, ADDRESS_SIZE, start );
// 		offset = *((uint *) offsetData); 
// 		
// 		// calculate the size of the object using the previous offset and number of bytes between the start of the object and 
// 		// the value of its offset
// 		size = sinx->byteCount( start, offset ); 
// 		newOffset = size + prevOffset + ADDRESS_SIZE;  // new offset of object is size of object+ prev offset, plus room for subject size
// 		newObjectWritePos = prevOffset + currWritePos;  //since currWritePos has been incremented, we do not ned to add ADDRESS_SIZE
// 														//in order to make room for the num of subobjects
// 		
// 		// set the previous offset to the just calculated offset for the next loop iteration
// 		prevOffset = newOffset;
// 		
// 		// write the new offset to the new blob
// 		offsetData = (unsigned char *) &newOffset;
// 		newBlob->write( ADDRESS_SIZE, offsetData, ADDRESS_SIZE, currWritePos );
// 		currWritePos += ADDRESS_SIZE;
// 		
// 		// recursively write the actual object to the new blob
// 		resequence(locate( loc, i ), newStore, newObjectWritePos );
// 	}
// 	
// 	// write the num of subObjects to the end of the object
// 	offsetData = (unsigned char *) &loc.elements;
// 	newBlob->write( ADDRESS_SIZE, offsetData, ADDRESS_SIZE, prevOffset+currWritePos );
// 	
// 	//now the object is resequenced for happily ever after...or until an update occurs
	
}

uint mSlob::length(const Locator& loc)
{
	if (mem!=NULL){
		return mem->length(loc);
	}
		
	//actual length of the object in bytes, can be taken from sequence index
	uint datastart=sinx->logicalAdd(loc.getStart(),DUMMY_SIZE);

	if (loc.getElements()==0 && datastart!=loc.getEnd()) //its a base object
		return sinx->byteCount(datastart,loc.getEnd()); //does this make sense?
	else
		return sinx->byteCount(datastart,loc.getEnd())-1; //remove last byte counted that belongs to dummy
}

uint mSlob::count(const Locator& loc) //number of subobjects of loc
{
	if (mem!=NULL){
		return mem->count(loc);
	}
	
	return loc.getElements();
}

uint  mSlob::fragmentation() //gives the percentage of fragmentation in the mSlob. A fully sequenced mSlob should return 0
{
	if (mem!=NULL){
		return mem->fragmentation();
	}
	
	openStore();
	
//	return 100-((length(*global)*100)/blb->length());
	return 100-((length(*global)*100)/readLength());
}

bool mSlob::beginBatch(){
	
	if (mem!=NULL){
		return mem->beginBatch();
	}
	
	if (openBatch) return false;
	openBatch=true;
	return true;
}

bool mSlob::commitBatch(){
	
	if (mem!=NULL){
		return mem->commitBatch();
	}
	
	if (openBatch){
		if (!readonly) writeSinx();
		openBatch=false;
		return true;
	}else
		return false;
}

void mSlob::writeSinx()
{
	/*Blob* blb=(Blob*)store*/;
	readonly=openStore();

	if (readonly) 
		throw runtime_error("Unable to open Blob for writing!");
	
	uint* idxsize=new uint;
	unsigned char seqidx[(sinx->rangeCount()*2)*ADDRESS_SIZE];
	
//	uint blength=blb->length()+1;
/*	uint blength=readLength()+1;*/
		
	*idxsize=sinx->getBytes(seqidx);
		
	//blb->writeChunk(*idxsize, seqidx, *idxsize, blength); //write the sequence index
	appendData(*idxsize, seqidx, *idxsize); //write the sequence index
		
	//blb->writeChunk(ADDRESS_SIZE, (unsigned char*)idxsize, ADDRESS_SIZE, blength + *idxsize); //write the sequence index size
	appendData(ADDRESS_SIZE, (unsigned char*)idxsize, ADDRESS_SIZE); //write the sequence index size
	
//	DEBUG("SINX POS",blength);
//	DEBUG("SINX SIZE",*idxsize);
	//sinx->debug();

	delete idxsize;
	//delete seqidx; ///why comment it? its giving segfault, but WHY???
}
