/*
 * generateMultiple.cpp
 *
 *  Created on: Feb 16, 2013
 *      Author: dung
 */

#include "generateMultiple.h"
#include <vector>
#include "Random.h"
#include "Graph.h"
#include <stdio.h>
#include <set>
#include "coupling.h"

vector<int> random_combination(int k, vector<int> source) {
	vector<int> indices = rand_comb(k, source.size());
	vector<int> result(k);
	for (int i = 0; i < k; i++) {
		result[i] = source[indices[i]];
	}

	return result;
}

Graph* ER_generate(vector<int> ids, float p) {
	Graph *newGraph = new Graph();
	Graph &g = *newGraph;

	// Add vertex ids
	for (int i = 0; i < ids.size(); i++) {
		g.add_vertex(ids[i], 1);
	}

	// Add edge
	for (int i = 0; i < ids.size(); i++) {
		for (int j = 0; j < ids.size(); j++) {
			if (i == j) continue;
			double rd = random(0.0, 1.0);
			//add new edge
			if (rd <= p) {
				g.add_edge(ids[i], ids[j], random(0.0, 1.0));
			}
		}
	}

	// Readjust the threshold
	vector<int> vs = g.vertex_ids();
	for (int i = 0; i < vs.size(); i++) {
		double total = 0;
		vector<Edge*> inedges = g.in_edges(vs[i]);
		for (int j = 0; j < inedges.size(); j++){
			total += inedges[j]->weight;
		}
		g[vs[i]].threshold = random(0.0, total);
		if (g[vs[i]].threshold == 0) g[vs[i]].threshold = 1;
	}

	return newGraph;
}

Graph* ER_generate(int n, float p) {
	Graph *newGraph = new Graph();
	Graph &g = *newGraph;
	int d = 2* n * p;
	Random *rd = new Random();
	// Add vertex ids
	for (int i = 0; i < n; i++) {
		g.add_vertex(i, 1);
	}

	// Add edge
	for (int i = 0; i < n; i++) {
		int deg = rd->genInt(0, d);
		for (int k = 0; k < deg; k++) {
			int j = rd->genInt(0, n - 1);
			g.add_edge(i, j, rd->genDouble(0.0, 1.0/n));
		}
	}

	cout << "done1" << endl;
	// Readjust the threshold
	vector<int> vs = g.vertex_ids();
	for (int i = 0; i < vs.size(); i++) {
		double total = 0;
		vector<Edge*> inedges = g.in_edges(vs[i]);
		for (int j = 0; j < inedges.size(); j++){
			total += inedges[j]->weight;
		}
		g[vs[i]].threshold = random(0.0, total);
		if (g[vs[i]].threshold == 0) g[vs[i]].threshold = 1;
	}

	delete rd;
	return newGraph;
}

Graph* ER_lossy(vector<Graph*> gs, int usize, map<int, float> (*f)(Graph &)) {
	vector<int> user(usize);
	int k = gs.size();

	for (int i = 0; i < usize; i++) user[i] = i;

	//lossless couple networks
	Graph *coupledGraph = new Graph();
	Graph &cg = *coupledGraph;

	for (int i = 0; i < usize; i++) {
		cg.add_vertex(i, 0);
	}


	// Import edge weight and vertex threshold of each network
	for (int n = 0; n < k; n++) {
		map<int, float> ea = f(*gs[n]);

		Graph &g = *gs[n];

		// Change the threshold of nodes
		vector<int> ids = gs[n]->vertex_ids();
		for (int i = 0; i < ids.size(); i++) {
			cg[ids[i]].threshold = cg[ids[i]].threshold + ea[ids[i]] * g[ids[i]].threshold;
		}

		// add edges
		vector<Edge*> es = g.edge_set();
		for (int i = 0; i < es.size(); i++) {
			Edge e = *es[i];
			if (cg.is_edge(e.source, e.dest)) {
				cg(e.source, e.dest).weight = cg(e.source, e.dest).weight + ea[e.dest] *  e.weight;
			} else {
				cg.add_edge(e.source, e.dest, ea[e.dest] * e.weight);
			}
		}
	}

	for (int i = 0; i < usize; i++) {
		if (cg[i].threshold == 0) cg[i].threshold = 1;
	}

	return coupledGraph;
}

map<int, float> easiness_(Graph &g) {
	vector<int> ids = g.vertex_ids();
	map<int, float> ea;
	int n = g.order();

	for (int i = 0; i < ids.size(); i++) {
		ea[ids[i]] = 0;
	}


	for (int i = 0; i < n; i++) {
		int u = ids[i];

		vector<Edge*> in_edges = g.in_edges(u);
		double total = 0;
		for (int i = 0; i < in_edges.size(); i++) {
			total += in_edges[i]->weight;
		}
		ea[u] = total/(g[u].threshold * g[u].threshold);
	}

	return ea;
}

map<int, float> average_(Graph &g) {
	vector<int> ids = g.vertex_ids();

	map<int, float> avg;
	int n = g.order();

	for (int i = 0; i < ids.size(); i++) {
		avg[ids[i]] = 1.0/g[ids[i]].threshold;
	}

	return avg;
}

map<int, float> involvement_(Graph &g) {
	vector<int> ids = g.vertex_ids();

	map<int, float> ea;
	map<int, int> isNei;

	for (int i = 0; i < ids.size(); i++) {
		ea[ids[i]] = 0;
		isNei[ids[i]] = -1;
	}

	for (int i = 0; i < ids.size(); i++) {
		int u = ids[i];
		set<int> nei;
		nei.insert(u);
		isNei[u] = u;
		vector<int> in_nei = g.in_neighbor_ids(u);
		for (int i = 0; i < in_nei.size(); i++) {
			isNei[in_nei[i]] = u;
			nei.insert(in_nei[i]);
		}

		vector<int> out_nei = g.out_neighbor_ids(u);
		for (int i = 0; i < out_nei.size(); i++) {
			isNei[out_nei[i]] = u;
			nei.insert(out_nei[i]);
		}

		double total = 0;

		for (set<int>::iterator it = nei.begin(); it != nei.end(); it++) {
			int v = *it;
			vector<Edge*> in_edges = g.in_edges(v);

			for (int i = 0; i < in_edges.size(); i++) {
				Edge e = *in_edges[i];
				if (isNei[e.source] == u && isNei[e.dest] == u && g[e.dest].threshold > 0){
					total += e.weight/g[e.dest].threshold;
				}
			}
		}

		ea[u] = total/g[u].threshold;
	}
	return ea;
}

void generate(int k, float p, int gsize, int usize, string name) {
	vector<Graph*> gs;
	vector<int> user(usize, 0);

	for (int i = 0; i < usize; i++) user[i] = i;

	for (int i = 1; i <= k; i++) {
		vector<int> ids = random_combination(gsize, user);
		Graph *g = ER_generate(ids, p);
		gs.push_back(g);
		char gv[40], ge[40], losslessv[40], losslesse[40], easinessv[40], easinesse[40], avgv[40], avge[40], involv[40], invole[40];

		sprintf(gv, "%sv%d.txt", name.c_str(), i);
		sprintf(ge, "%se%d.txt", name.c_str(), i);
		sprintf(losslessv, "%s_losslessv%d.txt", name.c_str(), i);
		sprintf(losslesse, "%s_losslesse%d.txt", name.c_str(), i);
		sprintf(easinessv, "%s_easinessv%d.txt", name.c_str(), i);
		sprintf(easinesse, "%s_easinesse%d.txt", name.c_str(), i);
		sprintf(avgv, "%s_avgv%d.txt", name.c_str(), i);
		sprintf(avge, "%s_avge%d.txt", name.c_str(), i);
		sprintf(involv, "%s_involvv%d.txt", name.c_str(), i);
		sprintf(invole, "%s_involve%d.txt", name.c_str(), i);

		g->print2file(gv, ge);

		Graph *g1 = clique_lossless(gs, usize);
		g1->print2file(losslessv, losslesse);
		delete(g1);

		Graph *g2 = ER_lossy(gs, usize, easiness_);
		g2->print2file(easinessv, easinesse);
		delete(g2);

		Graph *g3 = ER_lossy(gs, usize, average_);
		g3->print2file(avgv, avge);
		delete(g3);

		Graph *g4 = ER_lossy(gs, usize, involvement_);
		g4->print2file(involv, invole);
		delete(g4);
	}

	for (int i = 0; i < gs.size(); i++) delete gs[i];
}

