/*
 * coupling.cpp
 *
 *  Created on: Nov 26, 2012
 *      Author: dung
 */

#include "coupling.h"
#include <iostream>
#include <fstream>
#include <map>
#include <set>

double random(double max) {
	return ((double)rand()/(double)RAND_MAX) * max;
}

Graph* clique_scheme(Graph& g1, Graph &g2, int ov) {
	int n1 = g1.order();
	int n2 = g2.order();
	int n = n1 + n2 - ov; // number of users

	Graph *gr = new Graph();
	Graph &g = *gr;
	for (int i = 0; i < 3*n; i++) {
		g.add_vertex(i, 1);
	}

	for (int i = 0; i < ov; i++) {
		g[3*i + 1].threshold = g1[i].threshold;
		g[3*i + 2].threshold = g2[i].threshold;
	}

	//vertices in network 1
	for (int i = ov; i < n1; i++) {
		g[3*i + 1].threshold = g1[i].threshold;
	}

	for (int i = ov; i < n2; i++) {
		g[3*(i - ov + n1) + 2].threshold = g2[i].threshold;
	}

	//synchronized edges
	for (int i = 0; i < n; i++) {
		g.add_edge(3*i + 1, 3*i, 1);
		g.add_edge(3*i + 2, 3*i, 1);
		g.add_edge(3*i + 1, 3*i + 2, g[3*i + 2].threshold);
		g.add_edge(3*i + 2, 3*i + 1, g[3*i + 1].threshold);
		g.add_edge(3*i, 3*i + 2, g[3*i + 2].threshold);
		g.add_edge(3*i, 3*i + 1, g[3*i + 1].threshold);
	}

	// add normal edge
	vector<Edge*> es1 = g1.edge_set();
	for (int i = 0; i < es1.size(); i++) {
		Edge e = *es1[i];
		g.add_edge(e.source * 3, e.dest * 3 + 1, e.weight);
	}

	vector<Edge*> es2 = g2.edge_set();
	for (int i = 0; i < es2.size(); i++) {
		Edge e = *es2[i];
		int s = (e.source >= ov)? e.source - ov + n1 : e.source;
		int d = (e.dest >= ov)? e.dest - ov + n1 : e.dest;
		g.add_edge(s * 3, d * 3 + 2, e.weight);
	}

	return gr;
}

Graph* lossy(Graph& g1, Graph &g2, int ov, vector<double> ea1, vector<double> ea2) {
	int n1 = g1.order();
	int n2 = g2.order();
	int n = n1 + n2 - ov; // number of users
	Graph *gr = new Graph();
	Graph &g = *gr;
	for (int i = 0; i < n; i++) {
		g.add_vertex(i, 1);
	}
	for (int i = 0; i < ov; i++) {
		g[i].threshold = ea1[i] * g1[i].threshold + ea2[i] * g2[i].threshold;
		if (g[i].threshold == 0) g[i].threshold = 1;
	}

	//vertices in network 1
	for (int i = ov; i < n1; i++) {
		g[i].threshold = g1[i].threshold;
	}
	for (int i = ov; i < n2; i++) {
		g[i - ov + n1].threshold = g2[i].threshold;
	}
	// add edge of g1
	// add normal edge
	vector<Edge*> es1 = g1.edge_set();
	for (int i = 0; i < es1.size(); i++) {
		Edge e = *es1[i];
		if (e.dest < ov) {
			g.add_edge(e.source, e.dest, ea1[e.dest] * e.weight);
		} else g.add_edge(e.source, e.dest, e.weight);
	}
	vector<Edge*> es2 = g2.edge_set();
	for (int i = 0; i < es2.size(); i++) {
		Edge e = *es2[i];
		int s = (e.source >= ov)? e.source - ov + n1 : e.source;
		int d = (e.dest >= ov)? e.dest - ov + n1 : e.dest;
		if (d < ov) {
			if (g.is_edge(s, d)){
				g(s, d).weight += ea2[d] * e.weight;
			} else {
				g.add_edge(s, d, ea2[d] * e.weight);
			}
		} else g.add_edge(s, d, e.weight);
	}

	return gr;
}

Graph* lossy(Graph& g1, Graph &g2, int ov) {
	int n1 = g1.order();
	int n2 = g2.order();
	int n = n1 + n2 - ov; // number of users
	Graph *gr = new Graph();
	Graph &g = *gr;
	for (int i = 0; i < n; i++) {
		g.add_vertex(i, 1);
	}
	vector<double> ea1(ov, 0);
	vector<double> ea2(ov, 0);
	for (int u = 0; u < ov; u++) {
		vector<Edge*> in_nei_1 = g1.in_edges(u);
		double total = 0;
		for (int i = 0; i < in_nei_1.size(); i++) {
			total += in_nei_1[i]->weight;
		}
		ea1[u] = total/g1[u].threshold;
	}

	for (int u = 0; u < ov; u++) {
		vector<Edge*> in_nei_2 = g2.in_edges(u);
		double total = 0;
		for (int i = 0; i < in_nei_2.size(); i++) {
			total += in_nei_2[i]->weight;
		}
		ea2[u] = total/g2[u].threshold;
	}
	for (int i = 0; i < ov; i++) {
		g[i].threshold = ea1[i] * g1[i].threshold + ea2[i] * g2[i].threshold;
		if (g[i].threshold == 0) g[i].threshold = 1;
	}

	//vertices in network 1
	for (int i = ov; i < n1; i++) {
		g[i].threshold = g1[i].threshold;
	}
	for (int i = ov; i < n2; i++) {
		g[i - ov + n1].threshold = g2[i].threshold;
	}
	// add edge of g1
	// add normal edge
	vector<Edge*> es1 = g1.edge_set();
	for (int i = 0; i < es1.size(); i++) {
		Edge e = *es1[i];
		if (e.dest < ov) {
			g.add_edge(e.source, e.dest, ea1[e.dest] * e.weight);
		} else g.add_edge(e.source, e.dest, e.weight);
	}
	vector<Edge*> es2 = g2.edge_set();
	for (int i = 0; i < es2.size(); i++) {
		Edge e = *es2[i];
		int s = (e.source >= ov)? e.source - ov + n1 : e.source;
		int d = (e.dest >= ov)? e.dest - ov + n1 : e.dest;
		if (d < ov) {
			if (g.is_edge(s, d)){
				g(s, d).weight += ea2[d] * e.weight;
			} else {
				g.add_edge(s, d, ea2[d] * e.weight);
			}
		} else g.add_edge(s, d, e.weight);
	}

	return gr;
}

vector<double> involment(Graph &g, int ov) {
	int n = g.order();
	vector<double> ea(ov, 0);
	vector<int> nei1(n, -1);

	for (int u = 0; u < ea.size(); u++) {
		set<int> nei;
		nei.insert(u);
		nei1[u] = u;
		vector<int> in_nei = g.in_neighbor_ids(u);
		for (int i = 0; i < in_nei.size(); i++) {
			nei1[in_nei[i]] = u;
			nei.insert(in_nei[i]);
		}

		vector<int> out_nei = g.out_neighbor_ids(u);
		for (int i = 0; i < out_nei.size(); i++) {
			nei1[out_nei[i]] = u;
			nei.insert(out_nei[i]);
		}

		double total = 0;

		for (set<int>::iterator it = nei.begin(); it != nei.end(); it++) {
			int v = *it;
			vector<Edge*> in_edges = g.in_edges(v);

			for (int i = 0; i < in_edges.size(); i++) {
				Edge e = *in_edges[i];
				if (nei1[e.source] == u && nei1[e.dest] == u && g[e.dest].threshold > 0){
					total += e.weight/g[e.dest].threshold;
				}
			}
		}

		ea[u] = total/g[u].threshold;
	}
	return ea;
}

Graph* lossy_involment(Graph& g1, Graph &g2, int ov) {
	return lossy(g1, g2, ov, involment(g1, ov), involment(g2, ov));
}
vector<double> easiness2(Graph &g, int ov) {
	int n = g.order();
	vector<double> ea1(n, 0);

	for (int u = 0; u < n; u++) {
		vector<Edge*> in_edges = g.in_edges(u);
		double total = 0;
		for (int i = 0; i < in_edges.size(); i++) {
			total += in_edges[i]->weight;
		}
		ea1[u] = total/g[u].threshold;
	}

	vector<double> ea2(ov, 0);

	for (int u = 0; u < ea2.size(); u++) {
		ea2[u] = ea1[u];
		vector<int> in_neis = g.in_neighbor_ids(u);
		for (int i = 0; i < in_neis.size(); i++) {
			ea2[u] += ea1[in_neis[i]];
		}

		ea2[u] = ea2[u]/g[u].threshold;
	}

	return ea2;
}

vector<double> easiness(Graph &g, int ov) {
	int n = g.order();
	vector<double> ea1(ov, 0);

	for (int u = 0; u < ov; u++) {
		vector<Edge*> in_edges = g.in_edges(u);
		double total = 0;
		for (int i = 0; i < in_edges.size(); i++) {
			total += in_edges[i]->weight;
		}
		ea1[u] = total/(g[u].threshold * g[u].threshold);
	}

	return ea1;
}
Graph* lossy_easyness2(Graph& g1, Graph &g2, int ov) {
	return lossy(g1, g2, ov, easiness2(g1, ov), easiness2(g2, ov));
}

Graph* lossy_easyness(Graph& g1, Graph &g2, int ov) {
	return lossy(g1, g2, ov, easiness(g1, ov), easiness2(g2, ov));
}

vector<double> average(Graph &g) {
	vector<double> avg(g.order(), 1);
	for ( int u = 0; u < g.order(); u++)
		avg[u] = avg[u]/g[u].threshold;
	return avg;
}

Graph* lossy_average(Graph& g1, Graph &g2, int ov) {
	return lossy(g1, g2, ov, average(g1), average(g2));
}

Graph* generate(string file, string vertex, string edge, bool isweighted, bool isDirected, bool isUniform) {
	ifstream infile;
	infile.open(file.c_str());
	if (!infile.is_open())
	{
		cerr <<"Unable to open file \""<< file << "\"."<<endl;
		exit(1);
	}

	Graph *g = new Graph();

	int max = 0;
	while (!infile.eof())
	{
		int source, dest;

		infile >> dest >> source;
		g->add_vertex(source, 1);
		g->add_vertex(dest, 1);
		max = (max < source) ? source : max;
		max = (max < dest) ? dest : max;
		if (source != dest) {
			if (isweighted) {
				float w;
				infile >> w;
				g->add_edge(source, dest, w);
				if (!isDirected) g->add_edge(dest, source, w);
			} else {
				g->add_edge(source, dest, random(1));
				if (!isDirected) g->add_edge(dest, source, random(1));
			}
		}
	}

	//For the current dataset only, introduce vertex 0
	/*
	for (int i = 0; i <= max; i++) {
		g->add_vertex(i, 1);
	}
	*/
	//If the uniform scheme is selected
	if (!isweighted && isUniform) {
		vector<Edge*> es = g->edge_set();
		for (int i = 0; i < es.size(); i++) {
			Edge &e = *es[i];
			(*g)(e.source, e.dest).weight = 1.0/g->in_degree(e.dest);
		}
	}

	vector<int> vs = g->vertex_ids();
	for (int i = 0; i < vs.size(); i++) {
		double total = 0;
		vector<Edge*> inedges = g->in_edges(vs[i]);
		for (int j = 0; j < inedges.size(); j++){
			total += inedges[j]->weight;
		}

		(*g)[vs[i]].threshold = random(total);
		if ((*g)[vs[i]].threshold == 0) (*g)[vs[i]].threshold = 1;

	}
	g->print2file(vertex, edge);
	return g;
}

Graph* generate(string file, string vertex, string edge, bool isweighted) {
	return generate(file, vertex, edge, isweighted, true, false);
}
Graph* generate(string file, string vertex, string edge) {
	return generate(file, vertex, edge, false, true, false);
}

/*
 * Read all integers in a file
 */
set<int> read_ints(string file) {
	set<int> numbers;
	ifstream infile;
	infile.open(file.c_str());
	if (!infile.is_open()) {
		cerr <<"Unable to open file \""<< file << "\"."<<endl;
		exit(1);
	}

	while (!infile.eof()) {
		int num;
		infile >> num;
		numbers.insert(num);
	}
	infile.close();
	return numbers;
}

/*
 * Check whether a number is in a set or not
 */
bool exist(int num, set<int>& numbers) {
	return numbers.find(num) != numbers.end();
}

bool exist(int num, map<int, int>& numbers) {
	return numbers.find(num) != numbers.end();
}

void preprocess(string twitter, string square, string crosstwitter, string crosssquare, string newtwitter, string newsquare, string newcrosstwitter, string newcrosssquare) {
	set<int> vs_1 = read_ints(twitter);
	set<int> vs_2 = read_ints(square);

	map<int, int> id2index_twitter;
	for (set<int>::iterator it = vs_1.begin(); it != vs_1.end(); it++) id2index_twitter[*it] = -1;
	map<int, int> id2index_square;
	for (set<int>::iterator it = vs_2.begin(); it != vs_2.end(); it++) id2index_square[*it] = -1;
	int index = 0;

	ifstream tcross, scross;
	ofstream newtcross, newscross;
	tcross.open(crosstwitter.c_str());
	scross.open(crosssquare.c_str());
	newtcross.open(newcrosstwitter.c_str());
	newscross.open(newcrosssquare.c_str());

	if (!tcross.is_open() || !scross.is_open() || !newtcross.is_open() || !newscross.is_open()) {
		cerr << "Unable to open file" << endl;
		exit(1);
	}

	//Indexing crossing nodes
	while (!tcross.eof() && !scross.eof() ){
		int u, v;
		tcross >> u;
		scross >> v;
		if (exist(u, vs_1) && exist(v, vs_2) && id2index_twitter[u] == -1) {
			id2index_twitter[u] = index;
			id2index_square[v] = index;
			newtcross << index << "\n";
			newscross << index << "\n";
			index++;
		}
	}
	tcross.close();
	scross.close();
	newtcross.close();
	newscross.close();

	int index_twitter = index;
	ifstream twitter_file;
	ofstream new_twitter_file;
	twitter_file.open(twitter.c_str());
	new_twitter_file.open(newtwitter.c_str());

	while (!twitter_file.eof()) {
		int source, dest;
		twitter_file >> source >> dest;
		//Update index
		if (id2index_twitter[source] == -1) {
			id2index_twitter[source] = index_twitter;
			index_twitter++;
		}
		if (id2index_twitter[dest] == -1) {
			id2index_twitter[dest] = index_twitter;
			index_twitter++;
		}

		new_twitter_file << id2index_twitter[source] << "\t" << id2index_twitter[dest] << "\n";
	}
	twitter_file.close();
	new_twitter_file.close();

	int index_square = index;
	ifstream square_file;
	ofstream new_square_file;
	square_file.open(square.c_str());
	new_square_file.open(newsquare.c_str());

	while (!square_file.eof()) {
		int source, dest;
		square_file >> source >> dest;
		//Update index
		if (id2index_square[source] == -1) {
			id2index_square[source] = index_square;
			index_square++;
		}
		if (id2index_square[dest] == -1) {
			id2index_square[dest] = index_square;
			index_square++;
		}

		new_square_file << id2index_square[source] << "\t" << id2index_square[dest] << "\n";
	}
	square_file.close();
	new_square_file.close();
}

void preprocess_coauthor(string network1, string network2, string cross, string new_network1, string new_network2, string newcross) {
	// read set of nodes in each network
	set<int> vs_1;
	ifstream net1;
	net1.open(network1.c_str());
	if (!net1.is_open()) {
		cerr << "Can not open file " << network1 << endl;
		exit(1);
	}

	while (!net1.eof()) {
		int s = -1, d = -1;
		double w;
		net1 >> s >> d >> w;

		//cout << s << "\t" << d << "\t" << w << endl;
		if (s + d >= 0 && s != d) {
			vs_1.insert(s);
			vs_1.insert(d);
		}
		//if (s + d < 0) break;
	}
	net1.close();

	set<int> vs_2;
	ifstream net2;
	net2.open(network2.c_str());

	if (!net2.is_open()) {
		cerr << "Can not open file " << network2 << endl;
		exit(1);
	}

	while (!net2.eof()) {
		int s, d;
		double w;
		net2 >> s >> d >> w;
		if (s + d >= 0 && s != d) {
			vs_2.insert(s);
			vs_2.insert(d);
		}
	}
	net2.close();

	map<int, int> id2index_1;
	for (set<int>::iterator it = vs_1.begin(); it != vs_1.end(); it++) id2index_1[*it] = -1;
	map<int, int> id2index_2;
	for (set<int>::iterator it = vs_2.begin(); it != vs_2.end(); it++) id2index_2[*it] = -1;
	int index = 0;

	ifstream crossfile;
	ofstream newcrossfile;

	crossfile.open(cross.c_str());
	newcrossfile.open(newcross.c_str());

	if (!crossfile.is_open() || !newcrossfile.is_open()) {
		cerr << "Unable to open file" << endl;
		exit(1);
	}

	//Indexing crossing nodes
	while (!crossfile.eof()){
		int u = -1, v = -1;
		crossfile >> u >> v;
		//cout << u << "\t" << v << "\t" << w << endl;
		if (exist(u, vs_1) && exist(v, vs_2) && id2index_1[u] == -1 && u + v >= 0) {
			id2index_1[u] = index;
			id2index_2[v] = index;
			index++;
		}

		if (exist(u, vs_1) && exist(v, vs_2)) newcrossfile << id2index_1[u] << "\t" << id2index_2[v] << "\n";
	}
	crossfile.close();
	newcrossfile.close();

	int index_1 = index;

	ofstream new_net1;
	net1.open(network1.c_str());
	new_net1.open(new_network1.c_str());

	while (!net1.eof()) {
		int source = -1, dest = -1;
		double w;
		net1 >> source >> dest >> w;
		if (w <= 0) w = 1;
		//Update index
		if (source + dest >= 0 && source != dest) {
			if (id2index_1[source] == -1) {
				id2index_1[source] = index_1;
				index_1++;
			}
			if (id2index_1[dest] == -1) {
				id2index_1[dest] = index_1;
				index_1++;
			}

			new_net1 << id2index_1[source] << "\t" << id2index_1[dest] << "\t" << w << "\n";
		}
	}
	net1.close();
	new_net1.close();

	int index_2 = index;

	ofstream new_net2;
	net2.open(network2.c_str());
	new_net2.open(new_network2.c_str());

	while (!net2.eof()) {
		int source = -1, dest = -1;
		double w;
		net2 >> source >> dest >> w;
		if (w <= 0) w = 1;
		//Update index
		if (source + dest >= 0 && source != dest) {
			if (id2index_2[source] == -1) {
				id2index_2[source] = index_2;
				index_2++;
			}
			if (id2index_2[dest] == -1) {
				id2index_2[dest] = index_2;
				index_2++;
			}

			new_net2 << id2index_2[source] << "\t" << id2index_2[dest] << "\t" << w << "\n";
		}
	}
	net2.close();
	new_net2.close();
}

Graph* clique_lossless(vector<Graph*> gs, int usize) {
	vector<int> user(usize);
	int k = gs.size();

	for (int i = 0; i < usize; i++) user[i] = i;

	//lossless couple networks
	Graph *coupledGraph = new Graph();
	Graph &cg = *coupledGraph;

	for (int i = 0; i < (k+1) * usize; i++) {
		cg.add_vertex(i, 1);
	}

	// Import edge weight and vertex threshold of each network
	for (int n = 0; n < k; n++) {
		int add = (n + 1) * usize;
		Graph &g = *gs[n];

		// Change the threshold of nodes
		vector<int> ids = gs[n]->vertex_ids();
		for (int i = 0; i < ids.size(); i++) {
			cg[ids[i] + add].threshold = g[ids[i]].threshold;
		}

		// add edges
		vector<Edge*> es = g.edge_set();
		for (int i = 0; i < es.size(); i++) {
			Edge e = *es[i];
			cg.add_edge(e.source, e.dest + add, e.weight);
		}
	}

	//Add synchronization edges
	for (int u = 0; u < usize; u++) {
		for (int n = 1; n <= k; n++) {
			cg.add_edge(u, u + n * usize, cg[u + n * usize].threshold);
			cg.add_edge(u + n * usize, u, cg[u].threshold);
			for (int n1 = 1; n1 <= k; n1++) {
				if (n1 != n) {
					cg.add_edge(u + n * usize, u + n1 * usize, cg[u + n1 * usize].threshold);
				}
			}
		}
	}

	return coupledGraph;
}

/*
 * Couple multiple networks into one network
 */
vector<Graph*> couple_multiple(vector<Graph*> gs, vector<vector<vector<pair<int, int> > > > overlap, string vinfo, string einfo) {
	vector<map<int, int> > idmap;
	vector<Graph*> results;
	int index = 0;
	for (int i = 0; i < gs.size(); i++) {
		Graph &g = *gs[i];
		vector<int> ids = g.vertex_ids();
		map<int, int> newid;
		//Map the overlapped nodes first
		for (int k = 0; k < i; k++) {
			for (int p = 0; p < overlap[i][k].size(); p++) {
				//cout << "die" << endl;
				pair<int, int> pa = overlap[i][k][p];
				newid[pa.first] = idmap[k][pa.second];

			}
		}

		//Map remaind nodes
		for (int u = 0; u < ids.size(); u++) {
			if (newid.find(ids[u]) == newid.end()){ // new id
				newid[u] = index;
				index++;
			}
		}

		//Create new graph
		Graph* ng = new Graph();
		for (int u = 0; u < ids.size(); u++) {
			ng->add_vertex(newid[ids[u]], g[ids[u]].threshold);
		}

		vector<Edge*> es = g.edge_set();
		for (int uv = 0; uv < es.size(); uv++) {
			Edge e = *es[uv];
			ng->add_edge(newid[e.source], newid[e.dest], e.weight);
		}
		results.push_back(ng);
		idmap.push_back(newid);
		cout << i << endl;
	}

	//cout << "test" << endl;
	Graph *couple = clique_lossless(results, index);
	couple->print2file(vinfo, einfo);
	delete couple;

	return results;
}

/*
 * Design for chn style
 * Easines
 */
map<int, double> easinesschn(Graph &g) {
	int n = g.order();
	vector<int> ids = g.vertex_ids();
	map<int, double> ea1;

	for (int i = 0; i < ids.size(); i++) {
		int u = ids[i];
		vector<Edge*> in_edges = g.in_edges(u);
		double total = 0;
		for (int i = 0; i < in_edges.size(); i++) {
			total += in_edges[i]->weight;
		}
		ea1[u] = total/(g[u].threshold * g[u].threshold);
	}

	return ea1;
}

/*
 * Design for chn style
 * Average
 */

map<int, double> averagechn(Graph &g) {
	vector<int> ids = g.vertex_ids();
	map<int, double> avg;

	for (int i = 0; i < ids.size(); i++) {
		avg[ids[i]] = 1/g[ids[i]].threshold;
	}

	return avg;
}
/*
 * Design for chn style
 * Involvement
 */
map<int, double> involmentchn(Graph &g) {
	int n = g.order();
	vector<int> ids = g.vertex_ids();
	map<int, double> ea;

	map<int, int> nei1;
	for (int i = 0; i < n; i++) {
		nei1[ids[i]] = -1;
	}

	for (int i = 0; i < n; i++) {
		int u = ids[i];
		set<int> nei;
		nei.insert(u);
		nei1[u] = u;
		vector<int> in_nei = g.in_neighbor_ids(u);
		for (int i = 0; i < in_nei.size(); i++) {
			nei1[in_nei[i]] = u;
			nei.insert(in_nei[i]);
		}

		vector<int> out_nei = g.out_neighbor_ids(u);
		for (int i = 0; i < out_nei.size(); i++) {
			nei1[out_nei[i]] = u;
			nei.insert(out_nei[i]);
		}

		double total = 0;

		for (set<int>::iterator it = nei.begin(); it != nei.end(); it++) {
			int v = *it;
			vector<Edge*> in_edges = g.in_edges(v);

			for (int i = 0; i < in_edges.size(); i++) {
				Edge e = *in_edges[i];
				if (nei1[e.source] == u && nei1[e.dest] == u && g[e.dest].threshold > 0){
					total += e.weight/g[e.dest].threshold;
				}
			}
		}

		ea[u] = total/g[u].threshold;
	}
	return ea;
}

/*
 * Lossy coupling for chn style
 */
Graph* lossy(vector<Graph*> gs, vector<map<int, double>> eas) {
	Graph *gr = new Graph();
	Graph &g = *gr;

	//Add vertices
	for (int i = 0; i < gs.size(); i++) {
		vector<int> ids = gs[i]->vertex_ids();
		for (int j = 0; j < ids.size(); j++) {
			if (!g.is_vertex(ids[j])) g.add_vertex(ids[j], 0);
		}
	}

	//Compute vertex threshold
	for (int i = 0; i < gs.size(); i++) {
		vector<int> ids = gs[i]->vertex_ids();
		for (int j = 0; j < ids.size(); j++) {
			int u = ids[j];
			g[u].threshold += eas[i][u] * (*gs[i])[u].threshold;
		}
	}
	// Treat 0 threshold
	vector<int> vids = g.vertex_ids();
	for (int i = 0; i < vids.size(); i++) {
		if (g[vids[i]].threshold <= 0) g[vids[i]].threshold = 1;
	}

	//Add edges
	for (int i = 0; i < gs.size(); i++) {
		vector<Edge*> es = gs[i]->edge_set();
		for (int j = 0; j < es.size(); j++) {
			Edge e = *es[j];
			if (!g.is_edge(e.source, e.dest)) {
				g.add_edge(e.source, e.dest, eas[i][e.dest] * e.weight);
			} else { //add weight to the existing edge
				g(e.source, e.dest).weight += eas[i][e.dest] * e.weight;
			}
		}
	}
	return gr;
}
