/* Copyright 2010, Thang N. Dinh, CISE, UF */

/*
	 * INPUT: Graph G=(V,E) of size |V|=Nodes with E given by an adjacency matrix a[][].
	 * OUTPUT: Clustering of G with maximum modularity (Newman)	
	 
	 * (Symmetric/Asymmetric) Adjacent matrix:	a[i][j] = 1 if i is adjacent to j and 0 otherwise
	 * x[i][j] = 1 if i,j are in DIFFERENT clusters, 0 otherwise
     * 
	 * IP:
	 * 		Min	 Sum (i < j)  m(i,j) x(i,j)
	 * s.t	 
	 * (3)		x[i][j] +x[j][k] - x[i][k]  >= 0// For all i <> j <> k - Transitivity
	 * (5)		x[i][j] =x[j][i]
	 * (6)		x[i][j] = 0 or 1
     *
     *			x[i][i] = 0
     * Where
     * 			m(i, j) = 2m * a(i,j) - d_i (out) * d_j (in)
     * 			2m = sum of d_i(out)
     * Modularity:
     * 			Q = -1/(4m^2) * (2 * Objective )
 */



#include "gurobi_c++.h"
#include "clustering.h"
#include "generalGraph.hpp"
#include "graphAlgorithm.hpp"
#include "edgeMap.hpp"
#include "config.hpp"
#include "ip.h"
#include "clustering.h"
#include "common.h"
#include "ModularityMetric.h"
#include <ctime>
#include <queue>
#include <cstdlib>
#include <cstdio>
#include <string>
#include <vector>
#include <algorithm>
#include <limits>

using namespace std;

Configuration	*par;

int		main(int   argc,     char *argv[])
{
  Configuration	cfg(argc, argv);
  par = &cfg;
  if (cfg.exists("help")) {
  	  cout <<"Finding a network clustering with the maximal Newman's modularity."<<endl
  		   <<"Usage: "<<argv[0]<<" -input <file name> -model <model name> -cut <cutting method> [options]"<<endl
  		   <<"Options: -model	<complete | mindegree | vertexcut | full_lp >" << endl
           <<"         -cut	<aggresive | normal | lazy | none >" << endl
           <<"         -directed  The graph is directed, NO graph preprocessing."<<endl
           <<"         -weighted  Graph is weighted."<<endl
  		   <<"         -init <cluster file> An initial clustering in Pajek format."<<endl
           <<"         -grb <file name> Additional parameter/model files for GUROBI."<<endl
  		   <<"         -cluster <file name> Write out the membership in Pajek format"<<endl
           <<"         -output <file name> Final cluster in Pajek format. If not provided,"<<endl
           <<"		   -light Turn off all heuristics e.g. leaves, 3-cyle, .etc"<<endl
           <<"the cluster will be written into the input file with extension '.clu'."<<endl;
           
      return 0;
  }
  if (!cfg.exists("model") || !cfg.exists("input") ) {
	  cerr <<"Error: Invalid syntax! See: "
			  << argv[0]<<" -help"<<endl;
      return 0;
  }
  GeneralGraph<double> g( cfg.value("input"), cfg.exists("weighted") );
  if (!cfg.exists("directed") )
      g.toUndirected();
  if (!cfg.exists("weighted")) 
      g.setWeight(1);
  Membership 	init;
  vector<string> grb_files;
  FOR(i, cfg.count("grb") )
      grb_files.push_back(cfg.value("grb",i+1));
  if (cfg.exists("init") )
	  init = loadPajekCluster(cfg.value("init"));
  if (cfg.exists("directed") ) {
	  //TODO: directed
  } else if (cfg.exists("compact")) {
	  //ModularityMetric mm(g, &cfg);
	  //Membership mbs = mm.optimize(init);
	  //writePajekCluster( (cfg.exists("output"))?cfg.value("output"):cfg.value("input")+".clu", mbs);
  } else {
      IpModel im = MODEL_NONE;
      if (cfg.exists("model")) {
	      if (cfg.match("model","full") )		  
              im = MODEL_FULL_IP;
	      else if (cfg.match("model","mindegree") )
		      im = MODEL_MIN_DEGREE;
	      else if (cfg.match("model","vertexcut") ) {
		      im = MODEL_VERTEX_CUT;
	      } else if (cfg.match("model","full_lp") ) {
              im = MODEL_FULL_LP;
          }
      }
      CuttingMethod cm = CM_LAZY;
      if (cfg.exists("cut")) {
          if (cfg.match("cut","aggressive"))
              cm = CM_AGGRESSIVE;
          else if (cfg.match("cut","normal"))
              cm = CM_NORMAL;
          else if (cfg.match("cut","lazy"))
              cm = CM_LAZY;
          else if (cfg.match("cut","none"))
              cm = CM_NONE;
      }
      clock_t	start = clock();
      //Membership mbs = modularity_ip(g, init, im, cm, grb_files, (!cfg.exists("light")) && (!cfg.exists("directed")));
      cout<<"Total clock time: "<<(clock()-start)/(1.0*CLOCKS_PER_SEC)<<" second(s)."<<endl;
      //writePajekCluster( (cfg.exists("output"))?cfg.value("output"):cfg.value("input")+".clu", mbs);

  }
  return 0;
}

