/*
 * Util.cpp
 *
 *  Created on: Feb 21, 2011
 *      Author: dung
 */

#include "Util.h"

/*
 * Export the edge list of graph to the text file
 */
void exportEdgeListTxt(AdjGraph *graph, const char* fileName){
	int n = graph->order();

	FILE *fo;
	fo = fopen(fileName, "w");
	if(fo == NULL) {
		cerr << "Error: File access denied "<< endl;
		return;
	}

	FOR(u, n) {
		vector<int> neis = graph->neighbors(u);
		FOR(i, neis.size()){
			if (neis[i] > u){
				fprintf(fo, "%d\t%d\n", u, neis[i]);
			}
		}
	}

	fclose(fo);
} //End of exportEdgeListTxt
/******************************************************************************/
/*
 * Convert text file to bin File
 * Format of text file: a list of edge. Each line has format: start end
 * Format of bin File: n, accumulated degree list, id2name, list of adjacentlist
 */

int convertFile(const char *txtFile,const char *binFile){
	FILE *fin = fopen(txtFile, "r");
	FILE *fo  = fopen(binFile, "wb");
	if (!fin || !fo) {
		cerr <<"Error: File access denied."<<endl;
		return 2;
	}
	//hashTable degree;
	hashTable nodes;

	int id = 0;
	int lineCount = 0;
	int from, to;
	while (!feof(fin) ) {
		from=-1; to=-1;
		fscanf(fin,"%d %d",&from, &to);
		lineCount++;
		if(lineCount % 1000 == 0)
			printf("line number %d\n", (lineCount/1000));
		if ((from + to > 0)) {
			if (nodes.find(from) == nodes.end() ) {
				nodes[ from ].first = id;
				nodes[ from ].second = 0;
				id++;
			}
			if (nodes.find(to) == nodes.end() ) {
				nodes[ to ].first = id;
				nodes[to].second = 0;
				id++;
			}
			nodes[from].second++;
			nodes[to].second++;
		}
	}

	//printf("Number of node %d and number of lines %d ", id, lineCount/2);

	//Number of node
	int n = id;
	int  *index = new int[n];
	int **adj = new int*[n];
	int *id2name = new int[n];

	FOREACH (itr, nodes) {
		id = itr->second.first;
		id2name[ id ] = itr->first;
		index[id] = 0;
		adj[id] = new int[itr->second.second];
	}
	// printf("id of node 10 %d number of node%d degree of 10 is %d" ,nodeId[10], n, degree[10]);

	// Reread the adjacent list
	fclose(fin);
	lineCount = 0;
	fin = fopen(txtFile,"r");
	while (!feof(fin) ) {
		int from=-1, to=-1;
		fscanf(fin,"%d %d",&from, &to);
		lineCount++;
		if(lineCount % 1000000 == 0)
			printf("line number %d\n", (lineCount/1000000));

		if ((from + to > 0)&&(from != to) ) {
			from  = nodes[ from].first;
			to 	  = nodes[ to ].first;
			adj[from][ index[from] ] = to;
			adj[to][ index[to]] = from;
			index[from]++;
			index[to]++;
		}
	}
	fclose(fin);

	int  sumDegree = 0;
	FOR(i, n)
	sumDegree += index[i];
	//	printf("Number of nodes: %d, Number of edges: %d\n", n, sumDegree/2);
	// Eliminate duplication

	int *check = new int[n];
	FOR(i,n) {
		check[i] = -1;
	}

	//Eliminate duplication
	FOR(u, n){
		// cout << u << endl;	cout.flush();
		FOR(j, index[u]){
			int v = adj[u][j];
			if (check[v] != u){
				check[v] = u;
			}else{
				adj[u][j] = adj[u][index[u]-1];
				index[u]--;
				j--;
			}
		}
	}
	sumDegree = 0;
	FOR(i, n)
	sumDegree += index[i];
	//	printf("After duplicate removal: Number of nodes: %d, Number of edges: %d\n", n, sumDegree/2);
	int c1, c2, c3;
	c1 = fwrite((void *)&n, 	sizeof(n), 1, fo);
	c2 = fwrite((void *)index, sizeof(int), n, fo );
	c3 = fwrite((void *)id2name, sizeof(int), n, fo);
	if (c1 != 1 || c2 != n || c3 != n) {
		cerr << "Error: cannot write to file" << binFile << endl;
		return 2;
	}
	FOR(u, n) {
		int c= fwrite( (void *)adj[u], sizeof(int), index[u], fo);
		if (c != index[u]) {
			cerr << "Error: cannot write to file" << binFile << endl;
			return 2;
		}
	}
	delete[] id2name;
	delete[] index;
	delete[] check;
	FOR(i, n)
	delete[] adj[i];
	delete[] adj;
	fclose(fo);
	return 0;
} //End convertFile
/******************************************************************************/
/*
 * Convert a text file dataset to binary file dataset
 * The text file includes multiple rows, each row represent an edge with format: from to where from, to are numbers
 * The binfile includes:
 * 	+ The first number is the number of nodes n
 * 	+ The next n numbers are the out-degrees of all nodes
 * 	+ The next n numbers are the map from node id (1 to n) to the name given in the text file
 * 	+ The last part includes the out-neighbor list of all nodes
 */
int convertFile(const char *txtFile,const char *binFile, bool isDirected, bool isRevert){
	FILE *fin = fopen(txtFile, "r");
	FILE *fo  = fopen(binFile, "wb");
	if (!fin || !fo) {
		cerr <<"Error: File access denied."<<endl;
		return 2;
	}
	//hashTable degree;
	hashTable nodes;

	int id = 0;
	int lineCount = 0;
	int from, to, mediate;
	//Map the name of node (which is a number) in the text file to the new id and count the number of out-edges
	while (!feof(fin) ) {
		from=-1; to=-1;
		fscanf(fin,"%d %d",&from, &to);
		lineCount++;
		if(lineCount % 10000 == 0)
			printf("line number %d x 10000\n", (lineCount/10000));
		if ((from + to > 0) && from != to) {
			if (isRevert) {
				mediate = from;
				from = to;
				to = mediate;
			}
			if (nodes.find(from) == nodes.end() ) {
				nodes[ from ].first = id;
				nodes[ from ].second = 0;
				id++;
			}
			nodes[from].second++;
			if (!isDirected) {
				if (nodes.find(to) == nodes.end() ) {
					nodes[ to ].first = id;
					nodes[to].second = 0;
					id++;
				}
				nodes[to].second++;
			}
		}
	}

	int n = id; //Number of node
	int  *index = new int[n]; // The position to write the next out-going neighbor of each node
	int **adj = new int*[n]; // Lists of out-going neighbors of all nodes
	int *id2name = new int[n]; // project the id into the name of node in text file

	FOREACH (itr, nodes) {
		id = itr->second.first;
		id2name[ id ] = itr->first;
		index[id] = 0;
		adj[id] = new int[itr->second.second];
	}

	// Reread the adjacent list
	fclose(fin);
	lineCount = 0;
	fin = fopen(txtFile,"r");
	while (!feof(fin) ) {
		int from=-1, to=-1;
		fscanf(fin,"%d %d",&from, &to);
		lineCount++;
		if(lineCount % 10000 == 0)
			printf("line number %d x 10000\n", (lineCount/10000));

		if ((from + to > 0)&&(from != to) ) {
			if (isRevert) {
				mediate = from;
				from = to;
				to = mediate;
			}
			from  = nodes[ from].first;
			to 	  = nodes[ to ].first;
			adj[from][ index[from] ] = to;
			index[from]++;

			if (!isDirected) {
				adj[to][ index[to]] = from;
				index[to]++;
			}
		}
	}

	fclose(fin);

	// Eliminate duplication
	int *check = new int[n];
	FOR(i,n) {
		check[i] = -1;
	}

	//Eliminate duplication
	FOR(u, n){
		FOR(j, index[u]){
			int v = adj[u][j];
			if (check[v] != u){
				check[v] = u;
			}else{
				adj[u][j] = adj[u][index[u]-1];
				index[u]--;
				j--;
			}
		}
	}

	int  sumDegree = 0;
	FOR(i, n)
		sumDegree += index[i];
	// Print to file
	int c1, c2, c3;
	c1 = fwrite((void *)&n, 	sizeof(n), 1, fo);
	c2 = fwrite((void *)index, sizeof(int), n, fo );
	c3 = fwrite((void *)id2name, sizeof(int), n, fo);
	if (c1 != 1 || c2 != n || c3 != n) {
		cerr << "Error: cannot write to file" << binFile << endl;
		return 2;
	}
	FOR(u, n) {
		int c= fwrite( (void *)adj[u], sizeof(int), index[u], fo);
		if (c != index[u]) {
			cerr << "Error: cannot write to file" << binFile << endl;
			return 2;
		}
	}
	delete[] id2name;
	delete[] index;
	delete[] check;
	FOR(i, n)
	delete[] adj[i];
	delete[] adj;
	fclose(fo);
	return 0;
} //End convertFile
/*
 * Convert pajek file to bin File
 * Format of text file: a list of edge. Each line has format: start end
 * Format of bin File: n, accumulated degree list, id2name, list of adjacentlist
 */

int convertParjekFile(const char *txtFile,const char *binFile){
	FILE *fin = fopen(txtFile, "r");
	FILE *fo  = fopen(binFile, "wb");
	if (!fin || !fo) {
		cerr <<"Error: File access denied."<<endl;
		return 2;
	}
	//hashTable degree;
	hashTable nodes;

	int id = 0;
	int lineCount = 0;
	int from, to;
	char line[20];
	while(fgets(line, 20, fin) != NULL){
		string s = string(line);
		if (s.find("*Edges") != string::npos)
			break;
	}
	while (!feof(fin) ) {
		//Find the edge

		from=-1; to=-1;
		fscanf(fin,"%d %d",&from, &to);
		lineCount++;
		if(lineCount % 1000000 == 0)
			printf("line number %d\n", (lineCount/1000000));
		if ((from + to > 0)) {
			if (nodes.find(from) == nodes.end() ) {
				nodes[ from ].first = id;
				nodes[ from ].second = 0;
				id++;
			}
			if (nodes.find(to) == nodes.end() ) {
				nodes[ to ].first = id;
				nodes[to].second = 0;
				id++;
			}
			nodes[from].second++;
			nodes[to].second++;
		}
	}

	//printf("Number of node %d and number of lines %d ", id, lineCount/2);

	//Number of node
	int n = id;
	int  *index = new int[n];
	int **adj = new int*[n];
	int *id2name = new int[n];

	FOREACH (itr, nodes) {
		id = itr->second.first;
		id2name[ id ] = itr->first;
		index[id] = 0;
		adj[id] = new int[itr->second.second];
	}
	// printf("id of node 10 %d number of node%d degree of 10 is %d" ,nodeId[10], n, degree[10]);

	// Reread the adjacent list
	fclose(fin);
	lineCount = 0;
	fin = fopen(txtFile,"r");
	while(fgets(line, 20, fin) != NULL){
		string s = string(line);
		if (s.find("*Edges") != string::npos)
			break;
	}
	while (!feof(fin) ) {


		int from=-1, to=-1;
		fscanf(fin,"%d %d",&from, &to);

		if ((from + to > 0)&&(from != to) ) {
			from  = nodes[ from].first;
			to 	  = nodes[ to ].first;
			adj[from][ index[from] ] = to;
			adj[to][ index[to]] = from;
			index[from]++;
			index[to]++;
		}
	}
	fclose(fin);

	int  sumDegree = 0;
	FOR(i, n)
	sumDegree += index[i];
	//	printf("Number of nodes: %d, Number of edges: %d\n", n, sumDegree/2);
	// Eliminate duplication

	int *check = new int[n];
	FOR(i,n) {
		check[i] = -1;
	}

	//Eliminate duplication
	FOR(u, n){
		// cout << u << endl;	cout.flush();
		FOR(j, index[u]){
			int v = adj[u][j];
			if (check[v] != u){
				check[v] = u;
			}else{
				adj[u][j] = adj[u][index[u]-1];
				index[u]--;
				j--;
			}
		}
	}
	sumDegree = 0;
	FOR(i, n)
	sumDegree += index[i];
	//	printf("After duplicate removal: Number of nodes: %d, Number of edges: %d\n", n, sumDegree/2);
	int c1, c2, c3;
	c1 = fwrite((void *)&n, 	sizeof(n), 1, fo);
	c2 = fwrite((void *)index, sizeof(int), n, fo );
	c3 = fwrite((void *)id2name, sizeof(int), n, fo);
	if (c1 != 1 || c2 != n || c3 != n) {
		cerr << "Error: cannot write to file" << binFile << endl;
		return 2;
	}
	FOR(u, n) {
		int c= fwrite( (void *)adj[u], sizeof(int), index[u], fo);
		if (c != index[u]) {
			cerr << "Error: cannot write to file" << binFile << endl;
			return 2;
		}
	}
	delete[] id2name;
	delete[] index;
	delete[] check;
	FOR(i, n)
	delete[] adj[i];
	delete[] adj;
	fclose(fo);
	return 0;
} //End convertFile

/*
 * Compute the distant between two vector
 */
bool dist(vector<double> x, vector<double> y, int norm, double& result) {
	double sum = 0;

	if (x.size() != y.size()) return false;

	FOR (i, x.size()) {
		sum += pow(abs(x[ i ] - y[ i ]), norm);
	}

	result = pow(sum, 1.0/norm);
	return true;
}

/*
 * Generate the next combination from the given combination
 */
int next_comb(vector<int> &comb, int k, int n) {
	int i = k - 1;
	while (i>0 && comb[i] >= n - k + i) {
		i--;
	}

	if (comb[i] >= n - k + i) return 0; // There is no more combination to generate

	comb[i]++;

	for (i = i + 1; i < k; i++) {
		comb[i] = comb[i-1] + 1;
	}

	return 1;
}

/*
 * Randomly generate a random random number
 */
int rand(int min, int max) {
	//random_device rd;
	srand(time(NULL));
	int result = min + floor(rand()*(max - min)/(RAND_MAX*1.0)  + 0.5);	
	return result;
}

/*
 * Generate a random combination of k numbers from 0 to n - 1
 */
vector<int> rand_comb(int k, int n) {
	vector<int> comb(k);
	vector<int> source(n);
	for (int i = 0; i < n; i++) source[i] = i;
	for (int i = 0; i < k; i++) {
		int index = rand(0, n - i -1);
		int element = source[index];
		int intermediate = source[index];
		source[index] = source[n-i-1];
		comb[i] = element;
	}
	return comb;
}
