/*
* ICM.cpp
*
*  Created on: Jul 25, 2010
*      Author: seven
*/

#include "ICM.h"
#include "IDS.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <queue>
#include <ctime>



//Get the coverage
//Given a graph, probability of each node and latency. Find the number of influenced nodes
int getCoverage(AdjGraph& graph, double probability, int latency, int* pids, int pidsSize){
	int n = graph.order();
	int numInfluence = 0;
	//printf("%d", pids[0]);

	bool* active = new bool[n];
	for (int i=0; i<n; ++i)
		active[i] = false;

	queue<int> q;
	//cout << n <<" : "<< require << " : " << pidsSize << endl;
	for (int i = 0; i < pidsSize; ++i){
		numInfluence++;
		active[pids [ i ] ] = true;
		q.push(pids[ i ]);
	}

	int round = 0;
	int levelSize;

	while (round <= latency){
		round++;
		levelSize = q.size();

		for (int j = 0; j < levelSize;j++){
			int node = q.front();
			q.pop();
			vector<int> neis = graph.neighbors(node);
			for (int k=0; k< neis.size();k++){
				int adjNode = neis[ k ];
				if (!active[ adjNode ]) {
					int rn = rand();
					if (rn < probability * RAND_MAX){
						active[ adjNode ] = true;
						q.push(adjNode);
						numInfluence++;
					}
				}
			}
		}
	}
	delete [] active;
	return numInfluence;
}// End of chek pids

/*
* Compute the expected number of users influenced by the set of seed nodes
* The monte carlo method is used with round number of trials.
*/
double icmExpectedInfluence(AdjGraph& graph, vector<int> seeds, int round){
	int n = graph.order();
	int numInfluence = 0;

	bool* active = new bool[n];

	srand(time(NULL));
	FOR (r, round) {
		for (int i=0; i<n; ++i)
			active[i] = false;

		queue<int> q;
		FOR (i, seeds.size()) {
			numInfluence++;
			active[seeds [ i ] ] = true;
			q.push(seeds[ i ]);
		}

		while (q.size() > 0){
			int node = q.front();
			q.pop();

			vector<int> neis = graph.neighbors(node);
			vector<double> p = graph.probability(node);

			FOR (i, neis.size()){
				int adjNode = neis[ i ];
				if (!active[ adjNode ]) {
					int rn = rand();
					if (rn < p[i] * RAND_MAX){
						active[ adjNode ] = true;
						q.push(adjNode);
						numInfluence++;
					}
				}
			}
		}
	}

	delete [] active;
	return numInfluence/(1.0 * round);
} // End of expectedInfluence
//=======================================================================================

/*
* Greedy algorithm to find top k influence using Independence cascade model
* round Number of simulations
*/
vector<int> icmTopGreedy(AdjGraph& graph, int k, int round){
	int n = graph.order();
	vector<int> top;

	bool* selected = new bool[n];
	FOR (i, n)
		selected[i] = false;

	FOR (t, k) {
		cout << " k " << t << endl;
		double maxValue = 0;
		double s;
		int bestNode;

		FOR (u, n) {
			if (!selected[u]) {
				top.push_back(u);
				s = icmExpectedInfluence(graph, top, round);
				if (s > maxValue) {
					maxValue = s;
					bestNode = u;
				}
				top.pop_back();
			}
		}

		if (maxValue == 0) break; // There is no more useful node

		top.push_back(bestNode);
		selected[bestNode] = true;
	}

	delete [] selected;
	return top;
} // End of icmTopGreedy
//===========================================================================
/*
* Find top k influence nodes using iMeter measurement. Top k ones are selected.
*/
vector<int> icmTopIMeter(AdjGraph& graph, int k, int len, int round) {
	vector<int> orderList = iMeterOrder(graph, len, round);
	vector<int> top;

	if (k > graph.order()) k = graph.order();

	FOR (i, k) {
		top.push_back(orderList[i]);
	}

	return top;
} // End of icmTopIMeter
//==========================================================================
/*
* Find top k influence nodes using the number of hit count of random walks measurement. Top k ones are selected.
*/
vector<int> icmTopRW(AdjGraph& graph, int k, int len, int numRW) {
	vector<double> ini;
	vector<int> orderList = hitCountSort(graph, len, numRW, ini);
	vector<int> top;

	if (k > graph.order()) k = graph.order(); // Can not get more than the number of node

	FOR (i, k) {
		top.push_back(orderList[i]);
	}
	return top;
} // End of icmTopRW
//==========================================================================
//===========================================================================
/*
* Find top k influence nodes using FusedWalk measurement. Top k ones are selected.
*/
vector<int> icmTopFusedWalk(AdjGraph& graph, int k, int len, int round) {
	vector<int> orderList = fusedWalkOrder(graph, len, round);
	vector<int> top;

	if (k > graph.order()) k = graph.order();

	FOR (i, k) {
		top.push_back(orderList[i]);
	}

	return top;
} // End of icmFusedWalk

/*
* Use the Fused Walk to initialize the influence value of nodes
* During the selection process, the influence value of a node is degree if its neighbor is selected
*/
vector<int> icmTopFuseWalkDiscount(AdjGraph& graph, int k, int len, int round){
	int n = graph.order();
	vector<int> fw = fusedWalk(graph, len, round);

	vector<heapElement> nodeList;
	for (int u = 0; u < n; ++u){
		heapElement e = {u, fw[u]};
		nodeList.push_back(e);
	}

	MaxHeap *nodeHeap = new MaxHeap(nodeList);
	nodeHeap->build();

	vector<int> top;
	top.reserve(k);

	//Select top k nodes
	FOR (i, k) {
		heapElement selectedElement = nodeHeap->extractMax();
		int selectedNode = selectedElement.id;
		double influenceValue = selectedElement.key;
		double factor = (influenceValue*1.0)/graph.getDegree(selectedNode);

		top.push_back(selectedNode);



		vector<int> inNeis = graph.neighbors(selectedNode);

		//Discount the influence of neighbors
		FOR (i, inNeis.size()) {
			int adjNode = inNeis[i];
			nodeHeap->changeKey(adjNode, -factor);
		}
	}
	delete nodeHeap;
	return top;
} //end pidsGreedHighestDomination

/*
* Compute the expected number of users influenced by the set of seed nodes
* The monte carlo method is used with round number of trials.
*/

double icmExpectedInfluence(Graph& g, vector<Vertex> seeds, int round){
	int n = num_vertices(g);
	int numInfluence = 0;

	bool* active = new bool[n];

	srand(time(NULL));
	FOR (r, round) {
		for (int i=0; i<n; ++i)
			active[i] = false;

		queue<Vertex> q;
		FOR (i, seeds.size()) {
			numInfluence++;
			active[ g[ seeds[ i ] ].index ] = true;
			q.push( seeds[ i ] );
		}

		while (q.size() > 0){
			Vertex node = q.front();
			q.pop();

			pair<out_edge_iterator, out_edge_iterator> outEdges = out_edges(node, g);

			for(; outEdges.first != outEdges.second; ++outEdges.first) {
				Edge e = *outEdges.first;
				Vertex adjNode = target(e, g);
				if ( !active[ g [ adjNode ].index ] ) {
					int rn = rand();
					if (rn < g[ e ].weight * RAND_MAX){
						active[ g [ adjNode ].index ] = true;
						q.push(adjNode);
						numInfluence++;
					}
				}
			}
		}
	}

	delete [] active;
	return numInfluence/(1.0 * round);
} // End of expectedInfluence

/*
* Greedy algorithm to find top k influence using Independence cascade model
* round Number of simulations
*/

vector<Vertex> icmTopGreedy(Graph& g, int k, int round){
	FILE *fo;
	char outName[40];
	sprintf(outName, "TopkGreedy_%d_%d.txt", num_vertices(g), num_edges(g));
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#Greedy algorithm \n");
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;

	int n = num_vertices(g);
	double *increasement = new double[n];
	double currentInfluence = 0;

	if (k > n) k = n;

	vector<Vertex> top;

	bool* selected = new bool[n];
	FOR (i, n) {
		selected[i] = false;
		increasement[i] = 0;
	}

	FOR (t, k) {
		assert((start = clock())!=-1);
		double maxValue = 0;
		double s;
		int bestNode;

		FOR (u, n) {
			Vertex v = vertex(u, g);
			if ( ! selected[ g [ v ].index ] ) {
				if (increasement[g[v].index] < maxValue)
					continue;
				top.push_back(v);
				s = icmExpectedInfluence(g, top, round) - currentInfluence;
				increasement[g[v].index] = s;
				if (s > maxValue) {
					maxValue = s;
					bestNode = u;
				}
				top.pop_back();
			}
		}

		if (maxValue == 0) break; // There is no more useful node

		top.push_back(vertex(bestNode,g));
		selected[bestNode] = true;

		stop = clock();
		rt += (double) (stop-start)/CLOCKS_PER_SEC;
		currentInfluence = icmExpectedInfluence(g, top, round);

		fprintf(fo, "%d\t%f\t%f\n", t + 1, currentInfluence, rt);
		fflush(fo);
	}

	fclose(fo);
	delete [] selected;
	return top;
} // End of icmTopGreedy
//===========================================================================
/*
* Find top k influence nodes using FusedWalk measurement. Top k ones are selected.
*/
vector<Vertex> icmTopFusedWalk(Graph& g, int k, int len, int round) {
	FILE *fo;
	char outName[40];
	sprintf(outName, "TopkFusedWalk_%d_%d.txt", num_vertices(g), num_edges(g));
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#FusedWalk algorithm \n");
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;
	assert((start = clock())!=-1);
	vector<Vertex> orderList = fusedWalkOrder(g, len, round);
	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	vector<Vertex> top;

	if (k > num_vertices( g ) ) k = num_vertices( g );

	FOR (i, k) {
		top.push_back(orderList[i]);
		fprintf(fo, "%d\t%f\t%f\n", i + 1, icmExpectedInfluence(g, top, round), rt);
		fflush(fo);
	}

	fclose(fo);
	return top;
} // End of icmFusedWalk

vector<Vertex> icmTopIMeter(Graph& g, int k, int len, int round) {
	FILE *fo;
	char outName[40];
	sprintf(outName, "TopkImeter_%d_%d.txt", num_vertices(g), num_edges(g));
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#Imeter algorithm k = %d len = %d round = %d \n", k, len, round);
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;
	assert((start = clock())!=-1);
	vector<Vertex> orderList = iMeterOrder(g, len, round);
	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	vector<Vertex> top;

	if (k > num_vertices( g ) ) k = num_vertices( g );

	FOR (i, k) {
		top.push_back(orderList[i]);
		fprintf(fo, "%d\t%f\t%f\n", i + 1, icmExpectedInfluence(g, top, round), rt);
		fflush(fo);
	}

	fclose(fo);
	return top;
} // End of icmFusedWalk

vector<Vertex> icmTopRW(Graph& g, int k, int len, int numRW) {
	int round = 1000;
	FILE *fo;
	char outName[40];
	sprintf(outName, "TopkRW_%d_%d.txt", num_vertices(g), num_edges(g));
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#RW algorithm \n");
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;
	assert((start = clock())!=-1);
	vector<double> ini;
	vector<Vertex> orderList = hitCountSort(g, len, numRW, ini);
	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	vector<Vertex> top;

	if (k > num_vertices( g ) ) k = num_vertices( g );

	FOR (i, k) {
		top.push_back(orderList[i]);
		fprintf(fo, "%d\t%f\t%f\n", i + 1, icmExpectedInfluence(g, top, round), rt);
		fflush(fo);
	}

	fclose(fo);
	return top;
} // End of icmFusedWalk
/*
* Greedy algorithm to find top k influence using Independence cascade model
* round Number of simulations
*/

vector<Vertex> icmTopLazyGreedy(Graph& g, int k, int round){
	FILE *fo;
	char outName[40];
	sprintf(outName, "TopkLazyGreedy_%d_%d.txt", num_vertices(g), num_edges(g));
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#LazyGreedy algorithm \n");
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;

	int n = num_vertices(g);
	double currentInfluence = 0;

	if (k > n) k = n;

	vector<Vertex> top;

	bool* selected = new bool[n];
	FOR (i, n) {
		selected[i] = false;
	}

	vector<Node> nodes;
	nodes.reserve(n);
	assert((start = clock())!=-1);
	FOR (u, n) {
		Vertex v = vertex(u, g);
		top.push_back(v);
		Node newNode = {u, icmExpectedInfluence(g, top, round) };
		nodes.push_back(newNode);
		top.pop_back();
	}
	sort(nodes.begin(), nodes.end(), comp);
	top.push_back(vertex(nodes[0].id, g));
	selected[nodes[0].id] = true;

	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	currentInfluence = icmExpectedInfluence(g, top, round);
	fprintf(fo, "%d\t%f\t%f\n", 1, currentInfluence, rt);

	for (int t=1; t < k; t++) {
		assert((start = clock())!=-1);
		double maxValue = 0;
		double s;
		int bestNode;
		sort(nodes.begin(), nodes.end(), comp);
		FOR (i, n) {
			int u = nodes[i].id;
			Vertex v = vertex(u, g);
			if ( ! selected[ g [ v ].index ] ) {
				if (nodes[g[v].index].key < maxValue)
					continue;
				top.push_back(v);
				s = icmExpectedInfluence(g, top, round) - currentInfluence;
				nodes[i].key = s;
				if (s > maxValue) {
					maxValue = s;
					bestNode = u;
				}
				top.pop_back();
			}
		}

		if (maxValue == 0) break; // There is no more useful node

		top.push_back(vertex(bestNode,g));
		selected[bestNode] = true;

		stop = clock();
		rt += (double) (stop-start)/CLOCKS_PER_SEC;
		currentInfluence = icmExpectedInfluence(g, top, round);

		fprintf(fo, "%d\t%f\t%f\n", t + 1, currentInfluence, rt);
		fflush(fo);
	}


	fclose(fo);
	delete [] selected;
	return top;
} // End of icmTopLazyGreedy
//===========================================================================

/*
* Local search
*/
vector<Vertex> localSwap(Graph& g, vector<Vertex> origin, int round) {
	int n = num_vertices(g);
	bool increaseable = true;
	bool *mark = new bool[n];
	FOR (i, origin.size()) {
		mark[ g[ origin[ i ] ].index ] = true;
	}

	double current = icmExpectedInfluence(g, origin, round);
	while (increaseable) {
		increaseable = false;

		FOR (i, origin.size()) {
			Vertex swappedVertex = origin[ i ];
			FOR (u, n) {
				Vertex newVertex = vertex(u, g);
				if (!mark[ g[ newVertex ].index ]) {
					origin[ i ] = newVertex;
					double newInfluence = icmExpectedInfluence(g, origin, round);
					if (current < newInfluence) {
						current = newInfluence;
						mark[ g[ swappedVertex ].index ] = false;
						mark[  g[ newVertex ].index ] = true;
						increaseable = true;
						swappedVertex = newVertex;
					} else origin[ i ] = swappedVertex;
				}
			}
		}
	}
	return origin;
} // End of localSwap
vector<Vertex> icmTopIMeterSwap(Graph& g, int k, int len, int round) {
	FILE *fo;
	char outName[40];
	sprintf(outName, "TopkImeterSwap_%d_%d.txt", num_vertices(g), num_edges(g));
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#icmTopIMeterSwap algorithm k = %d len = %d round = %d \n", k, len, round);
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;
	assert((start = clock())!=-1);
	vector<Vertex> orderList = iMeterOrder(g, len, round);
	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	vector<Vertex> top;

	if (k > num_vertices( g ) ) k = num_vertices( g );

	FOR (i, k) {
		top.push_back(orderList[i]);
		double currentInfluence = icmExpectedInfluence(g, top, round);
		top = localSwap(g, top, round);
		double swapInfluence = icmExpectedInfluence(g, top, round);
		fprintf(fo, "%d\t%f\t%f\t%f\n", i + 1, currentInfluence , swapInfluence, rt);
		fflush(fo);
	}

	fclose(fo);
	return top;
} // End of icmFusedWalk
vector<Vertex> localSwap(Graph& g, vector<Vertex> origin, vector<Vertex> candidate, int round) {
	int n = num_vertices(g);
	bool increaseable = true;
	bool *mark = new bool[n];
	FOR (i, origin.size()) {
		mark[ g[ origin[ i ] ].index ] = true;
	}

	double current = icmExpectedInfluence(g, origin, round);
	while (increaseable) {
		increaseable = false;

		FOR (i, origin.size()) {
			Vertex swappedVertex = origin[ i ];
			FOR (u, candidate.size()) {
				Vertex newVertex = candidate [u];
				if (!mark[ g[ newVertex ].index ]) {
					origin[ i ] = newVertex;
					double newInfluence = icmExpectedInfluence(g, origin, round);
					if (current < newInfluence) {
						current = newInfluence;
						mark[ g[ swappedVertex ].index ] = false;
						mark[  g[ newVertex ].index ] = true;
						increaseable = true;
						swappedVertex = newVertex;
					} else origin[ i ] = swappedVertex;
				}
			}
		}
	}
	return origin;
} // End of localSwap

vector<Vertex> icmTopLazyGreedySwap(Graph& g, int k, int round){
	FILE *fo;
	char outName[40];
	sprintf(outName, "TopkLazyGreedySwap_%d_%d_%d.txt", num_vertices(g), num_edges(g), round);
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#LazyGreedy algorithm k = %d round = %d \n", k, round);
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;

	int n = num_vertices(g);
	double currentInfluence = 0;

	if (k > n) k = n;

	vector<Vertex> top;
	vector<Vertex> orderList = iMeterOrder(g, 10, round);

	bool* selected = new bool[n];
	FOR (i, n) {
		selected[i] = false;
	}

	vector<Node> nodes;
	nodes.reserve(n);
	assert((start = clock())!=-1);
	FOR (u, n) {
		Vertex v = vertex(u, g);
		top.push_back(v);
		Node newNode = {u, icmExpectedInfluence(g, top, round) };
		nodes.push_back(newNode);
		top.pop_back();
	}
	sort(nodes.begin(), nodes.end(), comp);
	top.push_back(vertex(nodes[0].id, g));
	selected[nodes[0].id] = true;

	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	currentInfluence = icmExpectedInfluence(g, top, round);
	fprintf(fo, "%d\t%f\t%f\n", 1, currentInfluence, rt);

	for (int t=1; t < k; t++) {
		assert((start = clock())!=-1);
		double maxValue = 0;
		double s;
		int bestNode;
		sort(nodes.begin(), nodes.end(), comp);
		FOR (i, n) {
			int u = nodes[i].id;
			Vertex v = vertex(u, g);
			if ( ! selected[ g [ v ].index ] ) {
				if (nodes[g[v].index].key < maxValue)
					continue;
				top.push_back(v);
				s = icmExpectedInfluence(g, top, round) - currentInfluence;
				nodes[i].key = s;
				if (s > maxValue) {
					maxValue = s;
					bestNode = u;
				}
				top.pop_back();
			}
		}

		if (maxValue == 0) break; // There is no more useful node

		top.push_back(vertex(bestNode,g));
		selected[bestNode] = true;

		stop = clock();
		rt += (double) (stop-start)/CLOCKS_PER_SEC;
		currentInfluence = icmExpectedInfluence(g, top, round);
		vector<Vertex> input, output, candidate;
		FOR (i, top.size()) {
			input.push_back(top[top.size() - i -1]);
			candidate.push_back(orderList[i]);
		}
		output = localSwap(g, input, candidate, round);
		double swapValue = icmExpectedInfluence(g, output, round);

		fprintf(fo, "%d\t%f\t%f\t%f\n", t + 1, currentInfluence, swapValue, rt);

		fflush(fo);
	}

	fclose(fo);
	delete [] selected;
	return top;
} // End of icmTopLazyGreedySwap

vector<Vertex> icmUpperBound(Graph& g, int k, int round){
	FILE *fo;
	char outName[40];
	sprintf(outName, "TopkBound_%d_%d_%d.txt", num_vertices(g), num_edges(g), round);
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#UpperBound algorithm k = %d round = %d\n", k, round);
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;

	int n = num_vertices(g);
	double currentInfluence = 0;

	if (k > n) k = n;

	vector<Vertex> top;

	bool* selected = new bool[n];
	FOR (i, n) {
		selected[i] = false;
	}

	vector<Node> nodes;
	nodes.reserve(n);
	assert((start = clock())!=-1);
	FOR (u, n) {
		Vertex v = vertex(u, g);
		top.push_back(v);
		Node newNode = {u, icmExpectedInfluence(g, top, round) };
		nodes.push_back(newNode);
		top.pop_back();
	}
	sort(nodes.begin(), nodes.end(), comp);
	double sum = 0;
	for (int t=0; t < k; t++) {
		sum += nodes[t].key;

		fprintf(fo, "%d\t%f\n", t + 1, sum);
		fflush(fo);
	}


	fclose(fo);
	delete [] selected;
	return top;
} // End of icmUpperBound
//===========================================================================

vector<Vertex> optimalICM (Graph& g, double fraction, int round) {
	int n = num_vertices(g);

	vector<bool> selected(n);
	for (int i = 0; i<n; i++) selected[ i ] = false;

	bool found = false;
	int k = 1;
	while (!found) {
		vector<int> optimal(k);
		vector<int> comb(k);
		FOR (i, k) comb[i] = i;

		double max = 0;

		do {
			vector<Vertex> top;
			FOR (i, k) {
				top.push_back(vertex(comb[i], g));
			}

			double influence = icmExpectedInfluence(g, top, round);


			if (influence > max ) {
				FOR (i, k) {
					optimal[i] = comb[i];
				}
				max = influence;
			}

		} while (next_comb(comb, k, n));
		if (max > fraction*n) {
			for (int i=0; i<k; i++) {
				selected[ comb[i] ] = true;
				found = true;
			}
		} else k++;
	}

	vector<Vertex> top;
	FOR (i, n) {
		if (selected[i]) top.push_back(vertex(i, g));
	}

	return top;
}