/*
 * Effector.cpp
 *
 *  Created on: Jul 28, 2011
 *      Author: dung
 */

#include "Effector.h"
#include "IDS.h"
#include "Util.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <queue>
#include <ctime>

/*
 * Generate a set of adopters from the given seeds. This set is random.
 */
vector<Vertex> generateTarget(Graph& g, vector<Vertex> seeds) {
	int n = num_vertices(g);

	bool* active = new bool[n];

	srand(time(NULL));
	for (int i=0; i<n; ++i)
		active[i] = false;

	queue<Vertex> q;
	FOR (i, seeds.size()) {
		active[ g[ seeds[ i ] ].index ] = true;
		q.push( seeds[ i ] );
	}

	while (q.size() > 0){
		Vertex node = q.front();
		q.pop();

		pair<out_edge_iterator, out_edge_iterator> outEdges = out_edges(node, g);

		for(; outEdges.first != outEdges.second; ++outEdges.first) {
			Edge e = *outEdges.first;
			Vertex adjNode = target(e, g);
			if ( !active[ g [ adjNode ].index ] ) {
				int rn = rand();
				if (rn < g[ e ].weight * RAND_MAX){
					active[ g [ adjNode ].index ] = true;
					q.push(adjNode);
				}
			}
		}
	}

	vector<Vertex> adopters;
	FOR (i, n) {
		if (active[ i ]) {
			adopters.push_back(vertex(i, g));
		}
	}
	delete [] active;
	return adopters;
} // End of generateTarget
//=======================================================================================

vector<double> icmExpectedInfluenceVector(Graph& g, vector<Vertex> seeds, int round){
	int n = num_vertices(g);

	int *hit = new int[n];
	FOR (i, n)
	hit[ i ] = 0;

	bool* active = new bool[n];

	srand(time(NULL));
	FOR (r, round) {
		for (int i=0; i<n; ++i)
			active[i] = false;

		queue<Vertex> q;
		FOR (i, seeds.size()) {
			active[ g[ seeds[ i ] ].index ] = true;
			hit[  g[ seeds[ i ] ].index ]  = 1;
			q.push( seeds[ i ] );
		}

		while (q.size() > 0){
			Vertex node = q.front();
			q.pop();

			pair<out_edge_iterator, out_edge_iterator> outEdges = out_edges(node, g);

			for(; outEdges.first != outEdges.second; ++outEdges.first) {
				Edge e = *outEdges.first;
				Vertex adjNode = target(e, g);
				if ( !active[ g [ adjNode ].index ] ) {
					int rn = rand();
					if (rn < g[ e ].weight * RAND_MAX){
						active[ g [ adjNode ].index ] = true;
						hit[ g [ adjNode ].index ]++;
						q.push(adjNode);
					}
				}
			}
		}
	}

	vector<double> expectedInfluence;
	FOR (i, n) {
		expectedInfluence.push_back(hit[ i ]/(1.0 * round));
	}
	delete [] hit;
	delete [] active;
	return expectedInfluence;
} // End of expectedInfluence

vector<Vertex> effectorTopkImeter(Graph& g,  int k, vector<Vertex> target, int len, int round) {
	FILE *fo;
	char outName[40];
	sprintf(outName, "EffectorImeter_%d_%d_%d.txt", num_vertices(g), num_edges(g), round);
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#Imeter algorithm k = %d len = %d round = %d \n", k, len, round);
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;
	assert((start = clock())!=-1);
	vector<Vertex> orderList = iMeterOrder(g, len, round, target);
	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	vector<Vertex> top;
	int n = num_vertices( g );

	if (k > num_vertices( g ) ) k = n;

	vector<double> object;
	object.reserve(n);

	FOR (i, n) {
		object.push_back(0);
	}
	FOR (i, target.size()) {
		object[ g[ target[ i ] ].index ] = 1;
	}

	FOR (i, k) {
		top.push_back(orderList[i]);
		vector<double> expectedInfluence = icmExpectedInfluenceVector(g, top, round);
		double dis = 0;
		dist(object, expectedInfluence, 1, dis);
		fprintf(fo, "%d\t%f\t%f\n", i + 1, dis, rt);
		fflush(fo);
	}

	printf("\n Selected nodes: \n");
	FOR (i, k) {
		fprintf(fo, "%d ", top[ i ]);
		printf("%d ", top[ i ]);
	}
	fclose(fo);
	return top;
} // End of effectorTopkImeter
///////////////////////////////////////////////////////////////////////////////////////////////////////////
vector<Vertex> effectorTopkFusedWalk(Graph& g,  int k, vector<Vertex> target, int len, int round) {
	FILE *fo;
	char outName[40];
	sprintf(outName, "EffectorFusedWalk_%d_%d_%d.txt", num_vertices(g), num_edges(g), round);
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#FusedWalk algorithm k = %d len = %d round = %d \n", k, len, round);
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;
	assert((start = clock())!=-1);
	vector<Vertex> orderList = fusedWalkOrder(g, len, round, target);
	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	vector<Vertex> top;
	int n = num_vertices( g );

	if (k > num_vertices( g ) ) k = n;

	vector<double> object;
	object.reserve(n);

	FOR (i, n) {
		object.push_back(0);
	}
	FOR (i, target.size()) {
		object[ g[ target[ i ] ].index ] = 1;
	}

	FOR (i, k) {
		top.push_back(orderList[i]);
		vector<double> expectedInfluence = icmExpectedInfluenceVector(g, top, round);
		double dis = 0;
		dist(object, expectedInfluence, 1, dis);
		fprintf(fo, "%d\t%f\t%f\n", i + 1, dis, rt);
		fflush(fo);
	}

	printf("\n Selected nodes: \n");
	FOR (i, k) {
		fprintf(fo, "%d ", top[ i ]);
		printf("%d ", top[ i ]);
	}
	fclose(fo);
	return top;
}

/*
 * Evaluate the effect of one node, the distance between the activation vector and the expected influent vector caused
 * by the given node
 */
double evaluate(Graph& g, vector<Vertex> target, Vertex node, int round) {
	vector<Vertex> seeds;
	seeds.push_back(node);

	vector<double> expectedVector = icmExpectedInfluenceVector(g, seeds, round);

	int n = num_vertices(g);
	vector<double> object;
	object.reserve(n);

	FOR (i, n) {
		object.push_back(0);
	}
	FOR (i, target.size()) {
		object[ g[ target[ i ] ].index ] = 1;
	}

	double dis = 0;
	dist(object, expectedVector, 1, dis);

	return dis;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
 * Greedy order
 */
vector<Vertex> effectorTopkGreedyOrder(Graph& g, vector<Vertex> target, int round) {
	vector<Node> nodes;
	vector<Vertex> order;
	nodes.reserve(target.size());
	order.reserve(target.size());

	FOR(i, target.size()){
		Node newNode = {g[ target[ i ] ].index, evaluate(g, target, target[i], round)};
		nodes.push_back(newNode);
	}

	sort(nodes.begin(), nodes.end(), comp);

	int s = target.size();
	for (int u = s - 1; u>=0; u--)
	{
		order.push_back( vertex(nodes[u ].id, g) );
	}

	return order;
}
/*
 * Evaluate each node base on the effector paper's criterion
 */
vector<Vertex> effectorTopkGreedySort(Graph& g,  int k, vector<Vertex> target, int len, int round) {
	FILE *fo;
	char outName[200];
	sprintf(outName, "EffectorGreedySort_%d_%d_%d.txt", num_vertices(g), num_edges(g), round);
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#GreedySort algorithm k = %d len = %d round = %d \n", k, len, round);
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;
	assert((start = clock())!=-1);
	vector<Vertex> orderList = effectorTopkGreedyOrder(g, target, round);
	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	vector<Vertex> top;
	int n = num_vertices( g );

	if (k > target.size() ) k = target.size();

	vector<double> object;
	object.reserve(n);

	FOR (i, n) {
		object.push_back(0);
	}
	FOR (i, target.size()) {
		object[ g[ target[ i ] ].index ] = 1;
	}

	FOR (i, k) {
		top.push_back(orderList[i]);
		//cout << top.size() << ":" << orderList[i] << endl;
		//vector<double> expectedInfluence = icmExpectedInfluenceVector(g, top, round);
		double dis = 0;
		//dist(object, expectedInfluence, 1, dis);
		//fprintf(fo, "%d\t%f\t%f\n", i + 1, dis, rt);
		fflush(fo);
	}

	//printf("\n Selected nodes: \n");
	//FOR (i, k) {
	//	fprintf(fo, "%d ", top[ i ]);
	//	printf("%d ", top[ i ]);
	//}
	fclose(fo);
	return top;
}

/*
 * Compute the expected number of influenced targeted nodes given the seed set
 */
double effectorExpectedInfluence(Graph& g, vector<Vertex> seeds, vector<Vertex> target, int round){
	int n = num_vertices(g);
	int numInfluence = 0;

	bool* active = new bool[n];
	bool* inTarget = new bool[n];
	FOR (u, n) {
		inTarget[ u ] = false;
	}

	FOR (i, target.size()) {
		inTarget[ g[ target[ i ] ].index ] = true;
	}

	srand(time(NULL));
	FOR (r, round) {
		for (int i=0; i<n; ++i)
			active[i] = false;

		queue<Vertex> q;
		FOR (i, seeds.size()) {
			if (inTarget[ g[ seeds[i] ].index ]) numInfluence++;
			active[ g[ seeds[ i ] ].index ] = true;
			q.push( seeds[ i ] );
		}

		while (q.size() > 0){
			Vertex node = q.front();
			q.pop();

			pair<out_edge_iterator, out_edge_iterator> outEdges = out_edges(node, g);

			for(; outEdges.first != outEdges.second; ++outEdges.first) {
				Edge e = *outEdges.first;
				Vertex adjNode = boost::target(e, g);
				if ( !active[ g [ adjNode ].index ] ) {
					int rn = rand();
					if (rn < g[ e ].weight * RAND_MAX){
						active[ g [ adjNode ].index ] = true;
						q.push(adjNode);
						if (inTarget[ g [ adjNode ].index ]) numInfluence++;
					}
				}
			}
		}
	}
	delete [] active;
	delete [] inTarget;
	return numInfluence/(1.0 * round);
} // End of effectorExpectedInfluence
/*
 * Find the top k influent nodes from the source set that maximize the influence on the target set
 */
vector<Vertex> effectorTopLazyGreedy(Graph& g, int k, vector<Vertex> source, vector<Vertex> target, int round){
	FILE *fo;
	char outName[200];
	sprintf(outName, "TopkEffectorLazyGreedy_%d_%d.txt", num_vertices(g), num_edges(g));
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#LazyGreedy algorithm \n");
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;

	int n = num_vertices(g);
	double currentInfluence = 0;

	if (k > source.size()) k = source.size();

	vector<Vertex> top;

	bool* selected = new bool[n];
	bool* inSource = new bool[ n ];
	FOR (i, n) {
		selected[i] = false;
		inSource[i]  = false;
	}

	FOR (i, source.size()) {
		inSource[ g[ source[ i ] ].index ] = true;
	}


	vector<Node> nodes;
	nodes.reserve(n);
	assert((start = clock())!=-1);
	FOR (u, n) {
		Vertex v = vertex(u, g);
		top.push_back(v);
		double influence = 0;
		if (inSource[ g[v].index ]) influence = effectorExpectedInfluence(g, top, target, round);
		Node newNode = {u, influence};
		nodes.push_back(newNode);
		top.pop_back();
	}
	sort(nodes.begin(), nodes.end(), comp);
	top.push_back(vertex(nodes[0].id, g));
	selected[nodes[0].id] = true;

	stop = clock();
	rt += (double) (stop-start)/CLOCKS_PER_SEC;

	currentInfluence = effectorExpectedInfluence(g, top, target, round);
	fprintf(fo, "%d\t%f\t%f\n", 1, currentInfluence, rt);

	for (int t=1; t < k; t++) {
		assert((start = clock())!=-1);
		double maxValue = 0;
		double s;
		int bestNode;
		sort(nodes.begin(), nodes.end(), comp);
		FOR (i, source.size()) {
			int u = nodes[i].id;
			Vertex v = vertex(u, g);
			if ( ! selected[ g [ v ].index ] ) {
				if (nodes[g[v].index].key < maxValue)
					continue;
				top.push_back(v);
				s = effectorExpectedInfluence(g, top, target, round) - currentInfluence;
				nodes[i].key = s;
				if (s > maxValue) {
					maxValue = s;
					bestNode = u;
				}
				top.pop_back();
			}
		}

		if (maxValue == 0) break; // There is no more useful node

		top.push_back(vertex(bestNode,g));
		selected[bestNode] = true;

		stop = clock();
		rt += (double) (stop-start)/CLOCKS_PER_SEC;
		currentInfluence = effectorExpectedInfluence(g, top, target, round);

		fprintf(fo, "%d\t%f\t%f\n", t + 1, currentInfluence, rt);
		fflush(fo);
	}


	fclose(fo);
	delete [] selected;
	delete [] inSource;
	return top;
} // End of icmTopLazyGreedy
//===========================================================================
/*
 * Optimal greedy
 */
vector<Vertex> effectorOptimalGreedy(Graph& g, int k, vector<Vertex> source, vector<Vertex> target, int round){
	int n = num_vertices(g);
	double currentInfluence = 0;
	int ssize = source.size();

	vector<int> optimal(k);
	vector<int> comb(k);
	for (int i = 0; i<k; i++) comb[i] = i;

	double max = 0;

	do {
		vector<Vertex> top;
		FOR (i, k) {
			top.push_back(source[comb[i]]);
		}

		double influence = effectorExpectedInfluence(g, top, target, round);

		if (influence > max) {
			FOR (i, k) {
				optimal[i] = comb[i];
			}
			max  = influence;
		}
	} while (next_comb(comb, k, source.size()));
	vector<Vertex> top;
	FOR (i, k) {
		top.push_back(source[optimal[i]]);
	}
	return top;
} // End of icmTopLazyGreedy

/*
 * Optimal distance
 */
vector<Vertex> effectorOptimalDistance(Graph& g, int k, vector<Vertex> source, vector<Vertex> target, int round){
	int n = num_vertices(g);
	double currentInfluence = 0;
	int ssize = source.size();

	vector<int> optimal(k);
	vector<int> comb(k);
	FOR (i, k) comb[i] = i;

	double min = n;

	vector<double> object;
	object.reserve(n);

	FOR (i, n) {
		object.push_back(0);
	}
	FOR (i, target.size()) {
		object[ g[ target[ i ] ].index ] = 1;
	}

	do {
		vector<Vertex> top;
		FOR (i, k) {
			top.push_back(source[comb[i]]);
		}

		vector<double> expectedVector = icmExpectedInfluenceVector(g, top, round);

		double distance = 0;

		dist(object, expectedVector, 1, distance);

		if (distance < min) {
			FOR (i, k) {
				optimal[i] = comb[i];
			}
			min = distance;
		}

	} while (next_comb(comb, k, source.size()));
	vector<Vertex> top;
	FOR (i, k) {
		top.push_back(source[optimal[i]]);
	}

	return top;
} // End of icmTopLazyGreedy
vector<Vertex> effectorMaxExpectedSort(Graph& g, int k, vector<Vertex> source, vector<Vertex> target, int round){
	FILE *fo;
	char outName[200];
	sprintf(outName, "TopkEffectorLazyGreedy_%d_%d.txt", num_vertices(g), num_edges(g));
	fo = fopen(outName, "w");
	fprintf(fo, "#%d vertices %d edges \n", num_vertices(g), num_edges(g));
	fprintf(fo, "#LazyGreedy algorithm \n");
	fprintf(fo, "#k || Expected influenced nodes || Time \n");

	clock_t start, stop;
	double rt = 0;

	int n = num_vertices(g);
	double currentInfluence = 0;

	if (k > source.size()) k = source.size();

	vector<Vertex> top;

	bool* selected = new bool[n];
	bool* inSource = new bool[ n ];
	FOR (i, n) {
		selected[i] = false;
		inSource[i]  = false;
	}

	FOR (i, source.size()) {
		inSource[ g[ source[ i ] ].index ] = true;
	}


	vector<Node> nodes;
	nodes.reserve(n);
	assert((start = clock())!=-1);
	FOR (u, n) {
		Vertex v = vertex(u, g);
		top.push_back(v);
		double influence = 0;
		if (inSource[ g[v].index ]) influence = effectorExpectedInfluence(g, top, target, round);
		Node newNode = {u, influence};
		nodes.push_back(newNode);
		top.pop_back();
	}
	sort(nodes.begin(), nodes.end(), comp);

	FOR (i, k)
		top.push_back(vertex(nodes[i].id, g));


	fclose(fo);
	delete [] selected;
	delete [] inSource;
	return top;
} // End of icmTopLazyGreedy

double evaluateProduct(Graph& g, vector<Vertex> seeds, vector<Vertex> target, int round, double inside_threshold, double outside_threshold){
	int n = num_vertices(g);

	bool* infected = new bool[ n ];
	FOR (i, n) {
		infected[ i ] = false;
	}

	FOR (i, target.size()) {
		infected[ g[ target[ i ] ].index ] = true;
	}

	vector<double> expectedInfluence = icmExpectedInfluenceVector(g, seeds, round);

	double product = 1;

	FOR (i, n) {
		if (infected[ i ]) {
			if (expectedInfluence[ i ] >= inside_threshold)
				product *= expectedInfluence[ i ];
		} else {
			if (1 - expectedInfluence[ i ] >= outside_threshold) {
				product *= ( 1 - expectedInfluence[ i ]);
			}
		}
	}

	delete [] infected;
	return product;
}

vector<Vertex> effectorMaxProductSort(Graph& g, int k, vector<Vertex> source, vector<Vertex> target, int round, double inside_threshold, double outside_threshold){
	
	int n = num_vertices(g);

	if (k > source.size()) k = source.size();

	vector<Vertex> top;

	bool* inSource = new bool[ n ];
	FOR (i, n) {
		inSource[i]  = false;
	}

	FOR (i, source.size()) {
		inSource[ g[ source[ i ] ].index ] = true;
	}
	
	vector<Node> nodes;
	nodes.reserve(n);
	FOR (u, n) {
		Vertex v = vertex(u, g);
		top.push_back(v);
		double product = 0;
		if (inSource[ g[v].index ]) product = evaluateProduct(g, top, target, round, inside_threshold, outside_threshold);
		Node newNode = {u, product};
		nodes.push_back(newNode);
		top.pop_back();
	}
	sort(nodes.begin(), nodes.end(), comp);

	FOR (i, k)
		top.push_back(vertex(nodes[i].id, g));

	delete [] inSource;
	return top;
} // End of icmTopLazyGreedy